/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.downloadbypass.util;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.texboobcat.downloadbypass.Downloadbypass;
import org.texboobcat.downloadbypass.config.ModDependency;

public class VersionConflictResolver {
    private static final Pattern VERSION_PATTERN = Pattern.compile("([0-9]+)\\.([0-9]+)(?:\\.([0-9]+))?(?:[\\-\\+].*)?");

    public static List<Conflict> detectConflicts(Path modsDir, List<ModDependency> dependencies) {
        ArrayList<Conflict> conflicts = new ArrayList<Conflict>();
        if (!Files.exists(modsDir, new LinkOption[0])) {
            return conflicts;
        }
        try {
            List<InstalledMod> installed = VersionConflictResolver.scanInstalledMods(modsDir);
            conflicts.addAll(VersionConflictResolver.detectDuplicates(installed));
            conflicts.addAll(VersionConflictResolver.detectVersionMismatches(installed, dependencies));
            Downloadbypass.LOGGER.info("[EXTERNAL-MODS] Conflict detection complete: {} conflicts found", (Object)conflicts.size());
        }
        catch (IOException e) {
            Downloadbypass.LOGGER.error("[EXTERNAL-MODS] Error during conflict detection: {}", (Object)e.getMessage());
        }
        return conflicts;
    }

    private static List<InstalledMod> scanInstalledMods(Path modsDir) throws IOException {
        ArrayList<InstalledMod> mods = new ArrayList<InstalledMod>();
        try (Stream<Path> stream = Files.list(modsDir);){
            stream.filter(p -> p.getFileName().toString().endsWith(".jar")).forEach(p -> {
                try {
                    InstalledMod mod = VersionConflictResolver.extractModInfo(p);
                    if (mod != null) {
                        mods.add(mod);
                    }
                }
                catch (Exception e) {
                    Downloadbypass.LOGGER.debug("[EXTERNAL-MODS] Could not read mod {}: {}", (Object)p.getFileName(), (Object)e.getMessage());
                }
            });
        }
        return mods;
    }

    private static InstalledMod extractModInfo(Path jarPath) {
        String fileName = jarPath.getFileName().toString();
        String nameWithoutExt = fileName.substring(0, fileName.length() - 4);
        Matcher matcher = VERSION_PATTERN.matcher(nameWithoutExt);
        String modId = nameWithoutExt;
        String version = null;
        if (matcher.find()) {
            version = matcher.group(0);
            modId = nameWithoutExt.substring(0, matcher.start()).replaceAll("[\\-_]+$", "");
        }
        try (JarFile jar = new JarFile(jarPath.toFile());){
            Manifest manifest = jar.getManifest();
            if (manifest != null) {
                Attributes attrs = manifest.getMainAttributes();
                String forgeModId = attrs.getValue("Implementation-Title");
                String forgeVersion = attrs.getValue("Implementation-Version");
                if (forgeModId != null && !forgeModId.isBlank()) {
                    modId = forgeModId;
                }
                if (forgeVersion != null && !forgeVersion.isBlank()) {
                    version = forgeVersion;
                }
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return new InstalledMod(jarPath, modId, version);
    }

    private static List<Conflict> detectDuplicates(List<InstalledMod> mods) {
        ArrayList<Conflict> conflicts = new ArrayList<Conflict>();
        Map<String, List<InstalledMod>> byModId = mods.stream().collect(Collectors.groupingBy(m -> m.modId.toLowerCase()));
        for (Map.Entry<String, List<InstalledMod>> entry : byModId.entrySet()) {
            List<InstalledMod> versions = entry.getValue();
            if (versions.size() <= 1) continue;
            conflicts.add(new Conflict(ConflictType.DUPLICATE, entry.getKey(), versions.stream().map(m -> m.version).collect(Collectors.toList()), "Multiple versions of the same mod detected", versions.stream().map(m -> m.path).collect(Collectors.toList())));
        }
        return conflicts;
    }

    private static List<Conflict> detectVersionMismatches(List<InstalledMod> installed, List<ModDependency> dependencies) {
        ArrayList<Conflict> conflicts = new ArrayList<Conflict>();
        for (ModDependency dep : dependencies) {
            if (dep == null || dep.getName() == null) continue;
            String depName = dep.getName().toLowerCase();
            Optional<InstalledMod> match = installed.stream().filter(m -> m.modId.toLowerCase().contains(depName) || depName.contains(m.modId.toLowerCase())).findFirst();
            if (!match.isPresent() || dep.getVersion() == null || dep.getVersion().isBlank()) continue;
            InstalledMod mod = match.get();
            String requiredVersion = dep.getVersion();
            if (mod.version == null || VersionConflictResolver.isVersionCompatible(mod.version, requiredVersion)) continue;
            conflicts.add(new Conflict(ConflictType.VERSION_MISMATCH, dep.getName(), Arrays.asList(mod.version, requiredVersion), "Installed version does not match required version", Collections.singletonList(mod.path)));
        }
        return conflicts;
    }

    private static boolean isVersionCompatible(String installed, String required) {
        if (installed.equals(required)) {
            return true;
        }
        String[] installedParts = installed.split("\\.");
        String[] requiredParts = required.split("\\.");
        if (installedParts.length >= 2 && requiredParts.length >= 2) {
            return installedParts[0].equals(requiredParts[0]) && installedParts[1].equals(requiredParts[1]);
        }
        return false;
    }

    public static List<Conflict> resolveConflicts(List<Conflict> conflicts, boolean autoResolve) {
        ArrayList<Conflict> resolved = new ArrayList<Conflict>();
        for (Conflict conflict : conflicts) {
            if (conflict.type != ConflictType.DUPLICATE || !autoResolve) continue;
            resolved.add(VersionConflictResolver.resolveDuplicate(conflict));
        }
        return resolved;
    }

    private static Conflict resolveDuplicate(Conflict conflict) {
        List sorted2 = conflict.affectedFiles.stream().sorted((p1, p2) -> {
            try {
                return Files.getLastModifiedTime(p2, new LinkOption[0]).compareTo(Files.getLastModifiedTime(p1, new LinkOption[0]));
            }
            catch (IOException e) {
                return 0;
            }
        }).collect(Collectors.toList());
        ArrayList<Path> removed = new ArrayList<Path>();
        for (int i = 1; i < sorted2.size(); ++i) {
            try {
                Files.delete((Path)sorted2.get(i));
                removed.add((Path)sorted2.get(i));
                Downloadbypass.LOGGER.info("[EXTERNAL-MODS] Removed duplicate: {}", (Object)((Path)sorted2.get(i)).getFileName());
                continue;
            }
            catch (IOException e) {
                Downloadbypass.LOGGER.error("[EXTERNAL-MODS] Failed to remove {}: {}", (Object)((Path)sorted2.get(i)).getFileName(), (Object)e.getMessage());
            }
        }
        return new Conflict(ConflictType.DUPLICATE, conflict.modName, conflict.versions, "Resolved: kept newest version", removed);
    }

    private static class InstalledMod {
        final Path path;
        final String modId;
        final String version;

        InstalledMod(Path path, String modId, String version) {
            this.path = path;
            this.modId = modId;
            this.version = version;
        }
    }

    public static class Conflict {
        public final ConflictType type;
        public final String modName;
        public final List<String> versions;
        public final String description;
        public final List<Path> affectedFiles;

        public Conflict(ConflictType type, String modName, List<String> versions, String description, List<Path> affectedFiles) {
            this.type = type;
            this.modName = modName;
            this.versions = versions;
            this.description = description;
            this.affectedFiles = affectedFiles;
        }

        public String toString() {
            return String.format("[%s] %s: %s (versions: %s, files: %d)", new Object[]{this.type, this.modName, this.description, String.join((CharSequence)", ", this.versions), this.affectedFiles.size()});
        }
    }

    public static enum ConflictType {
        DUPLICATE,
        VERSION_MISMATCH,
        INCOMPATIBLE;

    }
}

