/*
 * Decompiled with CFR 0.152.
 */
package com.dooji.electricity.main.power;

import com.dooji.electricity.api.power.PowerDeliveryEvent;
import com.dooji.electricity.block.ElectricCabinBlockEntity;
import com.dooji.electricity.block.PowerBoxBlockEntity;
import com.dooji.electricity.block.UtilityPoleBlockEntity;
import com.dooji.electricity.block.WindTurbineBlockEntity;
import com.dooji.electricity.main.network.ElectricityNetworking;
import com.dooji.electricity.main.weather.GlobalWeatherManager;
import com.dooji.electricity.main.weather.WeatherSnapshot;
import com.dooji.electricity.main.wire.WireConnection;
import com.dooji.electricity.main.wire.WireManager;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Vec3i;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.util.Mth;
import net.minecraft.util.RandomSource;
import net.minecraft.world.level.block.entity.BlockEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PowerNetwork {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"electricity");
    private final ServerLevel level;
    private final Map<Integer, PowerNode> powerNodes = new HashMap<Integer, PowerNode>();
    private final Map<String, PowerConnection> powerConnections = new HashMap<String, PowerConnection>();
    private final Map<BlockPos, List<PowerNode>> nodesByPosition = new HashMap<BlockPos, List<PowerNode>>();
    private final WireManager wireManager;
    private final Map<BlockPos, Double> lastSyncedPower = new HashMap<BlockPos, Double>();
    private final Set<Integer> surgeImpactedNodes = new HashSet<Integer>();
    private final Map<Integer, PowerDeliveryEvent> nodeEvents = new HashMap<Integer, PowerDeliveryEvent>();
    private final Map<Integer, GeneratorEvent> generatorEvents = new HashMap<Integer, GeneratorEvent>();

    public PowerNetwork(ServerLevel level, WireManager wireManager) {
        this.level = level;
        this.wireManager = wireManager;
    }

    public void updatePowerNetwork() {
        this.clearNetwork();
        this.buildNetworkFromWires();
        this.calculatePowerFlow();
        this.syncToClients();
    }

    private void clearNetwork() {
        this.powerNodes.clear();
        this.powerConnections.clear();
        this.nodesByPosition.clear();
        this.surgeImpactedNodes.clear();
        this.generatorEvents.clear();
    }

    private void buildNetworkFromWires() {
        WireManager.WireSavedData savedData = this.wireManager.getSavedData(this.level);
        if (savedData == null) {
            return;
        }
        for (WireConnection wireConnection : savedData.getAllWireConnections()) {
            this.addWireConnection(wireConnection);
        }
    }

    private void addWireConnection(WireConnection wireConnection) {
        int startId = wireConnection.getStartInsulatorId();
        int endId = wireConnection.getEndInsulatorId();
        PowerNode startNode = this.getOrCreateNode(startId, wireConnection.getStartBlockPos(), wireConnection.getStartBlockType());
        PowerNode endNode = this.getOrCreateNode(endId, wireConnection.getEndBlockPos(), wireConnection.getEndBlockType());
        if (startNode != null && endNode != null) {
            double distance = this.calculateDistance(startNode.position, endNode.position);
            String connectionKey = Math.min(startId, endId) + "_" + Math.max(startId, endId);
            PowerConnection connection = new PowerConnection(startNode, endNode, distance, wireConnection.getStartPowerType(), wireConnection.getEndPowerType());
            this.powerConnections.put(connectionKey, connection);
        }
    }

    private PowerNode getOrCreateNode(int insulatorId, BlockPos blockPos, String blockType) {
        if (this.powerNodes.containsKey(insulatorId)) {
            return this.powerNodes.get(insulatorId);
        }
        BlockPos immutablePos = blockPos.m_7949_();
        BlockEntity blockEntity = this.level.m_7702_(immutablePos);
        if (blockEntity == null) {
            LOGGER.warn("Block entity not found at {} for insulator {}", (Object)blockPos, (Object)insulatorId);
            this.removeStaleConnections(insulatorId);
            return null;
        }
        boolean typeMatches = false;
        if ("wind_turbine".equals(blockType) && blockEntity instanceof WindTurbineBlockEntity) {
            typeMatches = true;
        } else if ("electric_cabin".equals(blockType) && blockEntity instanceof ElectricCabinBlockEntity) {
            typeMatches = true;
        } else if ("utility_pole".equals(blockType) && blockEntity instanceof UtilityPoleBlockEntity) {
            typeMatches = true;
        } else if ("power_box".equals(blockType) && blockEntity instanceof PowerBoxBlockEntity) {
            typeMatches = true;
        }
        if (!typeMatches) {
            LOGGER.warn("Block type mismatch at {} - client reported {} but server found {}", new Object[]{blockPos, blockType, blockEntity.getClass().getSimpleName()});
            this.removeStaleConnections(insulatorId);
            return null;
        }
        PowerNode node = new PowerNode(insulatorId, immutablePos, blockEntity);
        this.powerNodes.put(insulatorId, node);
        this.nodesByPosition.computeIfAbsent(immutablePos, pos -> new ArrayList()).add(node);
        return node;
    }

    private void removeStaleConnections(int insulatorId) {
        if (insulatorId <= 0) {
            return;
        }
        this.wireManager.removeConnectionsForInsulators(this.level, new int[]{insulatorId});
    }

    private double calculateDistance(BlockPos pos1, BlockPos pos2) {
        return Math.sqrt(pos1.m_123331_((Vec3i)pos2));
    }

    private void calculatePowerFlow() {
        HashMap<Integer, Double> nodePower = new HashMap<Integer, Double>();
        this.nodeEvents.clear();
        for (PowerNode node : this.powerNodes.values()) {
            double generatedPower;
            if (!(node.blockEntity instanceof WindTurbineBlockEntity) || (generatedPower = node.getOutputPower()) <= 0.0) continue;
            PowerDeliveryEvent generatorEvent = this.createGeneratorEvent((WindTurbineBlockEntity)node.blockEntity);
            Map<Integer, Double> powerDistribution = this.distributePower(node.insulatorId, generatedPower, new HashSet<Integer>(), true, node.hasLocalSurge(), generatorEvent, this.nodeEvents);
            for (Map.Entry<Integer, Double> entry : powerDistribution.entrySet()) {
                nodePower.merge(entry.getKey(), entry.getValue(), Double::sum);
            }
        }
        for (PowerNode node : this.powerNodes.values()) {
            double power = nodePower.getOrDefault(node.insulatorId, 0.0);
            node.setPower(power);
            node.setEvent(this.nodeEvents.getOrDefault(node.insulatorId, PowerDeliveryEvent.none()));
        }
    }

    private Map<Integer, Double> distributePower(int fromNodeId, double availablePower, Set<Integer> visited, boolean generationAlreadyIncluded, boolean surgeActive, PowerDeliveryEvent incomingEvent, Map<Integer, PowerDeliveryEvent> eventOut) {
        BlockEntity blockEntity;
        HashMap<Integer, Double> distribution = new HashMap<Integer, Double>();
        if (availablePower <= 0.0) {
            return distribution;
        }
        PowerNode startNode = this.powerNodes.get(fromNodeId);
        if (startNode == null) {
            return distribution;
        }
        double totalPower = availablePower;
        if (!generationAlreadyIncluded && (blockEntity = startNode.blockEntity) instanceof WindTurbineBlockEntity) {
            WindTurbineBlockEntity turbine = (WindTurbineBlockEntity)blockEntity;
            totalPower += Math.max(0.0, turbine.getGeneratedPower());
            surgeActive = surgeActive || turbine.isSurging();
        }
        boolean localSurge = surgeActive || startNode.hasLocalSurge();
        PowerDeliveryEvent currentEvent = incomingEvent;
        List<PowerNode> clusterNodes = this.nodesByPosition.getOrDefault(startNode.position, Collections.singletonList(startNode));
        if (this.isClusterVisited(clusterNodes, visited)) {
            return distribution;
        }
        HashSet<Integer> clusterIds = new HashSet<Integer>();
        for (PowerNode node : clusterNodes) {
            visited.add(node.insulatorId);
            clusterIds.add(node.insulatorId);
            distribution.put(node.insulatorId, totalPower);
            if (localSurge) {
                this.surgeImpactedNodes.add(node.insulatorId);
            }
            eventOut.merge(node.insulatorId, currentEvent, this::mergeEvents);
        }
        List<ClusterConnection> externalConnections = this.collectExternalConnections(clusterNodes, clusterIds);
        if (externalConnections.isEmpty()) {
            return distribution;
        }
        Map<BlockPos, TargetGroup> targetGroups = this.groupConnectionsByTarget(externalConnections);
        if (targetGroups.isEmpty()) {
            return distribution;
        }
        ArrayList<TargetGroup> viableGroups = new ArrayList<TargetGroup>();
        for (TargetGroup group : targetGroups.values()) {
            if (this.isClusterVisited(group.targetNode.position, visited)) continue;
            viableGroups.add(group);
        }
        if (viableGroups.isEmpty()) {
            return distribution;
        }
        double powerPerGroup = totalPower / (double)viableGroups.size();
        for (TargetGroup group : viableGroups) {
            double deliveredPower = powerPerGroup * group.computeEfficiency();
            if (deliveredPower <= 0.0) continue;
            PowerNode target = group.targetNode;
            distribution.merge(target.insulatorId, deliveredPower, Double::max);
            PowerDeliveryEvent propagatedEvent = this.attenuateEvent(currentEvent, group.computeEfficiency());
            eventOut.merge(target.insulatorId, propagatedEvent, this::mergeEvents);
            Map<Integer, Double> subDistribution = this.distributePower(target.insulatorId, deliveredPower, new HashSet<Integer>(visited), false, localSurge || target.hasLocalSurge(), propagatedEvent, eventOut);
            for (Map.Entry<Integer, Double> entry : subDistribution.entrySet()) {
                if (clusterIds.contains(entry.getKey())) continue;
                double finalPower = Math.min(entry.getValue(), deliveredPower);
                distribution.merge(entry.getKey(), finalPower, Double::max);
            }
        }
        return distribution;
    }

    private PowerDeliveryEvent createGeneratorEvent(WindTurbineBlockEntity turbine) {
        int id = turbine.getInsulatorId(0);
        GeneratorEvent state = this.generatorEvents.get(id);
        if (state != null && state.remaining > 0) {
            int nextRemaining = state.remaining - 1;
            this.generatorEvents.put(id, new GeneratorEvent(state.severity, nextRemaining, state.disconnect, state.brownout));
            return new PowerDeliveryEvent(state.severity, nextRemaining, state.disconnect, state.brownout);
        }
        WeatherSnapshot weather = GlobalWeatherManager.get(this.level).sample(turbine.m_58899_());
        double turbulence = weather.turbulence();
        double windSpeed = weather.windSpeed();
        RandomSource random = this.level.m_213780_();
        double gustFactor = Mth.m_14008_((double)((windSpeed - 8.0) / 12.0), (double)0.0, (double)1.0);
        double baseSeverity = Math.max(0.0, (turbulence - 0.25) * 0.6 + gustFactor * 0.4);
        double severity = Math.max(0.0, baseSeverity + random.m_188500_() * 0.05);
        int duration = severity > 0.25 ? 4 + random.m_188503_(5) : 0;
        boolean disconnect = false;
        double brownout = 1.0;
        if (windSpeed > 18.0 || turbulence > 0.75) {
            double faultChance = Mth.m_14008_((double)((windSpeed - 18.0) * 0.04 + (turbulence - 0.75) * 0.25), (double)0.0, (double)0.5);
            if (random.m_188500_() < faultChance) {
                disconnect = true;
                duration = Math.max(duration, 6 + random.m_188503_(5));
                brownout = 0.4 + random.m_188500_() * 0.2;
            }
        }
        if (duration > 0 || disconnect || brownout < 1.0) {
            this.generatorEvents.put(id, new GeneratorEvent(severity, duration, disconnect, brownout));
        } else {
            this.generatorEvents.remove(id);
        }
        return new PowerDeliveryEvent(severity, duration, disconnect, brownout);
    }

    private PowerDeliveryEvent attenuateEvent(PowerDeliveryEvent event, double efficiency) {
        if (event == null) {
            return PowerDeliveryEvent.none();
        }
        double severity = event.surgeSeverity() * Mth.m_14008_((double)efficiency, (double)0.0, (double)1.0) * 0.95;
        int duration = event.surgeDuration() > 0 ? Math.max(0, event.surgeDuration() - 1) : 0;
        boolean ifDisconnect = event.disconnectActive() && duration > 0;
        double brownout = duration > 0 ? 1.0 - (1.0 - event.brownoutFactor()) * Mth.m_14008_((double)efficiency, (double)0.0, (double)1.0) : 1.0;
        return new PowerDeliveryEvent(severity, duration, ifDisconnect, brownout);
    }

    private PowerDeliveryEvent mergeEvents(PowerDeliveryEvent a, PowerDeliveryEvent b) {
        if (a == null) {
            return b == null ? PowerDeliveryEvent.none() : b;
        }
        if (b == null) {
            return a;
        }
        double severity = Math.max(a.surgeSeverity(), b.surgeSeverity());
        int duration = Math.max(a.surgeDuration(), b.surgeDuration());
        boolean ifDisconnect = a.disconnectActive() || b.disconnectActive();
        double brownout = Math.min(a.brownoutFactor(), b.brownoutFactor());
        return new PowerDeliveryEvent(severity, duration, ifDisconnect, brownout);
    }

    private List<ClusterConnection> collectExternalConnections(List<PowerNode> clusterNodes, Set<Integer> clusterIds) {
        ArrayList<ClusterConnection> connections = new ArrayList<ClusterConnection>();
        Set seenConnections = Collections.newSetFromMap(new IdentityHashMap());
        for (PowerNode node : clusterNodes) {
            List<PowerConnection> outgoing = this.getOutgoingConnections(node.insulatorId);
            for (PowerConnection connection : outgoing) {
                if (!seenConnections.add(connection)) continue;
                PowerNode other = connection.getOtherNode(node);
                if (clusterIds.contains(other.insulatorId)) continue;
                connections.add(new ClusterConnection(connection, node));
            }
        }
        return connections;
    }

    private Map<BlockPos, TargetGroup> groupConnectionsByTarget(List<ClusterConnection> connections) {
        HashMap<BlockPos, TargetGroup> groups = new HashMap<BlockPos, TargetGroup>();
        for (ClusterConnection clusterConnection : connections) {
            PowerNode targetNode = clusterConnection.connection.getOtherNode(clusterConnection.sourceNode);
            BlockPos targetPos = targetNode.position;
            TargetGroup group = groups.computeIfAbsent(targetPos, pos -> new TargetGroup(targetNode));
            group.addConnection(clusterConnection.connection);
        }
        return groups;
    }

    private List<PowerConnection> getOutgoingConnections(int fromNodeId) {
        ArrayList<PowerConnection> connections = new ArrayList<PowerConnection>();
        for (PowerConnection connection : this.powerConnections.values()) {
            boolean sameBlockPos;
            if (connection.startNode.insulatorId == fromNodeId) {
                sameBlockPos = connection.startNode.position.equals((Object)connection.endNode.position);
                if ("output".equals(connection.startPowerType) && ("input".equals(connection.endPowerType) || "bidirectional".equals(connection.endPowerType))) {
                    connections.add(connection);
                    continue;
                }
                if ("bidirectional".equals(connection.startPowerType) && ("input".equals(connection.endPowerType) || "bidirectional".equals(connection.endPowerType))) {
                    connections.add(connection);
                    continue;
                }
                if (!sameBlockPos || !"input".equals(connection.startPowerType) || !"output".equals(connection.endPowerType)) continue;
                connections.add(connection);
                continue;
            }
            if (connection.endNode.insulatorId != fromNodeId) continue;
            sameBlockPos = connection.startNode.position.equals((Object)connection.endNode.position);
            if ("output".equals(connection.endPowerType) && ("input".equals(connection.startPowerType) || "bidirectional".equals(connection.startPowerType))) {
                connections.add(connection);
                continue;
            }
            if ("bidirectional".equals(connection.endPowerType) && ("input".equals(connection.startPowerType) || "bidirectional".equals(connection.startPowerType))) {
                connections.add(connection);
                continue;
            }
            if (!sameBlockPos || !"output".equals(connection.endPowerType) || !"input".equals(connection.startPowerType)) continue;
            connections.add(connection);
        }
        return connections;
    }

    public double getPowerForInsulator(int insulatorId) {
        PowerNode node = this.powerNodes.get(insulatorId);
        return node != null ? node.getPower() : 0.0;
    }

    public void syncToClients() {
        HashMap<BlockPos, Double> blockPower = new HashMap<BlockPos, Double>();
        HashMap<BlockPos, PowerNode> representatives = new HashMap<BlockPos, PowerNode>();
        HashMap<BlockPos, PowerDeliveryEvent> blockEvents = new HashMap<BlockPos, PowerDeliveryEvent>();
        HashSet<BlockPos> updatedPositions = new HashSet<BlockPos>();
        for (PowerNode powerNode : this.powerNodes.values()) {
            blockPower.merge(powerNode.position, powerNode.power, Double::max);
            representatives.putIfAbsent(powerNode.position, powerNode);
            blockEvents.merge(powerNode.position, powerNode.event, this::mergeEvents);
        }
        for (Map.Entry entry : blockPower.entrySet()) {
            double power = (Double)entry.getValue();
            BlockPos position = (BlockPos)entry.getKey();
            PowerNode representative = (PowerNode)representatives.get(position);
            PowerDeliveryEvent event = blockEvents.getOrDefault(position, PowerDeliveryEvent.none());
            if (representative != null) {
                representative.syncToClient(power, event);
            } else {
                PowerNetwork.applyPower(this.level.m_7702_(position), power, event);
            }
            ElectricityNetworking.sendPowerUpdate(this.level, position, power);
            updatedPositions.add(position);
        }
        HashSet<BlockPos> stalePositions = new HashSet<BlockPos>(this.lastSyncedPower.keySet());
        stalePositions.removeAll(updatedPositions);
        for (BlockPos stalePos : stalePositions) {
            PowerNetwork.applyPower(this.level.m_7702_(stalePos), 0.0, PowerDeliveryEvent.none());
            ElectricityNetworking.sendPowerUpdate(this.level, stalePos, 0.0);
        }
        this.lastSyncedPower.clear();
        this.lastSyncedPower.putAll(blockPower);
    }

    private boolean isClusterVisited(BlockPos position, Set<Integer> visitedIds) {
        return this.isClusterVisited(this.nodesByPosition.get(position), visitedIds);
    }

    private boolean isClusterVisited(List<PowerNode> clusterNodes, Set<Integer> visitedIds) {
        if (clusterNodes == null || clusterNodes.isEmpty()) {
            return false;
        }
        for (PowerNode node : clusterNodes) {
            if (visitedIds.contains(node.insulatorId)) continue;
            return false;
        }
        return true;
    }

    private static void applyPower(BlockEntity blockEntity, double power, PowerDeliveryEvent event) {
        if (blockEntity == null) {
            return;
        }
        if (blockEntity instanceof WindTurbineBlockEntity) {
            WindTurbineBlockEntity turbine = (WindTurbineBlockEntity)blockEntity;
            turbine.setCurrentPower(power);
        } else if (blockEntity instanceof ElectricCabinBlockEntity) {
            ElectricCabinBlockEntity cabin = (ElectricCabinBlockEntity)blockEntity;
            cabin.setCurrentPower(power);
        } else if (blockEntity instanceof UtilityPoleBlockEntity) {
            UtilityPoleBlockEntity pole = (UtilityPoleBlockEntity)blockEntity;
            pole.setCurrentPower(power);
        } else if (blockEntity instanceof PowerBoxBlockEntity) {
            PowerBoxBlockEntity powerBox = (PowerBoxBlockEntity)blockEntity;
            powerBox.setCurrentPower(power);
            powerBox.setIncomingEvent(event);
        }
    }

    private static class PowerNode {
        final int insulatorId;
        final BlockPos position;
        final BlockEntity blockEntity;
        double power = 0.0;
        private PowerDeliveryEvent event = PowerDeliveryEvent.none();

        PowerNode(int insulatorId, BlockPos position, BlockEntity blockEntity) {
            this.insulatorId = insulatorId;
            this.position = position;
            this.blockEntity = blockEntity;
        }

        double getOutputPower() {
            BlockEntity blockEntity = this.blockEntity;
            if (blockEntity instanceof WindTurbineBlockEntity) {
                WindTurbineBlockEntity turbine = (WindTurbineBlockEntity)blockEntity;
                return turbine.getGeneratedPower();
            }
            return this.power;
        }

        void setPower(double power) {
            this.power = power;
        }

        double getPower() {
            return this.power;
        }

        boolean hasLocalSurge() {
            WindTurbineBlockEntity turbine;
            BlockEntity blockEntity = this.blockEntity;
            return blockEntity instanceof WindTurbineBlockEntity && (turbine = (WindTurbineBlockEntity)blockEntity).isSurging();
        }

        void setEvent(PowerDeliveryEvent event) {
            this.event = event != null ? event : PowerDeliveryEvent.none();
        }

        void syncToClient(double syncedPower, PowerDeliveryEvent event) {
            PowerNetwork.applyPower(this.blockEntity, syncedPower, event);
        }
    }

    private static class PowerConnection {
        final PowerNode startNode;
        final PowerNode endNode;
        final double distance;
        final String startPowerType;
        final String endPowerType;

        PowerConnection(PowerNode startNode, PowerNode endNode, double distance, String startPowerType, String endPowerType) {
            this.startNode = startNode;
            this.endNode = endNode;
            this.distance = distance;
            this.startPowerType = startPowerType;
            this.endPowerType = endPowerType;
        }

        PowerNode getOtherNode(PowerNode node) {
            return node == this.startNode ? this.endNode : this.startNode;
        }
    }

    private static class TargetGroup {
        final PowerNode targetNode;
        private double totalDistance = 0.0;
        private int wireCount = 0;

        TargetGroup(PowerNode targetNode) {
            this.targetNode = targetNode;
        }

        void addConnection(PowerConnection connection) {
            ++this.wireCount;
            this.totalDistance += connection.distance;
        }

        double computeEfficiency() {
            if (this.wireCount == 0) {
                return 0.0;
            }
            double averageDistance = this.totalDistance / (double)this.wireCount;
            double normalizedDistance = averageDistance / 100.0;
            double parallelBoost = 1.0 + 0.35 * (double)(this.wireCount - 1);
            double distancePenalty = normalizedDistance / parallelBoost;
            return Math.max(0.1, 1.0 - distancePenalty);
        }
    }

    private record GeneratorEvent(double severity, int remaining, boolean disconnect, double brownout) {
    }

    private static class ClusterConnection {
        final PowerConnection connection;
        final PowerNode sourceNode;

        ClusterConnection(PowerConnection connection, PowerNode sourceNode) {
            this.connection = connection;
            this.sourceNode = sourceNode;
        }
    }
}

