/*
 * Decompiled with CFR 0.152.
 */
package net.machiavelli.minecolonytax;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import com.minecolonies.api.IMinecoloniesAPI;
import com.minecolonies.api.colony.ICitizenData;
import com.minecolonies.api.colony.IColony;
import com.minecolonies.api.colony.IColonyManager;
import com.minecolonies.api.colony.buildings.IBuilding;
import com.minecolonies.api.colony.permissions.ColonyPlayer;
import com.minecolonies.api.colony.permissions.IPermissions;
import com.minecolonies.core.colony.CitizenData;
import com.minecolonies.core.entity.citizen.citizenhandlers.CitizenHappinessHandler;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.lang.reflect.Type;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import net.machiavelli.minecolonytax.TaxConfig;
import net.machiavelli.minecolonytax.WarSystem;
import net.machiavelli.minecolonytax.abandon.ColonyAbandonmentManager;
import net.machiavelli.minecolonytax.abandon.ColonyClaimingRaidManager;
import net.machiavelli.minecolonytax.data.WarData;
import net.machiavelli.minecolonytax.raid.RaidManager;
import net.machiavelli.minecolonytax.vassalization.VassalManager;
import net.minecraft.ChatFormatting;
import net.minecraft.network.chat.Component;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.level.Level;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.event.TickEvent;
import net.minecraftforge.event.server.ServerStoppingEvent;
import net.minecraftforge.eventbus.api.SubscribeEvent;
import net.minecraftforge.fml.common.Mod;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

@Mod.EventBusSubscriber
public class TaxManager {
    private static final Map<Integer, Integer> colonyTaxMap = new HashMap<Integer, Integer>();
    private static final Logger LOGGER = LogManager.getLogger(TaxManager.class);
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final String TAX_DATA_FILE = "config/warntax/colonyTaxData.json";
    private static final String TAX_TIMESTAMP_FILE = "config/warntax/lastTaxGeneration.json";
    private static MinecraftServer serverInstance;
    private static final Set<Integer> FROZEN_COLONIES;
    private static final Set<Integer> DISABLED_GENERATION;
    private static TickEventHandler tickEventHandler;
    private static long lastTaxGenerationTime;

    public static void initialize(MinecraftServer server) {
        if (TaxConfig.showTaxGenerationLogs()) {
            LOGGER.info("Initializing Tax Manager...");
        }
        serverInstance = server;
        TaxManager.loadTaxData(server);
        TaxManager.loadLastTaxGenerationTime();
        if (tickEventHandler != null) {
            MinecraftForge.EVENT_BUS.unregister((Object)tickEventHandler);
        }
        tickEventHandler = new TickEventHandler();
        MinecraftForge.EVENT_BUS.register((Object)tickEventHandler);
    }

    @SubscribeEvent
    public static void onServerStopping(ServerStoppingEvent event) {
        LOGGER.info("Server stopping. Saving tax data and timestamp...");
        TaxManager.saveTaxData();
        TaxManager.saveLastTaxGenerationTime();
        try {
            ColonyClaimingRaidManager.endAllClaimingRaids();
        }
        catch (Exception e) {
            LOGGER.error("Error ending claiming raids on shutdown", (Throwable)e);
        }
    }

    public static int claimTax(IColony colony, int amount) {
        int claimedAmount;
        int colonyId = colony.getID();
        int storedTax = colonyTaxMap.getOrDefault(colonyId, 0);
        LOGGER.info("[TAX DEBUG] Colony {}: Stored tax = {}, Requested amount = {}", (Object)colony.getName(), (Object)storedTax, (Object)amount);
        if (storedTax <= 0) {
            LOGGER.info("[TAX DEBUG] No tax available to claim for colony {} (stored: {})", (Object)colony.getName(), (Object)storedTax);
            return 0;
        }
        if (FROZEN_COLONIES.contains(colonyId)) {
            if (TaxConfig.showTaxGenerationLogs()) {
                LOGGER.info("Tax claims for colony {} are currently frozen.", (Object)colony.getName());
            }
            return 0;
        }
        if (RaidManager.getActiveRaidForColony(colonyId) != null) {
            if (TaxConfig.showTaxGenerationLogs()) {
                LOGGER.info("Tax claims blocked for colony {} - colony is currently being raided.", (Object)colony.getName());
            }
            return 0;
        }
        WarData activeWar = WarSystem.ACTIVE_WARS.get(colonyId);
        if (activeWar == null) {
            for (WarData wd : WarSystem.ACTIVE_WARS.values()) {
                if (wd.getAttackerColony() == null || wd.getAttackerColony().getID() != colonyId) continue;
                activeWar = wd;
                break;
            }
        }
        if (activeWar != null) {
            if (TaxConfig.showTaxGenerationLogs()) {
                LOGGER.info("Tax claims blocked for colony {} - colony is currently at war.", (Object)colony.getName());
            }
            return 0;
        }
        if (amount == -1) {
            claimedAmount = storedTax;
            colonyTaxMap.put(colonyId, 0);
        } else {
            claimedAmount = Math.min(amount, storedTax);
            colonyTaxMap.put(colonyId, storedTax - claimedAmount);
        }
        if (TaxConfig.showTaxGenerationLogs()) {
            LOGGER.info("Claimed {} tax for colony {}", (Object)claimedAmount, (Object)colony.getName());
        }
        TaxManager.saveTaxData(true);
        return claimedAmount;
    }

    public static int claimTax(IColony colony) {
        return TaxManager.claimTax(colony, -1);
    }

    public static int getStoredTaxForColony(IColony colony) {
        return colonyTaxMap.getOrDefault(colony.getID(), 0);
    }

    public static void incrementTaxRevenue(IColony colony, int taxAmount) {
        int maxTax;
        int currentTax = colonyTaxMap.getOrDefault(colony.getID(), 0);
        if (currentTax < (maxTax = TaxConfig.getMaxTaxRevenue())) {
            int newTax = Math.min(currentTax + taxAmount, maxTax);
            colonyTaxMap.put(colony.getID(), newTax);
        }
    }

    public static void deductColonyTax(IColony colony, double percentage) {
        int currentTax = colonyTaxMap.getOrDefault(colony.getID(), 0);
        int deduction = (int)((double)currentTax * percentage);
        colonyTaxMap.put(colony.getID(), currentTax - deduction);
        if (TaxConfig.showTaxGenerationLogs()) {
            LOGGER.info("Deducted {} tax as penalty from colony {}", (Object)deduction, (Object)colony.getName());
        }
        TaxManager.saveTaxData();
    }

    public static void adjustTax(IColony colony, int delta) {
        int id = colony.getID();
        int current = colonyTaxMap.getOrDefault(id, 0);
        colonyTaxMap.put(id, current + delta);
    }

    private static double calculateColonyAverageHappiness(IColony colony) {
        try {
            double totalHappiness = 0.0;
            int adultCitizenCount = 0;
            for (ICitizenData citizen : colony.getCitizenManager().getCitizens()) {
                if (citizen == null || citizen.isChild()) continue;
                try {
                    CitizenHappinessHandler happinessHandler = ((CitizenData)citizen).getCitizenHappinessHandler();
                    if (happinessHandler == null) continue;
                    double happiness = happinessHandler.getHappiness(colony, citizen);
                    totalHappiness += happiness;
                    ++adultCitizenCount;
                }
                catch (Exception e) {
                    if (!TaxConfig.showTaxGenerationLogs()) continue;
                    LOGGER.debug("Could not get happiness for citizen {} in colony {}: {}", (Object)citizen.getName(), (Object)colony.getName(), (Object)e.getMessage());
                }
            }
            if (adultCitizenCount > 0) {
                return totalHappiness / (double)adultCitizenCount;
            }
            return 5.0;
        }
        catch (Exception e) {
            if (TaxConfig.showTaxGenerationLogs()) {
                LOGGER.warn("Error calculating colony happiness for {}: {}", (Object)colony.getName(), (Object)e.getMessage());
            }
            return 5.0;
        }
    }

    public static void generateTaxesForAllColonies() {
        if (serverInstance != null) {
            serverInstance.m_129785_().forEach(world -> {
                IColonyManager colonyManager = IMinecoloniesAPI.getInstance().getColonyManager();
                colonyManager.getColonies((Level)world).forEach(colony -> {
                    ServerPlayer player;
                    int tributePaid;
                    double boostPercentage;
                    int boostAmount;
                    int inactivityThreshold;
                    int lastContactHours;
                    int colonyId = colony.getID();
                    if (TaxManager.isGenerationDisabled(colonyId)) {
                        LOGGER.debug("Skipping tax generation for disabled colony {}", (Object)colonyId);
                        return;
                    }
                    if (TaxConfig.isColonyInactivityTaxPauseEnabled() && (lastContactHours = colony.getLastContactInHours()) >= (inactivityThreshold = TaxConfig.getColonyInactivityHoursThreshold())) {
                        if (TaxConfig.showTaxGenerationLogs()) {
                            LOGGER.info("Skipping tax generation for inactive colony {} - Last contact: {} hours ago (threshold: {} hours)", (Object)colony.getName(), (Object)lastContactHours, (Object)inactivityThreshold);
                        }
                        return;
                    }
                    int totalGeneratedTax = 0;
                    int totalBaseTax = 0;
                    int totalMaintenance = 0;
                    int buildingCount = 0;
                    int maxLimitHits = 0;
                    boolean hasDebt = false;
                    int guardTowerCount = 0;
                    int requiredGuardTowers = TaxConfig.getRequiredGuardTowersForBoost();
                    double colonyAvgHappiness = TaxManager.calculateColonyAverageHappiness(colony);
                    double happinessMultiplier = TaxConfig.calculateHappinessTaxMultiplier(colonyAvgHappiness);
                    for (IBuilding building : colony.getBuildingManager().getBuildings().values()) {
                        int maxTax;
                        if (building.getBuildingLevel() <= 0 || !building.isBuilt()) continue;
                        ++buildingCount;
                        String buildingType = building.getBuildingDisplayName();
                        int buildingLevel = building.getBuildingLevel();
                        double baseTax = TaxConfig.getBaseTaxForBuilding(buildingType);
                        double upgradeTax = TaxConfig.getUpgradeTaxForBuilding(buildingType) * (double)buildingLevel;
                        double rawTax = baseTax + upgradeTax;
                        int generatedTax = (int)(rawTax * happinessMultiplier);
                        totalBaseTax += (int)rawTax;
                        totalGeneratedTax += generatedTax;
                        int currentTax = colonyTaxMap.getOrDefault(colonyId, 0);
                        if (currentTax >= (maxTax = TaxConfig.getMaxTaxRevenue())) {
                            ++maxLimitHits;
                        } else {
                            TaxManager.incrementTaxRevenue(colony, generatedTax);
                        }
                        double baseMaintenance = TaxConfig.getBaseMaintenanceForBuilding(buildingType);
                        double upgradeMaintenance = TaxConfig.getUpgradeMaintenanceForBuilding(buildingType) * (double)buildingLevel;
                        int totalMaintenanceForBuilding = (int)(baseMaintenance + upgradeMaintenance);
                        totalMaintenance += totalMaintenanceForBuilding;
                        if (totalMaintenanceForBuilding <= 0) continue;
                        currentTax = colonyTaxMap.getOrDefault(colonyId, 0);
                        int newTax = currentTax - totalMaintenanceForBuilding;
                        int debtLimit = TaxConfig.getDebtLimit();
                        if (debtLimit > 0 && newTax < -debtLimit) {
                            newTax = -debtLimit;
                            hasDebt = true;
                        }
                        colonyTaxMap.put(colonyId, newTax);
                    }
                    int finalTaxBalance = colonyTaxMap.getOrDefault(colonyId, 0);
                    for (IBuilding building : colony.getBuildingManager().getBuildings().values()) {
                        if (building.getBuildingLevel() <= 0 || !building.isBuilt()) continue;
                        String displayName = building.getBuildingDisplayName();
                        String className = building.getClass().getName().toLowerCase();
                        String toString = building.toString().toLowerCase();
                        if ((displayName == null || !"Guard Tower".equalsIgnoreCase(displayName)) && !className.contains("guardtower") && !toString.contains("guardtower") && !toString.contains("guard_tower")) continue;
                        ++guardTowerCount;
                    }
                    if (guardTowerCount >= requiredGuardTowers && (boostAmount = (int)((double)totalGeneratedTax * (boostPercentage = TaxConfig.getGuardTowerTaxBoostPercentage()))) > 0) {
                        TaxManager.incrementTaxRevenue(colony, boostAmount);
                        finalTaxBalance = colonyTaxMap.getOrDefault(colonyId, 0);
                        if (TaxConfig.showTaxGenerationLogs()) {
                            LOGGER.info("Applied guard tower tax boost to colony {}: {} tax ({} guard towers, {}% boost)", (Object)colony.getName(), (Object)boostAmount, (Object)guardTowerCount, (Object)((int)(boostPercentage * 100.0)));
                        }
                    }
                    if ((tributePaid = VassalManager.handleTaxIncome(colony, totalGeneratedTax)) > 0) {
                        finalTaxBalance = colonyTaxMap.getOrDefault(colonyId, 0);
                    }
                    if (TaxConfig.showTaxGenerationLogs()) {
                        LOGGER.info("Tax cycle completed for colony {} - Buildings: {}, Generated: {}, Maintenance: {}, Final Balance: {}", (Object)colony.getName(), (Object)buildingCount, (Object)totalGeneratedTax, (Object)totalMaintenance, (Object)finalTaxBalance);
                        if (TaxConfig.isHappinessTaxModifierEnabled()) {
                            int happinessTaxImpact = totalGeneratedTax - totalBaseTax;
                            String impactType = happinessTaxImpact > 0 ? "BONUS" : (happinessTaxImpact < 0 ? "PENALTY" : "NEUTRAL");
                            LOGGER.info("Colony {} - Happiness: {}/10.0, Tax Impact: {} coins ({})", (Object)colonyId, (Object)String.format("%.1f", colonyAvgHappiness), (Object)happinessTaxImpact, (Object)impactType);
                        }
                        if (maxLimitHits > 0) {
                            LOGGER.info("Colony {} reached tax revenue maximum limit on {} building calculations (Max: {})", (Object)colony.getName(), (Object)maxLimitHits, (Object)TaxConfig.getMaxTaxRevenue());
                        }
                        if (hasDebt) {
                            LOGGER.info("Colony {} hit debt limit during maintenance deductions", (Object)colony.getName());
                        }
                    }
                    IPermissions permissions = colony.getPermissions();
                    Set officers = permissions.getPlayersByRank(permissions.getRankOfficer());
                    UUID ownerId = permissions.getOwner();
                    Set recipients = officers.stream().map(ColonyPlayer::getID).collect(Collectors.toSet());
                    recipients.add(ownerId);
                    for (UUID playerId : recipients) {
                        String statusKey;
                        ChatFormatting balanceColor;
                        player = serverInstance.m_6846_().m_11259_(playerId);
                        if (player == null) continue;
                        player.m_213846_((Component)Component.m_237110_((String)"message.minecolonytax.tax_report_header", (Object[])new Object[]{colony.getName()}).m_130944_(new ChatFormatting[]{ChatFormatting.GOLD, ChatFormatting.BOLD}));
                        player.m_213846_((Component)Component.m_237115_((String)"message.minecolonytax.tax_report_separator").m_130940_(ChatFormatting.GRAY));
                        player.m_213846_((Component)Component.m_237110_((String)"message.minecolonytax.tax_report_generation", (Object[])new Object[]{buildingCount, totalGeneratedTax}).m_130940_(ChatFormatting.GREEN));
                        if (totalMaintenance > 0) {
                            player.m_213846_((Component)Component.m_237110_((String)"message.minecolonytax.tax_report_maintenance", (Object[])new Object[]{totalMaintenance}).m_130940_(ChatFormatting.RED));
                        }
                        if (TaxConfig.isHappinessTaxModifierEnabled()) {
                            int happinessTaxImpact = totalGeneratedTax - totalBaseTax;
                            ChatFormatting happinessColor = happinessTaxImpact > 0 ? ChatFormatting.GREEN : (happinessTaxImpact < 0 ? ChatFormatting.RED : ChatFormatting.YELLOW);
                            String impactSign = happinessTaxImpact > 0 ? "+" : "";
                            player.m_213846_((Component)Component.m_237110_((String)"message.minecolonytax.tax_report_happiness", (Object[])new Object[]{String.format("%.1f", colonyAvgHappiness), impactSign + happinessTaxImpact}).m_130940_(happinessColor));
                        }
                        if (guardTowerCount >= requiredGuardTowers) {
                            double boostPercentage2 = TaxConfig.getGuardTowerTaxBoostPercentage();
                            int boostAmount2 = (int)((double)totalGeneratedTax * boostPercentage2);
                            player.m_213846_((Component)Component.m_237110_((String)"message.minecolonytax.tax_report_guard_boost", (Object[])new Object[]{guardTowerCount, (int)(boostPercentage2 * 100.0), boostAmount2}).m_130940_(ChatFormatting.BLUE));
                        }
                        if (tributePaid > 0) {
                            player.m_213846_((Component)Component.m_237110_((String)"message.minecolonytax.tax_report_tribute", (Object[])new Object[]{tributePaid}).m_130940_(ChatFormatting.DARK_PURPLE));
                        }
                        player.m_213846_((Component)Component.m_237115_((String)"message.minecolonytax.tax_report_separator").m_130940_(ChatFormatting.GRAY));
                        if (finalTaxBalance < 0) {
                            balanceColor = ChatFormatting.DARK_RED;
                            statusKey = "message.minecolonytax.tax_report_status_debt";
                        } else if ((double)finalTaxBalance >= (double)TaxConfig.getMaxTaxRevenue() * 0.9) {
                            balanceColor = ChatFormatting.YELLOW;
                            statusKey = "message.minecolonytax.tax_report_status_near_max";
                        } else {
                            balanceColor = ChatFormatting.GREEN;
                            statusKey = "message.minecolonytax.tax_report_status_healthy";
                        }
                        player.m_213846_((Component)Component.m_237110_((String)"message.minecolonytax.tax_report_balance", (Object[])new Object[]{finalTaxBalance}).m_130944_(new ChatFormatting[]{balanceColor, ChatFormatting.BOLD}));
                        player.m_213846_((Component)Component.m_237110_((String)statusKey, (Object[])new Object[]{finalTaxBalance < 0 ? Math.abs(finalTaxBalance) : TaxConfig.getMaxTaxRevenue()}).m_130940_(balanceColor));
                        player.m_213846_((Component)Component.m_237115_((String)"message.minecolonytax.tax_report_footer").m_130940_(ChatFormatting.GRAY));
                    }
                    if (totalMaintenance > totalGeneratedTax) {
                        for (UUID playerId : recipients) {
                            player = serverInstance.m_6846_().m_11259_(playerId);
                            if (player == null) continue;
                            player.m_213846_((Component)Component.m_237110_((String)"message.minecolonytax.debt_warning", (Object[])new Object[]{colony.getName(), totalMaintenance, totalGeneratedTax}));
                        }
                    }
                });
            });
            TaxManager.saveTaxData();
            if (TaxConfig.showTaxGenerationLogs()) {
                LOGGER.info("Tax generation cycle completed for all colonies");
            }
        }
    }

    public static void updateTaxForBuilding(IColony colony, IBuilding building, int currentLevel) {
        if (currentLevel > 0 && building.isBuilt()) {
            String buildingType = building.getBuildingDisplayName();
            double baseTax = TaxConfig.getBaseTaxForBuilding(buildingType);
            double upgradeTax = TaxConfig.getUpgradeTaxForBuilding(buildingType) * (double)currentLevel;
            int totalTax = (int)(baseTax + upgradeTax);
            TaxManager.incrementTaxRevenue(colony, totalTax);
            if (totalTax > 0) {
                LOGGER.debug("Generated {} tax for building {} (level {}) in colony {}", (Object)totalTax, (Object)buildingType, (Object)currentLevel, (Object)colony.getName());
            }
        }
    }

    private static void saveTaxData() {
        TaxManager.saveTaxData(false);
    }

    private static void saveTaxData(boolean logSave) {
        File file = new File(TAX_DATA_FILE);
        file.getParentFile().mkdirs();
        try (FileWriter writer = new FileWriter(file);){
            GSON.toJson(colonyTaxMap, (Appendable)writer);
            if (logSave && TaxConfig.showTaxGenerationLogs()) {
                LOGGER.info("Saved tax data to file.");
            }
        }
        catch (IOException e) {
            LOGGER.error("Error saving tax data", (Throwable)e);
        }
    }

    private static void loadTaxData(MinecraftServer server) {
        File taxFile = new File(TAX_DATA_FILE);
        if (taxFile.exists()) {
            try (FileReader reader = new FileReader(taxFile);){
                Type taxDataType = new TypeToken<Map<Integer, Integer>>(){}.getType();
                Map loadedData = (Map)GSON.fromJson((Reader)reader, taxDataType);
                if (loadedData != null) {
                    colonyTaxMap.putAll(loadedData);
                    LOGGER.info("Loaded tax data from file.");
                }
            }
            catch (IOException e) {
                LOGGER.error("Error loading tax data", (Throwable)e);
            }
        } else {
            LOGGER.info("No existing tax data file found at: {}", (Object)taxFile.getAbsolutePath());
        }
    }

    public static void freezeColonyTax(final int colonyId, int freezeHours) {
        FROZEN_COLONIES.add(colonyId);
        if (TaxConfig.showTaxGenerationLogs()) {
            LOGGER.info("Colony {} tax is frozen for {} hours.", (Object)colonyId, (Object)freezeHours);
        }
        new Timer(true).schedule(new TimerTask(){

            @Override
            public void run() {
                FROZEN_COLONIES.remove(colonyId);
                if (TaxConfig.showTaxGenerationLogs()) {
                    LOGGER.info("Colony {} tax freeze expired.", (Object)colonyId);
                }
            }
        }, TimeUnit.HOURS.toMillis(freezeHours));
    }

    public static int payTaxDebt(IColony colony, int amount) {
        int colonyId = colony.getID();
        int currentTax = colonyTaxMap.getOrDefault(colonyId, 0);
        colonyTaxMap.put(colonyId, currentTax + amount);
        if (TaxConfig.showTaxGenerationLogs()) {
            LOGGER.info("Colony {} tax payment of {}. New tax value: {}", (Object)colony.getName(), (Object)amount, (Object)colonyTaxMap.get(colonyId));
        }
        TaxManager.saveTaxData(true);
        return amount;
    }

    public static void disableTaxGeneration(int colonyId) {
        DISABLED_GENERATION.add(colonyId);
        if (TaxConfig.showTaxGenerationLogs()) {
            LOGGER.info("Tax generation disabled for colony {}", (Object)colonyId);
        }
    }

    public static void enableTaxGeneration(int colonyId) {
        DISABLED_GENERATION.remove(colonyId);
        if (TaxConfig.showTaxGenerationLogs()) {
            LOGGER.info("Tax generation enabled for colony {}", (Object)colonyId);
        }
    }

    public static boolean isGenerationDisabled(int colonyId) {
        return DISABLED_GENERATION.contains(colonyId);
    }

    private static void loadLastTaxGenerationTime() {
        File timestampFile = new File(TAX_TIMESTAMP_FILE);
        if (timestampFile.exists()) {
            try (FileReader reader = new FileReader(timestampFile);){
                Type timestampType = new TypeToken<Map<String, Long>>(){}.getType();
                Map timestampData = (Map)GSON.fromJson((Reader)reader, timestampType);
                if (timestampData != null && timestampData.containsKey("lastTaxGeneration")) {
                    long loadedTimestamp = (Long)timestampData.get("lastTaxGeneration");
                    long currentTime = System.currentTimeMillis();
                    long oneYearAgo = currentTime - 31536000000L;
                    long oneYearFromNow = currentTime + 31536000000L;
                    if (loadedTimestamp < oneYearAgo || loadedTimestamp > oneYearFromNow) {
                        LOGGER.warn("Loaded timestamp appears corrupted ({}), starting fresh", (Object)new Date(loadedTimestamp));
                        lastTaxGenerationTime = 0L;
                    } else {
                        lastTaxGenerationTime = loadedTimestamp;
                        if (TaxConfig.showTaxGenerationLogs()) {
                            long minutesAgo = (currentTime - lastTaxGenerationTime) / 60000L;
                            LOGGER.info("Loaded last tax generation timestamp: {} minutes ago ({})", (Object)minutesAgo, (Object)new Date(lastTaxGenerationTime));
                        }
                    }
                } else {
                    LOGGER.warn("Timestamp file exists but is malformed, starting fresh");
                    lastTaxGenerationTime = 0L;
                }
            }
            catch (Exception e) {
                LOGGER.warn("Error loading tax generation timestamp, will start fresh: {}", (Object)e.getMessage());
                lastTaxGenerationTime = 0L;
                try {
                    timestampFile.delete();
                }
                catch (Exception ex) {
                    LOGGER.debug("Could not delete corrupted timestamp file: {}", (Object)ex.getMessage());
                }
            }
        } else {
            if (TaxConfig.showTaxGenerationLogs()) {
                LOGGER.info("No existing tax generation timestamp found, starting fresh");
            }
            lastTaxGenerationTime = 0L;
        }
    }

    private static void saveLastTaxGenerationTime() {
        if (lastTaxGenerationTime <= 0L) {
            if (TaxConfig.showTaxGenerationLogs()) {
                LOGGER.debug("Skipping save of invalid timestamp: {}", (Object)lastTaxGenerationTime);
            }
            return;
        }
        File timestampFile = new File(TAX_TIMESTAMP_FILE);
        timestampFile.getParentFile().mkdirs();
        try (FileWriter writer = new FileWriter(timestampFile);){
            HashMap<String, Long> timestampData = new HashMap<String, Long>();
            timestampData.put("lastTaxGeneration", lastTaxGenerationTime);
            timestampData.put("version", 1L);
            GSON.toJson(timestampData, (Appendable)writer);
            if (TaxConfig.showTaxGenerationLogs()) {
                LOGGER.debug("Saved tax generation timestamp: {} ({})", (Object)lastTaxGenerationTime, (Object)new Date(lastTaxGenerationTime));
            }
        }
        catch (IOException e) {
            LOGGER.error("CRITICAL: Failed to save tax generation timestamp! Tax intervals may be affected after restart: {}", (Object)e.getMessage());
        }
    }

    static {
        FROZEN_COLONIES = ConcurrentHashMap.newKeySet();
        DISABLED_GENERATION = ConcurrentHashMap.newKeySet();
        tickEventHandler = null;
        lastTaxGenerationTime = 0L;
    }

    public static class TickEventHandler {
        private int tickCount = 0;
        private int abandonmentTickCount = 0;
        private int cleanupTickCount = 0;
        private int nullOwnerCheckCount = 0;

        @SubscribeEvent
        public void onServerTick(TickEvent.ServerTickEvent event) {
            if (event.phase == TickEvent.Phase.END) {
                ++this.tickCount;
                ++this.abandonmentTickCount;
                ++this.cleanupTickCount;
                ++this.nullOwnerCheckCount;
                if (this.nullOwnerCheckCount >= 100) {
                    this.nullOwnerCheckCount = 0;
                    try {
                        ColonyAbandonmentManager.emergencyFixAllNullOwners();
                    }
                    catch (Exception e) {
                        LOGGER.error("Failed automatic null owner fix", (Throwable)e);
                    }
                }
                if (this.tickCount >= 20) {
                    this.tickCount = 0;
                    this.checkForTaxGeneration();
                }
                if (this.abandonmentTickCount >= 72000) {
                    this.abandonmentTickCount = 0;
                    this.checkColonyAbandonment();
                }
                if (this.cleanupTickCount >= 36000) {
                    this.cleanupTickCount = 0;
                    this.runPeriodicAbandonedCleanup();
                }
                if (this.tickCount == 0) {
                    ColonyClaimingRaidManager.updateClaimingRaids();
                }
                if (this.abandonmentTickCount % 6000 == 0) {
                    this.checkForOfficerChangesInAbandonedColonies();
                }
            }
        }

        private void runPeriodicAbandonedCleanup() {
            try {
                LOGGER.debug("Running periodic [abandoned] entries cleanup...");
                ColonyAbandonmentManager.cleanupAllColoniesAbandonedEntries();
            }
            catch (Exception e) {
                LOGGER.error("Error during periodic [abandoned] cleanup", (Throwable)e);
            }
        }

        private void checkForOfficerChangesInAbandonedColonies() {
            try {
                if (serverInstance == null) {
                    return;
                }
                for (IColony colony : IMinecoloniesAPI.getInstance().getColonyManager().getAllColonies()) {
                    try {
                        if (!ColonyAbandonmentManager.isColonyAbandoned(colony)) continue;
                        ColonyAbandonmentManager.checkForNewOfficers(colony);
                    }
                    catch (Exception e) {
                        LOGGER.error("Error checking colony {} for officer changes: {}", (Object)colony.getName(), (Object)e.getMessage());
                    }
                }
            }
            catch (Exception e) {
                LOGGER.error("Error during officer change check", (Throwable)e);
            }
        }

        private void checkForTaxGeneration() {
            long currentTime = System.currentTimeMillis();
            long intervalMs = Math.max(60000L, (long)TaxConfig.getTaxIntervalInMinutes() * 60L * 1000L);
            if (lastTaxGenerationTime > currentTime + 60000L) {
                if (TaxConfig.showTaxGenerationLogs()) {
                    LOGGER.warn("Last tax generation timestamp is in the future! Clock may have changed. Resetting timestamp.");
                }
                lastTaxGenerationTime = currentTime - intervalMs;
                TaxManager.saveLastTaxGenerationTime();
            }
            if (lastTaxGenerationTime == 0L || currentTime - lastTaxGenerationTime >= intervalMs) {
                if (TaxConfig.showTaxGenerationLogs()) {
                    if (lastTaxGenerationTime == 0L) {
                        LOGGER.info("First tax generation triggered (interval: {} minutes)", (Object)TaxConfig.getTaxIntervalInMinutes());
                    } else {
                        long elapsedMinutes = (currentTime - lastTaxGenerationTime) / 60000L;
                        LOGGER.info("Tax generation triggered after {} minutes elapsed (interval: {} minutes)", (Object)elapsedMinutes, (Object)TaxConfig.getTaxIntervalInMinutes());
                    }
                }
                lastTaxGenerationTime = currentTime;
                TaxManager.saveLastTaxGenerationTime();
                TaxManager.generateTaxesForAllColonies();
            }
        }

        private void checkColonyAbandonment() {
            if (serverInstance != null) {
                try {
                    ColonyAbandonmentManager.checkColoniesForAbandonment(serverInstance);
                }
                catch (Exception e) {
                    LOGGER.error("Error during colony abandonment check", (Throwable)e);
                }
            }
        }
    }
}

