/*
 * Decompiled with CFR 0.152.
 */
package net.machiavelli.minecolonytax.requirements;

import com.minecolonies.api.colony.IColony;
import com.minecolonies.api.colony.buildings.IBuilding;
import java.util.ArrayList;
import java.util.List;
import net.machiavelli.minecolonytax.TaxConfig;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class BuildingRequirementsManager {
    private static final Logger LOGGER = LogManager.getLogger(BuildingRequirementsManager.class);

    public static RequirementResult checkRaidRequirements(IColony colony) {
        if (!TaxConfig.isRaidBuildingRequirementsEnabled()) {
            return new RequirementResult(true, "Building requirements disabled for raids.");
        }
        String requirements = TaxConfig.getRaidBuildingRequirements();
        return BuildingRequirementsManager.checkBuildingRequirements(colony, requirements, "raid");
    }

    public static RequirementResult checkWarRequirements(IColony colony) {
        if (!TaxConfig.isWarBuildingRequirementsEnabled()) {
            return new RequirementResult(true, "Building requirements disabled for wars.");
        }
        String requirements = TaxConfig.getWarBuildingRequirements();
        return BuildingRequirementsManager.checkBuildingRequirements(colony, requirements, "war");
    }

    public static RequirementResult checkClaimingRequirements(IColony colony) {
        String requirements = TaxConfig.getClaimingBuildingRequirements();
        return BuildingRequirementsManager.checkBuildingRequirements(colony, requirements, "claiming");
    }

    private static RequirementResult checkBuildingRequirements(IColony colony, String requirements, String actionType) {
        LOGGER.info("Checking building requirements for {}: '{}' (Colony ID: {})", (Object)actionType, (Object)requirements, (Object)colony.getID());
        if (requirements == null || requirements.trim().isEmpty()) {
            LOGGER.info("No building requirements configured for {}", (Object)actionType);
            return new RequirementResult(true, "No building requirements configured for " + actionType + ".");
        }
        try {
            List<BuildingRequirement> parsedRequirements = BuildingRequirementsManager.parseRequirements(requirements);
            LOGGER.info("Parsed {} requirements: {}", (Object)parsedRequirements.size(), parsedRequirements);
            ArrayList<String> missingRequirements = new ArrayList<String>();
            for (BuildingRequirement requirement : parsedRequirements) {
                LOGGER.info("Checking requirement: {} level {}+ (need {})", (Object)requirement.buildingType, (Object)requirement.requiredLevel, (Object)requirement.requiredAmount);
                int availableCount = BuildingRequirementsManager.countBuildingsOfTypeAndLevel(colony, requirement.buildingType, requirement.requiredLevel);
                LOGGER.info("Available count for {} level {}+: {}", (Object)requirement.buildingType, (Object)requirement.requiredLevel, (Object)availableCount);
                if (availableCount < requirement.requiredAmount) {
                    String missing = requirement.toString() + " (current: " + availableCount + ")";
                    missingRequirements.add(missing);
                    LOGGER.warn("Missing requirement: {}", (Object)missing);
                    continue;
                }
                LOGGER.info("Requirement met: {} (has {} out of {} required)", (Object)requirement, (Object)availableCount, (Object)requirement.requiredAmount);
            }
            if (missingRequirements.isEmpty()) {
                LOGGER.info("All building requirements met for {}", (Object)actionType);
                return new RequirementResult(true, "All building requirements met for " + actionType + ".");
            }
            String message = "Missing building requirements for " + actionType + ": " + String.join((CharSequence)", ", missingRequirements);
            LOGGER.warn("Building requirements check failed: {}", (Object)message);
            return new RequirementResult(false, message, missingRequirements);
        }
        catch (Exception e) {
            LOGGER.error("Error checking building requirements for {}: {}", (Object)actionType, (Object)requirements, (Object)e);
            LOGGER.error("Exception details: ", (Throwable)e);
            return new RequirementResult(false, "Error checking building requirements - denying " + actionType + " for safety.");
        }
    }

    private static List<BuildingRequirement> parseRequirements(String requirements) {
        String[] pairs;
        ArrayList<BuildingRequirement> result = new ArrayList<BuildingRequirement>();
        if (requirements == null || requirements.trim().isEmpty()) {
            return result;
        }
        for (String pair : pairs = requirements.split(",")) {
            String[] parts = pair.trim().split(":");
            if (parts.length < 2 || parts.length > 3) {
                LOGGER.warn("Invalid building requirement format: {}", (Object)pair);
                continue;
            }
            try {
                String buildingType = parts[0].trim();
                int level = Integer.parseInt(parts[1].trim());
                int amount = parts.length == 3 ? Integer.parseInt(parts[2].trim()) : 1;
                result.add(new BuildingRequirement(buildingType, level, amount));
            }
            catch (NumberFormatException e) {
                LOGGER.warn("Invalid number in building requirement: {}", (Object)pair);
            }
        }
        return result;
    }

    private static int countBuildingsOfTypeAndLevel(IColony colony, String buildingType, int minLevel) {
        int count = 0;
        try {
            LOGGER.debug("Checking building requirements for colony {}: {} level {}+", (Object)colony.getID(), (Object)buildingType, (Object)minLevel);
            for (IBuilding building : colony.getBuildingManager().getBuildings().values()) {
                String actualBuildingType = building.getBuildingType().getRegistryName().m_135815_().toLowerCase();
                int buildingLevel = building.getBuildingLevel();
                LOGGER.debug("Found building: {} level {}", (Object)actualBuildingType, (Object)buildingLevel);
                if (BuildingRequirementsManager.matchesBuildingType(actualBuildingType, buildingType)) {
                    LOGGER.debug("Building type {} matches requirement {}", (Object)actualBuildingType, (Object)buildingType);
                    if (buildingLevel >= minLevel) {
                        LOGGER.debug("Building meets level requirement. Count: {}", (Object)(++count));
                        continue;
                    }
                    LOGGER.debug("Building level {} does not meet minimum requirement {}", (Object)buildingLevel, (Object)minLevel);
                    continue;
                }
                LOGGER.debug("Building type {} does not match requirement {}", (Object)actualBuildingType, (Object)buildingType);
            }
            LOGGER.debug("Final count for {} level {}+: {}", (Object)buildingType, (Object)minLevel, (Object)count);
        }
        catch (Exception e) {
            LOGGER.error("Error counting buildings of type {} for colony {}", (Object)buildingType, (Object)colony.getID(), (Object)e);
        }
        return count;
    }

    private static boolean matchesBuildingType(String actualType, String requiredType) {
        if (actualType.equals(requiredType)) {
            return true;
        }
        return switch (requiredType) {
            case "townhall" -> actualType.contains("townhall");
            case "guardtower" -> {
                if (actualType.contains("guardtower") || actualType.contains("guard")) {
                    yield true;
                }
                yield false;
            }
            case "buildershut", "buildershop" -> actualType.contains("builder");
            case "house", "residential" -> {
                if (actualType.contains("house") || actualType.contains("residence") || actualType.contains("citizen")) {
                    yield true;
                }
                yield false;
            }
            case "barracks" -> actualType.contains("barracks");
            case "archery" -> actualType.contains("archery");
            case "combatacademy" -> actualType.contains("combat");
            case "mine" -> actualType.contains("mine");
            case "farm" -> actualType.contains("farm");
            case "warehouse" -> actualType.contains("warehouse");
            case "deliveryman" -> actualType.contains("deliveryman");
            default -> actualType.contains(requiredType);
        };
    }

    public static List<String> getFormattedRequirements(String requirements) {
        ArrayList<String> formatted = new ArrayList<String>();
        try {
            List<BuildingRequirement> parsed = BuildingRequirementsManager.parseRequirements(requirements);
            for (BuildingRequirement req : parsed) {
                formatted.add(req.toString());
            }
        }
        catch (Exception e) {
            LOGGER.error("Error formatting requirements: {}", (Object)requirements, (Object)e);
            formatted.add("Error parsing requirements");
        }
        return formatted;
    }

    public static class RequirementResult {
        public final boolean meetsRequirements;
        public final String message;
        public final List<String> missingRequirements;

        public RequirementResult(boolean meetsRequirements, String message, List<String> missingRequirements) {
            this.meetsRequirements = meetsRequirements;
            this.message = message;
            this.missingRequirements = missingRequirements != null ? missingRequirements : new ArrayList();
        }

        public RequirementResult(boolean meetsRequirements, String message) {
            this(meetsRequirements, message, null);
        }
    }

    public static class BuildingRequirement {
        public final String buildingType;
        public final int requiredLevel;
        public final int requiredAmount;

        public BuildingRequirement(String buildingType, int requiredLevel, int requiredAmount) {
            this.buildingType = buildingType.toLowerCase();
            this.requiredLevel = requiredLevel;
            this.requiredAmount = requiredAmount;
        }

        public String toString() {
            if (this.requiredAmount == 1) {
                return this.buildingType + " level " + this.requiredLevel + "+";
            }
            return this.requiredAmount + "x " + this.buildingType + " level " + this.requiredLevel + "+";
        }

        public boolean isGuardRelated() {
            return this.buildingType.contains("guard") || this.buildingType.contains("tower") || this.buildingType.contains("barracks") || this.buildingType.contains("archery") || this.buildingType.contains("combat");
        }
    }
}

