package com.github.thedeathlycow.thermoo.api.environment.provider;

import com.mojang.serialization.Codec;
import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;
import java.util.Optional;
import net.minecraft.class_1937;
import net.minecraft.class_1944;
import net.minecraft.class_1959;
import net.minecraft.class_2338;
import net.minecraft.class_6880;
import net.minecraft.class_9323;

/**
 * Used to pick between two child providers based on a light level threshold. Can filter for {@link class_1944} and apply
 * or ignore {@link class_1937#method_8594() ambient darkness} to sky light.
 */
public class LightThresholdLightProvider implements EnvironmentProvider {
    public static final MapCodec<LightThresholdLightProvider> CODEC = RecordCodecBuilder.mapCodec(
            instance -> instance.group(
                    Codec.stringResolver(
                                    light -> light.name().toLowerCase(),
                                    name -> class_1944.valueOf(name.toUpperCase())
                            )
                            .optionalFieldOf("light_type")
                            .forGetter(LightThresholdLightProvider::lightLayer),
                    Codec.BOOL
                            .optionalFieldOf("apply_ambient_darkness", true)
                            .forGetter(LightThresholdLightProvider::applyAmbientDarkness),
                    Codec.intRange(0, 15)
                            .fieldOf("threshold")
                            .forGetter(LightThresholdLightProvider::threshold),
                    EnvironmentProvider.HOLDER_CODEC
                            .fieldOf("above")
                            .forGetter(LightThresholdLightProvider::above),
                    EnvironmentProvider.HOLDER_CODEC
                            .fieldOf("below")
                            .forGetter(LightThresholdLightProvider::below)
            ).apply(instance, LightThresholdLightProvider::new)
    );

    private final Optional<class_1944> lightLayer;
    private final boolean applyAmbientDarkness;
    private final int threshold;
    private final class_6880<EnvironmentProvider> above;
    private final class_6880<EnvironmentProvider> below;

    /**
     * Creates a new builder with the mandatory threshold, above, and below fields
     *
     * @param threshold The light level threshold - must be between 0 and 15 (inclusive)
     * @param above     The provider to use when a positions light level is at or above the {@code threshold}. Must not be null.
     * @param below     The provider to use when a positions light level is below the {@code threshold}. Must not be null.
     * @return Returns a new builder instance
     */
    @Contract("_,_,_->new")
    public static Builder builder(
            int threshold,
            @NotNull class_6880<EnvironmentProvider> above,
            @NotNull class_6880<EnvironmentProvider> below
    ) {
        if (threshold < 0 || threshold > 15) {
            throw new IllegalArgumentException("Threshold must be between 0 and 15 but is " + threshold);
        }
        Objects.requireNonNull(above);
        Objects.requireNonNull(below);

        return new Builder(threshold, above, below);
    }

    private LightThresholdLightProvider(
            Optional<class_1944> lightLayer,
            boolean applyAmbientDarkness,
            int threshold,
            class_6880<EnvironmentProvider> above,
            class_6880<EnvironmentProvider> below
    ) {
        this.lightLayer = lightLayer;
        this.applyAmbientDarkness = applyAmbientDarkness;
        this.threshold = threshold;
        this.above = above;
        this.below = below;
    }

    /**
     * Builds the current components of the world position based on light level. If the light level of the position is at
     * or above the threshold then uses the {@link #above()} provider. Otherwise, uses the {@link #below()} provider.
     * <p>
     * Filters for sky/block light and ambient darkness if requested.
     *
     * @param level   The world/level being queried
     * @param pos     The position in the world to query
     * @param biome   The biome at the position in the world
     * @param builder Component map builder to append to
     */
    @Override
    public void buildCurrentComponents(class_1937 level, class_2338 pos, class_6880<class_1959> biome, class_9323.class_9324 builder) {
        int lightLevel = this.lightLayer
                .map(type -> level.method_8314(type, pos))
                .orElseGet(() -> level.method_22339(pos));

        if (this.applyAmbientDarkness && this.lightLayer.orElse(null) == class_1944.field_9284) {
            lightLevel -= level.method_8594();
        }

        if (lightLevel >= this.threshold) {
            this.above.comp_349().buildCurrentComponents(level, pos, biome, builder);
        } else {
            this.below.comp_349().buildCurrentComponents(level, pos, biome, builder);
        }
    }

    @Override
    public EnvironmentProviderType<LightThresholdLightProvider> getType() {
        return EnvironmentProviderTypes.LIGHT_THRESHOLD;
    }

    /**
     * The optional light type of this provider. If not specified, uses {@link net.minecraft.class_4538#method_22339(class_2338)}
     * to determine light level.
     *
     * @return Returns the light layer of this provider.
     * @deprecated This method is named based on Yarn, use {@link #lightLayer()} to better conform to official mappings.
     */
    @Deprecated(since = "8.1.0", forRemoval = true)
    public Optional<class_1944> lightType() {
        return this.lightLayer;
    }

    /**
     * The optional light layer of this provider. If not specified, uses {@link net.minecraft.class_4538#method_22339(class_2338)}
     * to determine light level.
     *
     * @return Returns the light layer of this provider.
     */
    public Optional<class_1944> lightLayer() {
        return this.lightLayer;
    }

    /**
     * Whether ambient darkness should be applied when using the skylight light type (default: true)
     */
    public boolean applyAmbientDarkness() {
        return this.applyAmbientDarkness;
    }

    /**
     * Light level threshold that determines whether to use {@link #above()} or {@link #below()} when finding the
     * environment components. Must be between 0 and 15 (inclusive).
     */
    public int threshold() {
        return this.threshold;
    }

    /**
     * The provider to use when the light level is at or above the {@link #threshold()}
     */
    public class_6880<EnvironmentProvider> above() {
        return this.above;
    }

    /**
     * The provider to use when the light level is below the {@link #threshold()}
     */
    public class_6880<EnvironmentProvider> below() {
        return this.below;
    }

    /**
     * Builder class for light threshold providers
     */
    public static final class Builder {
        @Nullable
        private class_1944 lightType = null;
        private boolean applyAmbientDarkness = true;
        private final int threshold;
        private final class_6880<EnvironmentProvider> above;
        private final class_6880<EnvironmentProvider> below;

        private Builder(int threshold, class_6880<EnvironmentProvider> above, class_6880<EnvironmentProvider> below) {
            this.threshold = threshold;
            this.above = above;
            this.below = below;
        }

        /**
         * Ignore the ambient darkness of the position when using sky light
         *
         * @return Returns this builder
         */
        @Contract("->this")
        public Builder ignoreAmbientDarkness() {
            this.applyAmbientDarkness = false;
            return this;
        }

        /**
         * Sets a light type to filter on.
         *
         * @return Returns this builder
         */
        @Contract("_->this")
        public Builder withLightType(class_1944 lightType) {
            this.lightType = lightType;
            return this;
        }

        /**
         * @return Returns a new provider with the parameters of this builder
         */
        @Contract("->new")
        public LightThresholdLightProvider build() {
            return new LightThresholdLightProvider(
                    Optional.ofNullable(this.lightType),
                    this.applyAmbientDarkness,
                    this.threshold,
                    this.above,
                    this.below
            );
        }
    }
}