package com.github.thedeathlycow.thermoo.api.command;

import com.mojang.brigadier.Command;
import com.mojang.brigadier.arguments.IntegerArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import org.jetbrains.annotations.Contract;

import java.util.function.Supplier;
import net.minecraft.class_12099;
import net.minecraft.class_1297;
import net.minecraft.class_1309;
import net.minecraft.class_2168;
import net.minecraft.class_2186;
import net.minecraft.class_2561;
import net.minecraft.class_3532;

import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;

/**
 * Command relating to soaking. Allows soaking values to be modified in game.
 * <p>
 * Usage:
 * <p>
 * {@code thermoo soaking (get|set|add|remove) <target> <args>}
 */
public final class SoakingCommand {

    /**
     * Supplier for creating a new soaking command builder to be registered to the Minecraft server
     * <p>
     * Registered by the default implementation of this API.
     */
    public static final Supplier<LiteralArgumentBuilder<class_2168>> COMMAND_BUILDER = SoakingCommand::buildCommand;

    private static final String TARGET_KEY = "target";
    private static final String SCALE_KEY = "scale";
    private static final String MIN_KEY = "min";
    private static final String MAX_KEY = "max";
    private static final String VALUE_KEY = "value";

    @Contract("->new")
    private static LiteralArgumentBuilder<class_2168> buildCommand() {
        return method_9247("thermoo").then(
                (method_9247("soaking").requires(src -> src.method_75037()
                        .hasPermission(class_12099.field_63210)))
                        .then(buildGetCommand())
                        .then(buildSetCommand())
                        .then(buildAddCommand())
                        .then(buildRemoveCommand())
        );
    }

    private static LiteralArgumentBuilder<class_2168> buildRemoveCommand() {
        return method_9247("remove")
                .then(
                        method_9244(TARGET_KEY, class_2186.method_9309())
                                .then(
                                        method_9244(VALUE_KEY, IntegerArgumentType.integer(0))
                                                .executes(
                                                        context -> {
                                                            return remove(
                                                                    context.getSource(),
                                                                    class_2186.method_9313(context, TARGET_KEY),
                                                                    IntegerArgumentType.getInteger(context, VALUE_KEY)
                                                            );
                                                        }
                                                )
                                )
                );
    }

    private static LiteralArgumentBuilder<class_2168> buildAddCommand() {
        return method_9247("add")
                .then(
                        method_9244(TARGET_KEY, class_2186.method_9309())
                                .then(
                                        method_9244(VALUE_KEY, IntegerArgumentType.integer(0))
                                                .executes(
                                                        context -> {
                                                            return add(
                                                                    context.getSource(),
                                                                    class_2186.method_9313(context, TARGET_KEY),
                                                                    IntegerArgumentType.getInteger(context, VALUE_KEY)
                                                            );
                                                        }
                                                )
                                )
                );
    }

    private static LiteralArgumentBuilder<class_2168> buildSetCommand() {
        return method_9247("set")
                .then(
                        method_9244(TARGET_KEY, class_2186.method_9309())
                                .then(
                                        method_9244(VALUE_KEY, IntegerArgumentType.integer(0))
                                                .executes(
                                                        context -> {
                                                            return set(
                                                                    context.getSource(),
                                                                    class_2186.method_9313(context, TARGET_KEY),
                                                                    IntegerArgumentType.getInteger(context, VALUE_KEY)
                                                            );
                                                        }
                                                )
                                )
                );
    }

    private static LiteralArgumentBuilder<class_2168> buildGetCommand() {
        Command<class_2168> getCurrent = context -> {
            return getCurrent(
                    context.getSource(),
                    class_2186.method_9313(context, TARGET_KEY)
            );
        };

        var getScale = method_9247(SCALE_KEY)
                .then(
                        method_9244(SCALE_KEY, IntegerArgumentType.integer(1))
                                .executes(
                                        context -> {
                                            return getScale(
                                                    context.getSource(),
                                                    class_2186.method_9313(context, TARGET_KEY),
                                                    IntegerArgumentType.getInteger(context, SCALE_KEY)
                                            );
                                        }
                                )
                )
                .executes(
                        context -> {
                            return getScale(
                                    context.getSource(),
                                    class_2186.method_9313(context, TARGET_KEY),
                                    100
                            );
                        }
                );

        var getMin = method_9247(MIN_KEY)
                .executes(
                        context -> {
                            return getMin(
                                    context.getSource(),
                                    class_2186.method_9313(context, TARGET_KEY)
                            );
                        }
                );

        var getMax = method_9247(MAX_KEY)
                .executes(
                        context -> {
                            return getMax(
                                    context.getSource(),
                                    class_2186.method_9313(context, TARGET_KEY)
                            );
                        }
                );

        return method_9247("get")
                .then(
                        method_9244(TARGET_KEY, class_2186.method_9309())
                                .executes(getCurrent)
                                .then(method_9247("current").executes(getCurrent))
                                .then(getScale)
                                .then(getMin)
                                .then(getMax)
                );
    }

    private static int remove(class_2168 source, class_1297 target, int value) throws CommandSyntaxException {
        if (target instanceof class_1309 entity) {
            entity.thermoo$addWetTicks(-value);

            source.method_9226(
                    () -> class_2561.method_43469(
                            "commands.thermoo.soaking.remove.success",
                            target.method_5476(),
                            value,
                            entity.thermoo$getWetTicks()
                    ), true
            );

            return entity.thermoo$getWetTicks();
        } else {
            throw TemperatureCommand.NOT_LIVING_ENTITY.create();
        }
    }

    private static int add(class_2168 source, class_1297 target, int value) throws CommandSyntaxException {
        if (target instanceof class_1309 entity) {
            entity.thermoo$addWetTicks(value);

            source.method_9226(
                    () -> class_2561.method_43469(
                            "commands.thermoo.soaking.add.success",
                            target.method_5476(),
                            value,
                            entity.thermoo$getWetTicks()
                    ), true
            );

            return entity.thermoo$getWetTicks();
        } else {
            throw TemperatureCommand.NOT_LIVING_ENTITY.create();
        }
    }

    private static int set(class_2168 source, class_1297 target, int value) throws CommandSyntaxException {
        if (target instanceof class_1309 entity) {
            entity.thermoo$setWetTicks(value);

            source.method_9226(
                    () -> class_2561.method_43469(
                            "commands.thermoo.soaking.set.success",
                            target.method_5476(),
                            value,
                            entity.thermoo$getWetTicks()
                    ), true
            );

            return entity.thermoo$getWetTicks();
        } else {
            throw TemperatureCommand.NOT_LIVING_ENTITY.create();
        }
    }

    private static int getMax(class_2168 source, class_1297 target) throws CommandSyntaxException {
        if (target instanceof class_1309 entity) {
            int value = entity.thermoo$getMaxWetTicks();

            source.method_9226(
                    () -> class_2561.method_43469(
                            "commands.thermoo.soaking.get.max.success",
                            target.method_5476(),
                            value
                    ), false
            );

            return value;
        } else {
            throw TemperatureCommand.NOT_LIVING_ENTITY.create();
        }
    }

    private static int getMin(class_2168 source, class_1297 target) throws CommandSyntaxException {
        if (target instanceof class_1309) {
            int value = 0;

            source.method_9226(
                    () -> class_2561.method_43469(
                            "commands.thermoo.soaking.get.min.success",
                            target.method_5476(),
                            value
                    ), false
            );

            return value;
        } else {
            throw TemperatureCommand.NOT_LIVING_ENTITY.create();
        }
    }

    private static int getScale(class_2168 source, class_1297 target, int scale) throws CommandSyntaxException {
        if (target instanceof class_1309 entity) {
            int value = class_3532.method_15375(entity.thermoo$getSoakedScale() * scale);

            source.method_9226(
                    () -> class_2561.method_43469(
                            "commands.thermoo.soaking.get.scale.success",
                            target.method_5476(),
                            value
                    ), false
            );

            return value;
        } else {
            throw TemperatureCommand.NOT_LIVING_ENTITY.create();
        }
    }

    private static int getCurrent(class_2168 source, class_1297 target) throws CommandSyntaxException {
        if (target instanceof class_1309 entity) {
            int value = entity.thermoo$getWetTicks();

            source.method_9226(
                    () -> class_2561.method_43469(
                            "commands.thermoo.soaking.get.current.success",
                            target.method_5476(),
                            value
                    ), false
            );

            return value;
        } else {
            throw TemperatureCommand.NOT_LIVING_ENTITY.create();
        }
    }

    private SoakingCommand() {

    }
}
