package com.github.thedeathlycow.thermoo.api.environment.provider;

import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.MapCodec;
import java.util.Collections;
import java.util.EnumMap;
import java.util.Map;
import java.util.Optional;
import net.minecraft.class_1937;
import net.minecraft.class_1959;
import net.minecraft.class_2338;
import net.minecraft.class_3542;
import net.minecraft.class_6880;
import net.minecraft.class_9323;

/**
 * An environment provider that dispatches to another provider based on the current season state of a world.
 */
public abstract sealed class SeasonalEnvironmentProvider<S extends Enum<S> & class_3542> implements EnvironmentProvider
        permits TemperateSeasonEnvironmentProvider, TropicalSeasonEnvironmentProvider {
    private final Optional<S> fallbackSeason;
    private final Map<S, class_6880<EnvironmentProvider>> seasons;

    protected SeasonalEnvironmentProvider(
            Optional<S> fallbackSeason,
            Map<S, class_6880<EnvironmentProvider>> seasons,
            Class<S> seasonClass
    ) {
        this.fallbackSeason = fallbackSeason;
        this.seasons = new EnumMap<>(seasonClass);
        this.seasons.putAll(seasons);
    }

    /**
     * Builds the environment components based on the world's current season state, generally using the
     * {@link com.github.thedeathlycow.thermoo.api.season.ThermooSeasonEvents season API}.
     * <p>
     * If no seasons mod is installed, or if the tropical/temperate season state does not exist at this world position,
     * then this will use the components provided by the {@link #fallbackSeason fallback season}.
     * <p>
     * If there is no fallback season, then this does nothing.
     *
     * @param level   The world/level being queried
     * @param pos     The position in the world to query
     * @param biome   The biome at the position in the world
     * @param builder Component map builder to append to
     */
    @Override
    public final void buildCurrentComponents(class_1937 level, class_2338 pos, class_6880<class_1959> biome, class_9323.class_9324 builder) {
        Optional<S> season = this.getCurrentSeason(level, pos).or(this::fallbackSeason);
        if (season.isPresent()) {
            class_6880<EnvironmentProvider> provider = this.seasons.get(season.get());
            if (provider != null) {
                provider.comp_349().buildCurrentComponents(level, pos, biome, builder);
            }
        }
    }

    /**
     * The fallback season to use if no season mod is installed. If specified, the fallback season must be a key in the
     * {@link #seasons()} map. If no fallback season is provided, and there is no season mod installed, then this
     * provider will return nothing.
     *
     * @return Returns {@link #fallbackSeason}
     */
    public final Optional<S> fallbackSeason() {
        return this.fallbackSeason;
    }

    /**
     * The season-to-provider lookup back. Used to dispatch this provider to another provider based on the current season
     * of a world.
     *
     * @return Returns an unmodifiable map of {@link #seasons}
     */
    public final Map<S, class_6880<EnvironmentProvider>> seasons() {
        return Collections.unmodifiableMap(this.seasons);
    }

    /**
     * Gets the current season state of the world at a position (usually by delegating to a
     * {@linkplain com.github.thedeathlycow.thermoo.api.season.ThermooSeasonEvents season event}.
     *
     * @param level The world to query the season state of
     * @param pos   The position to query the season state at
     * @return Returns the season state of a particular world position, or empty if no season state exists there or if a
     * season mod is not loaded.
     */
    protected abstract Optional<S> getCurrentSeason(class_1937 level, class_2338 pos);

    protected static <S extends Enum<S> & class_3542> MapCodec<Map<S, class_6880<EnvironmentProvider>>> createSeasonMapCodec(
            Codec<S> baseCodec,
            S[] values
    ) {
        return Codec.simpleMap(
                baseCodec,
                EnvironmentProvider.HOLDER_CODEC,
                class_3542.method_28142(values)
        ).validate(seasonMap -> {
            if (seasonMap.isEmpty()) {
                return DataResult.error(() -> "No season key in: " + seasonMap);
            } else {
                return DataResult.success(seasonMap);
            }
        });
    }

    protected static <S extends Enum<S> & class_3542, T extends SeasonalEnvironmentProvider<S>> MapCodec<T> validate(MapCodec<T> codec) {
        return codec
                .validate(
                        provider -> {
                            Optional<S> season = provider.fallbackSeason();
                            if (season.isEmpty()) {
                                return DataResult.success(provider);
                            } else if (!provider.seasons().containsKey(season.get())) {
                                return DataResult.error(() -> "Fallback season '" + season.get().method_15434() + "' is not a key in: " + provider.seasons());
                            } else {
                                return DataResult.success(provider);
                            }
                        }
                );
    }
}