package com.github.thedeathlycow.thermoo.api.temperature.effects;

import com.github.thedeathlycow.thermoo.api.ThermooRegistries;
import com.mojang.serialization.Codec;
import org.jetbrains.annotations.ApiStatus;

import java.util.Optional;
import net.minecraft.class_1299;
import net.minecraft.class_1309;
import net.minecraft.class_173;
import net.minecraft.class_181;
import net.minecraft.class_1937;
import net.minecraft.class_2096;
import net.minecraft.class_3218;
import net.minecraft.class_47;
import net.minecraft.class_5341;
import net.minecraft.class_6885;
import net.minecraft.class_8567;

/**
 * Represents a configured instance of a {@link TemperatureEffect} type.
 * See the <a href="https://github.com/TheDeathlyCow/frostiful/wiki/Temperature-Effects">wiki page</a> for details on
 * how to implement this in a datapack.
 * <p>
 * A configured temperature effect is more like an instance of a temperature effect, and this is the class that is
 * directly instantiated from a temperature effect JSON file in a datapack.
 *
 * @param <C> The config type
 * @see TemperatureEffect
 */
public final class ConfiguredTemperatureEffect<C> {

    /**
     * The temperature effect type
     */
    private final TemperatureEffect<C> type;

    /**
     * The config of the effect
     */
    private final C config;

    /**
     * If not null, then only applies the effect to entities for which this predicate is TRUE.
     */
    private final Optional<class_5341> predicate;

    /**
     * If not null, then only applies this effect to entities of the specific type. This is more
     * performant than using predicates if you want to apply an effect only to one specific type.
     */
    private final class_6885<class_1299<?>> entityTypes;

    /**
     * The temperature scale at which this should be applied to an entity. This is more
     * performant than using predicates if you want to apply an effect only within a particular
     * temperature range
     */
    private final class_2096.class_2099 temperatureScaleRange;

    /**
     * Priority for loading. Effects with a higher priority at the same resource location will
     * not be overridden by effects with lower priority from other mods/datapacks at the same
     * resource location. Allows mods to reliably override the temperature effects of other
     * mods, regardless of mod load order (which is arbitrary in Fabric). Defaults to 0 if not
     * specified.
     */
    private final int loadingPriority;

    @ApiStatus.Internal
    public ConfiguredTemperatureEffect(
            TemperatureEffect<C> type,
            C config,
            Optional<class_5341> predicate,
            class_6885<class_1299<?>> entityTypes,
            class_2096.class_2099 temperatureScaleRange,
            int loadingPriority
    ) {
        this.type = type;
        this.config = config;
        this.predicate = predicate;
        this.entityTypes = entityTypes;
        this.temperatureScaleRange = temperatureScaleRange;
        this.loadingPriority = loadingPriority;
    }

    /**
     * Codec for all configured temperature effects. Dispatches config codec based on
     * {@linkplain TemperatureEffect type}.
     */
    public static final Codec<ConfiguredTemperatureEffect<?>> CODEC = ThermooRegistries.TEMPERATURE_EFFECTS
            .method_39673()
            .dispatch(
                    "type",
                    ConfiguredTemperatureEffect::type,
                    TemperatureEffect::getCodec
            );

    /**
     * Tests and applies this effect to a living entity if possible
     *
     * @param victim The living entity to possibly apply the effect to
     * @deprecated Use {@link #apply(class_1309)}
     */
    @Deprecated
    public void applyIfPossible(class_1309 victim) {
        this.apply(victim);
    }

    /**
     * Tests and applies this effect to a living entity, and returns success if it was applied.
     * <p>
     * Returns false on client.
     *
     * @param victim The living entity to possibly apply the effect to
     * @return Returns {@code true} if the effect was applied.
     */
    public boolean apply(class_1309 victim) {
        class_1937 world = victim.method_73183();

        if (world.method_8608()) {
            return false;
        }

        class_3218 serverLevel = (class_3218) world;
        boolean shouldApply = this.type.shouldApply(victim, this.config)
                && this.temperatureScaleRange.method_9047(victim.thermoo$getTemperatureScale())
                && this.testPredicate(victim, serverLevel);

        if (shouldApply) {
            this.type.apply(victim, serverLevel, this.config);
            return true;
        }

        return false;
    }

    /**
     * Called the first tick that a configured temperature effect could not be applied
     *
     * @param victim The entity the effect was applied to
     */
    public void remove(class_1309 victim) {
        class_1937 level = victim.method_73183();

        if (level.method_8608()) {
            return;
        }

        class_3218 serverLevel = (class_3218) level;
        this.type.remove(victim, serverLevel, this.config);
    }

    private boolean testPredicate(class_1309 victim, class_3218 world) {
        return this.predicate.isEmpty()
                || this.predicate.get().test(
                new class_47.class_48(
                        new class_8567.class_8568(world)
                                .method_51874(class_181.field_1226, victim)
                                .method_51874(class_181.field_24424, victim.method_73189())
                                .method_51875(class_173.field_20761)
                ).method_309(Optional.empty())
        );
    }

    public TemperatureEffect<C> type() {
        return type;
    }

    public C config() {
        return config;
    }

    public Optional<class_5341> predicate() {
        return predicate;
    }

    public class_6885<class_1299<?>> entityTypes() {
        return entityTypes;
    }

    public class_2096.class_2099 temperatureScaleRange() {
        return temperatureScaleRange;
    }

    public int loadingPriority() {
        return loadingPriority;
    }
}
