package com.github.thedeathlycow.thermoo.api.util;

import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.class_1076;
import net.minecraft.class_310;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;

public final class TemperatureUnitLookup {
    /**
     * ISO 3166 alpha-2 region codes.
     * <p>
     * Sets the following locales to use the Fahrenheit system (shown in order they are assigned):
     * <p>
     * United States, US Minor Outlying Islands, US Virgin Islands, American Samoa, Puerto Rico, Guam, Northern Mariana
     * Islands, Liberia, The Bahamas, The Federated States of Micronesia, Marshall Islands, The Cayman Islands, Belize,
     * Antigua and Barbuda, Saint Kitts and Nevis.
     * <p>
     * This set could change in the future if these adopt the Celsius scale as their primarily unit of temperature.
     *
     * @see <a href="https://www.iana.org/assignments/language-subtag-registry/language-subtag-registry">Language subtags</a>
     */
    private static final Set<String> ISO_REGIONS_THAT_USE_FAHRENHEIT = new HashSet<>(
            Set.of("US", "UM", "VI", "AS", "PR", "GU", "MP", "LR", "BS", "FM", "MH", "KY", "BZ", "AG", "KN")
    );

    /**
     * Gets the currently selected temperature unit from the player's currently selected language. For languages based
     * on a language from the United States locale, this will be Fahrenheit. For all other locales, this is Celsius.
     * <p>
     * Currently, the languages that will return Fahrenheit are English (U.S.), Hawaiian, and LOLCAT.
     * <p>
     * This method is only available on the client.
     *
     * @return Returns Fahrenheit if the currently selected language is from a US-based locale.
     */
    @Environment(EnvType.CLIENT)
    public static TemperatureUnit fromCurrentLanguage() {
        class_1076 manager = class_310.method_1551().method_1526();
        return manager.method_4669().endsWith("_us")
                ? TemperatureUnit.FAHRENHEIT
                : TemperatureUnit.CELSIUS;
    }

    /**
     * Gets the temperature unit from the given language Minecraft language code. For languages based on a language from
     * the United States locale, this will be Fahrenheit. For all other locales, this is Celsius.
     * <p>
     * Currently, the languages that will return Fahrenheit are English (U.S.), Hawaiian, and LOLCAT.
     *
     * @return Returns Fahrenheit if the language code is from a US-based locale.
     */
    public static TemperatureUnit fromLanguageCode(String mcLanguageCode) {
        return mcLanguageCode.endsWith("_us")
                ? TemperatureUnit.FAHRENHEIT
                : TemperatureUnit.CELSIUS;
    }

    /**
     * Gets the temperature unit from the default locale. For the United States, Liberia, and a few Caribbean countries,
     * this is Fahrenheit. For all others it is Celsius.
     *
     * @return Returns the temperature unit associated with the default locale
     */
    public static TemperatureUnit fromLocale() {
        return fromLocale(Locale.getDefault());
    }

    /**
     * Gets the temperature unit from the given locale. For the United States, Liberia, and a few Caribbean countries,
     * this is Fahrenheit. For all others it is Celsius.
     *
     * @return Returns the temperature unit associated with the given locale
     */
    public static TemperatureUnit fromLocale(Locale locale) {
        return ISO_REGIONS_THAT_USE_FAHRENHEIT.contains(locale.getCountry().toUpperCase())
                ? TemperatureUnit.FAHRENHEIT
                : TemperatureUnit.CELSIUS;
    }

    private TemperatureUnitLookup() {

    }
}