package com.github.thedeathlycow.thermoo.api.command;

import com.github.thedeathlycow.thermoo.api.temperature.HeatingModes;
import com.github.thedeathlycow.thermoo.api.temperature.TemperatureAware;
import com.mojang.brigadier.arguments.IntegerArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.SimpleCommandExceptionType;
import org.jetbrains.annotations.Contract;

import java.util.Collection;
import java.util.function.Supplier;
import net.minecraft.class_1297;
import net.minecraft.class_1309;
import net.minecraft.class_2168;
import net.minecraft.class_2186;
import net.minecraft.class_2561;
import net.minecraft.class_3532;

import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;

/**
 * Command relating to temperature. Allows temperature to be modified in game.
 * <p>
 * Usage:
 * <p>
 * {@code thermoo temperature <subcommand> <args>}
 */
public class TemperatureCommand {

    static final SimpleCommandExceptionType NOT_LIVING_ENTITY = new SimpleCommandExceptionType(
            class_2561.method_43471("commands.thermoo.temperature.exception.not_living_entity")
    );

    /**
     * Supplier for creating a new temperature command builder to be registered to the Minecraft server
     * <p>
     * Registered by the default implementation of this API.
     */
    public static final Supplier<LiteralArgumentBuilder<class_2168>> COMMAND_BUILDER = TemperatureCommand::buildCommand;

    @Contract("->new")
    private static LiteralArgumentBuilder<class_2168> buildCommand() {
        var getSubCommand = method_9247("get")
                .then(
                        method_9244("target", class_2186.method_9309())
                                .executes(context -> {
                                    return runGetCurrent(
                                            context.getSource(),
                                            class_2186.method_9313(context, "target")
                                    );
                                })
                                .then(method_9247("current")
                                        .executes(context -> {
                                            return runGetCurrent(
                                                    context.getSource(),
                                                    class_2186.method_9313(context, "target")
                                            );
                                        })
                                )
                                .then(method_9247("max")
                                        .executes(context -> {
                                            return runGetMax(
                                                    context.getSource(),
                                                    class_2186.method_9313(context, "target")
                                            );
                                        })
                                )
                                .then(method_9247("min")
                                        .executes(context -> {
                                            return runGetMin(
                                                    context.getSource(),
                                                    class_2186.method_9313(context, "target")
                                            );
                                        })
                                )
                                .then(method_9247("scale")
                                        .executes(context -> {
                                            return runGetScale(
                                                    context.getSource(),
                                                    class_2186.method_9313(context, "target"),
                                                    100
                                            );
                                        })
                                        .then(method_9244("scale", IntegerArgumentType.integer(1))
                                                .executes(context -> {
                                                    return runGetScale(
                                                            context.getSource(),
                                                            class_2186.method_9313(context, "target"),
                                                            IntegerArgumentType.getInteger(context, "scale")
                                                    );
                                                })
                                        )
                                )

                );


        var remove = method_9247("remove")
                .then(
                        method_9244("targets", class_2186.method_9306())
                                .then(
                                        method_9244("amount", IntegerArgumentType.integer(0))
                                                .executes(
                                                        context -> {
                                                            return runAdjust(
                                                                    context.getSource(),
                                                                    class_2186.method_9317(context, "targets"),
                                                                    IntegerArgumentType.getInteger(context, "amount"),
                                                                    HeatingModes.ABSOLUTE,
                                                                    true
                                                            );
                                                        }
                                                )
                                                .then(
                                                        method_9244("mode", HeatingModeArgumentType.heatingMode())
                                                                .executes(context -> {
                                                                            return runAdjust(
                                                                                    context.getSource(),
                                                                                    class_2186.method_9317(context, "targets"),
                                                                                    IntegerArgumentType.getInteger(context, "amount"),
                                                                                    HeatingModeArgumentType.getHeatingMode(context, "mode"),
                                                                                    true
                                                                            );
                                                                        }
                                                                )
                                                )
                                )
                );

        var add = method_9247("add")
                .then(
                        method_9244("targets", class_2186.method_9306())
                                .then(
                                        method_9244("amount", IntegerArgumentType.integer(0))
                                                .executes(
                                                        context -> {
                                                            return runAdjust(
                                                                    context.getSource(),
                                                                    class_2186.method_9317(context, "targets"),
                                                                    IntegerArgumentType.getInteger(context, "amount"),
                                                                    HeatingModes.ABSOLUTE,
                                                                    false
                                                            );
                                                        }
                                                )
                                                .then(
                                                        method_9244("mode", HeatingModeArgumentType.heatingMode())
                                                                .executes(context -> {
                                                                            return runAdjust(
                                                                                    context.getSource(),
                                                                                    class_2186.method_9317(context, "targets"),
                                                                                    IntegerArgumentType.getInteger(context, "amount"),
                                                                                    HeatingModeArgumentType.getHeatingMode(context, "mode"),
                                                                                    false
                                                                            );
                                                                        }
                                                                )
                                                )
                                )
                );

        var setSubCommand = method_9247("set")
                .then(
                        method_9244("targets", class_2186.method_9306())
                                .then(
                                        method_9244("amount", IntegerArgumentType.integer())
                                                .executes(context -> {
                                                    return runSet(context.getSource(),
                                                            class_2186.method_9317(context, "targets"),
                                                            IntegerArgumentType.getInteger(context, "amount"));
                                                })
                                )
                );

        return method_9247("thermoo").then(
                (method_9247("temperature").requires((src) -> src.method_9259(2)))
                        .then(getSubCommand)
                        .then(remove)
                        .then(add)
                        .then(setSubCommand)
        );
    }

    private static int runGetScale(class_2168 source, class_1297 target, int scale) throws CommandSyntaxException {
        if (target instanceof class_1309 livingEntity) {
            float progress = livingEntity.thermoo$getTemperatureScale();
            int result = class_3532.method_15375(progress * scale);

            source.method_9226(
                    () -> class_2561.method_43469(
                            "commands.thermoo.temperature.get.scale.success",
                            target.method_5476(),
                            result
                    ), false
            );

            return result;
        } else {
            throw NOT_LIVING_ENTITY.create();
        }
    }

    private static int runGetMax(class_2168 source, class_1297 target) throws CommandSyntaxException {

        if (target instanceof class_1309 livingEntity) {
            int amount = livingEntity.thermoo$getMaxTemperature();
            source.method_9226(
                    () -> class_2561.method_43469("commands.thermoo.temperature.get.max.success", target.method_5476(), amount),
                    false
            );
            return amount;
        } else {
            throw NOT_LIVING_ENTITY.create();
        }


    }

    private static int runGetMin(class_2168 source, class_1297 target) throws CommandSyntaxException {
        if (target instanceof class_1309 livingEntity) {
            int amount = livingEntity.thermoo$getMinTemperature();
            source.method_9226(
                    () -> class_2561.method_43469(
                            "commands.thermoo.temperature.get.min.success",
                            target.method_5476(),
                            amount
                    ),
                    false
            );
            return amount;
        } else {
            throw NOT_LIVING_ENTITY.create();
        }
    }

    private static int runGetCurrent(class_2168 source, class_1297 target) throws CommandSyntaxException {
        if (target instanceof class_1309 livingEntity) {
            int amount = livingEntity.thermoo$getTemperature();
            source.method_9226(
                    () -> class_2561.method_43469(
                            "commands.thermoo.temperature.get.current.success",
                            target.method_5476(),
                            amount
                    ),
                    false
            );
            return amount;
        } else {
            throw NOT_LIVING_ENTITY.create();
        }
    }

    private static int runAdjust(class_2168 source, Collection<? extends class_1297> targets, int amount, HeatingModes mode, boolean isRemoving) throws CommandSyntaxException {
        amount = isRemoving ? -amount : amount;
        int sum = 0;
        for (class_1297 target : targets) {
            if (target instanceof TemperatureAware temperatureAware) {
                temperatureAware.thermoo$addTemperature(amount, mode);
                sum += amount;
            } else if (targets.size() == 1) {
                throw NOT_LIVING_ENTITY.create();
            }
        }


        class_2561 msg;
        if (isRemoving) {
            if (targets.size() == 1) {
                var target = targets.iterator().next();
                msg = class_2561.method_43469(
                        "commands.thermoo.temperature.remove.success.single",
                        amount,
                        target.method_5477(),
                        ((TemperatureAware) target).thermoo$getTemperature()
                );
            } else {
                msg = class_2561.method_43469(
                        "commands.thermoo.temperature.remove.success.multiple",
                        amount,
                        targets.size()
                );
            }
        } else {
            if (targets.size() == 1) {
                var target = targets.iterator().next();
                msg = class_2561.method_43469(
                        "commands.thermoo.temperature.add.success.single",
                        amount,
                        target.method_5477(),
                        ((TemperatureAware) target).thermoo$getTemperature()
                );
            } else {
                msg = class_2561.method_43469(
                        "commands.thermoo.temperature.add.success.multiple",
                        amount,
                        targets.size()
                );
            }
        }

        source.method_9226(() -> msg, true);
        return sum;
    }

    private static int runSet(class_2168 source, Collection<? extends class_1297> targets, int amount) throws CommandSyntaxException {

        int sum = 0;
        for (class_1297 target : targets) {
            if (target instanceof class_1309 livingEntity) {
                livingEntity.thermoo$setTemperature(amount);
                sum += amount;
            } else if (targets.size() == 1) {
                throw NOT_LIVING_ENTITY.create();
            }
        }

        class_2561 msg;
        if (targets.size() == 1) {
            msg = class_2561.method_43469(
                    "commands.thermoo.temperature.set.success.single",
                    targets.iterator().next().method_5477(),
                    amount
            );
        } else {
            msg = class_2561.method_43469(
                    "commands.thermoo.temperature.set.success.multiple",
                    targets.size(),
                    amount
            );
        }
        source.method_9226(() -> msg, true);

        return sum;
    }

}

