package com.github.thedeathlycow.thermoo.api.environment.provider;

import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import org.jetbrains.annotations.Contract;

import java.util.Collections;
import java.util.EnumMap;
import java.util.Map;
import java.util.Objects;
import net.minecraft.class_1937;
import net.minecraft.class_1959;
import net.minecraft.class_2338;
import net.minecraft.class_3542;
import net.minecraft.class_6880;
import net.minecraft.class_9323;

/**
 * A provider that delegates to a child provider based on the precipitation-type of a biome. At least one precipitation
 * type provider must be given.
 */
public final class BiomePrecipitationTypeEnvironmentProvider implements EnvironmentProvider {
    public static final MapCodec<BiomePrecipitationTypeEnvironmentProvider> CODEC = RecordCodecBuilder.mapCodec(
            instance -> instance.group(
                    createPrecipitationMapCodec()
                            .fieldOf("precipitation_type")
                            .forGetter(BiomePrecipitationTypeEnvironmentProvider::precipitationType)
            ).apply(instance, BiomePrecipitationTypeEnvironmentProvider::new)
    );

    private final Map<class_1959.class_1963, class_6880<EnvironmentProvider>> precipitationTypeMap;

    private BiomePrecipitationTypeEnvironmentProvider(Map<class_1959.class_1963, class_6880<EnvironmentProvider>> precipitationTypeMap) {
        this.precipitationTypeMap = new EnumMap<>(class_1959.class_1963.class);
        this.precipitationTypeMap.putAll(precipitationTypeMap);
    }

    /**
     * Delegates to a child provider based on the local precipitation type.
     * <p>
     * <strong>IMPORTANT:</strong> This is not based on current weather state. For example, snowy biomes will ALWAYS
     * return the provider mapped to {@link class_1959.class_1963#field_9383}, even when it is not snowing.
     * <p>
     * If no provider is mapped to the local precipitation type, then nothing is built.
     *
     * @param world   The world/level being queried
     * @param pos     The position in the world to query
     * @param biome   The biome at the position in the world
     * @param builder A component map builder to append to
     */
    @Override
    public void buildCurrentComponents(class_1937 world, class_2338 pos, class_6880<class_1959> biome, class_9323.class_9324 builder) {
        class_1959.class_1963 biomePrecipitationType = biome.comp_349().method_48162(pos);
        class_6880<EnvironmentProvider> provider = this.precipitationTypeMap.get(biomePrecipitationType);
        if (provider != null) {
            provider.comp_349().buildCurrentComponents(world, pos, biome, builder);
        }
    }

    @Override
    public EnvironmentProviderType<BiomePrecipitationTypeEnvironmentProvider> getType() {
        return EnvironmentProviderTypes.PRECIPITATION_TYPE;
    }

    /**
     * Maps that is used to choose the child provider to delegate to based on local precipitation
     *
     * @return Returns an unmodifiable enum map
     */
    public Map<class_1959.class_1963, class_6880<EnvironmentProvider>> precipitationType() {
        return Collections.unmodifiableMap(precipitationTypeMap);
    }

    private static MapCodec<Map<class_1959.class_1963, class_6880<EnvironmentProvider>>> createPrecipitationMapCodec() {
        return Codec.simpleMap(
                class_1959.class_1963.field_46251,
                EnvironmentProvider.HOLDER_CODEC,
                class_3542.method_28142(class_1959.class_1963.values())
        ).validate(map -> {
            if (map.isEmpty()) {
                return DataResult.error(() -> "No precipitation key in: " + map);
            } else {
                return DataResult.success(map);
            }
        });
    }

    /**
     * A builder for local precipitation environment providers.
     */
    public static final class Builder {
        private final Map<class_1959.class_1963, class_6880<EnvironmentProvider>> precipitationMap = new EnumMap<>(class_1959.class_1963.class);

        private Builder() {

        }

        /**
         * Registers a child provider to a precipitation type.
         *
         * @param precipitation Precipitation type to add
         * @param child         The child for the precipitation type
         * @return Returns this builder
         */
        @Contract("_,_->this")
        public Builder addChild(class_1959.class_1963 precipitation, class_6880<EnvironmentProvider> child) {
            Objects.requireNonNull(precipitation);
            Objects.requireNonNull(child);

            this.precipitationMap.put(precipitation, child);

            return this;
        }

        /**
         * Builds into a new provider. At least one precipitation-child relationship must have been defined.
         *
         * @return Returns a new provider from this builder's state
         */
        @Contract("->new")
        public BiomePrecipitationTypeEnvironmentProvider build() {
            if (this.precipitationMap.keySet().isEmpty()) {
                throw new IllegalArgumentException("Precipitation map requires at least one key!");
            }

            return new BiomePrecipitationTypeEnvironmentProvider(this.precipitationMap);
        }
    }
}