package com.github.thedeathlycow.thermoo.api.command;

import com.github.thedeathlycow.thermoo.api.environment.EnvironmentLookup;
import com.github.thedeathlycow.thermoo.api.environment.component.EnvironmentComponentTypes;
import com.github.thedeathlycow.thermoo.api.environment.component.RelativeHumidityComponent;
import com.github.thedeathlycow.thermoo.api.environment.component.TemperatureRecordComponent;
import com.github.thedeathlycow.thermoo.api.environment.event.ServerPlayerEnvironmentTickEvents;
import com.github.thedeathlycow.thermoo.api.util.TemperatureUnit;
import com.github.thedeathlycow.thermoo.impl.LivingEntityTickUtil;
import com.github.thedeathlycow.thermoo.impl.environment.EnvironmentTickContextImpl;
import com.mojang.brigadier.arguments.DoubleArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import org.jetbrains.annotations.Contract;

import java.util.function.Supplier;
import net.minecraft.class_1959;
import net.minecraft.class_2168;
import net.minecraft.class_2186;
import net.minecraft.class_2262;
import net.minecraft.class_2338;
import net.minecraft.class_2561;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_5321;

import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;

/**
 * Command relating to environment effects
 * <p>
 * Usage:
 * <p>
 * {@code thermoo environment checktemperature <args>}
 * <p>
 * {@code thermoo environment printcontroller}
 * <p>
 * {@code thermoo environment temperature <pos> [<unit>] [<scale>]}
 * <p>
 * {@code thermoo environment relativehumidity <pos> [<scale>]}
 */
public final class EnvironmentCommand {
    private EnvironmentCommand() {

    }

    /**
     * Supplier for creating a new environment command builder to be registered to the Minecraft server
     * <p>
     * Registered by the default implementation of this API.
     */
    public static final Supplier<LiteralArgumentBuilder<class_2168>> COMMAND_BUILDER = EnvironmentCommand::buildCommand;

    @Contract("->new")
    private static LiteralArgumentBuilder<class_2168> buildCommand() {
        final String location = "location";
        final String target = "target";
        final String unit = "unit";
        final String scale = "scale";
        final TemperatureUnit fallbackUnit = TemperatureUnit.CELSIUS;
        final double fallbackTempScale = 1.0;

        var temperature = method_9247("temperature")
                .then(method_9244(target, class_2186.method_9305())
                        .executes(
                                context -> executeEntityTemperature(
                                        context.getSource(),
                                        class_2186.method_9315(context, target)
                                )
                        )
                )
                .then(method_9244(location, class_2262.method_9698())
                        .executes(
                                context -> executeTemperature(
                                        context.getSource(),
                                        class_2262.method_9696(context, location),
                                        TemperatureUnit.CELSIUS,
                                        fallbackTempScale
                                )
                        )
                        .then(
                                method_9244(unit, TemperatureUnitArgumentType.temperatureUnit())
                                        .executes(
                                                context -> executeTemperature(
                                                        context.getSource(),
                                                        class_2262.method_9696(context, location),
                                                        TemperatureUnitArgumentType.getTemperatureUnit(context, unit),
                                                        fallbackTempScale
                                                )
                                        )
                                        .then(
                                                method_9244(scale, DoubleArgumentType.doubleArg(0))
                                                        .executes(
                                                                context -> executeTemperature(
                                                                        context.getSource(),
                                                                        class_2262.method_9696(context, location),
                                                                        TemperatureUnitArgumentType.getTemperatureUnit(context, unit),
                                                                        DoubleArgumentType.getDouble(context, scale)
                                                                )
                                                        )
                                        )
                        )
                );

        final double fallbackHumidityScale = 100.0;

        var relativeHumidity = method_9247("relativehumidity").then(
                method_9244(location, class_2262.method_9698())
                        .executes(
                                context -> executeRelativeHumidity(
                                        context.getSource(),
                                        class_2262.method_9696(context, location),
                                        fallbackHumidityScale
                                )
                        )
                        .then(
                                method_9244(scale, DoubleArgumentType.doubleArg(0))
                                        .executes(
                                                context -> executeRelativeHumidity(
                                                        context.getSource(),
                                                        class_2262.method_9696(context, location),
                                                        DoubleArgumentType.getDouble(context, scale)
                                                )
                                        )
                        )
        );

        return method_9247("thermoo").then(
                (method_9247("environment").requires((src) -> src.method_9259(2)))
                        .then(temperature)
                        .then(relativeHumidity)
        );
    }

    private static int executeEntityTemperature(class_2168 source, class_3222 target) {
        class_2338 pos = LivingEntityTickUtil.getTemperatureTickPos(target);
        class_3218 world = target.method_51469();
        final EnvironmentTickContextImpl<class_3222> context = new EnvironmentTickContextImpl<>(
                target,
                world,
                pos,
                EnvironmentLookup.getInstance().findEnvironmentComponents(world, pos)
        );

        int tempChange = ServerPlayerEnvironmentTickEvents.GET_TEMPERATURE_CHANGE.invoker().addPointChange(context);
        double resistance = tempChange != 0
                ? tempChange > 0 ? target.thermoo$getEnvironmentHeatResistance() : target.thermoo$getEnvironmentColdResistance()
                : 0.0;

        if (resistance >= 0) {
            source.method_9226(
                    () -> class_2561.method_43469(
                            "commands.thermoo.environment.temperature.player.success",
                            target.method_5476(),
                            tempChange,
                            "%.2f%%".formatted(resistance * 100)
                    ),
                    false
            );
        } else {
            source.method_9226(
                    () -> class_2561.method_43469(
                            "commands.thermoo.environment.temperature.player.negative.success",
                            target.method_5476(),
                            tempChange,
                            "%.2f%%".formatted(resistance * -100)
                    ),
                    false
            );

        }

        return tempChange;
    }

    private static int executeTemperature(class_2168 source, class_2338 location, TemperatureUnit unit, double scale) {
        double temperature = EnvironmentLookup.getInstance().findEnvironmentComponents(
                        source.method_9225(), location
                ).method_58695(EnvironmentComponentTypes.TEMPERATURE, TemperatureRecordComponent.DEFAULT)
                .valueInUnit(unit);

        source.method_9226(
                () -> {
                    class_5321<class_1959> biome = source.method_9225().method_23753(location).method_40230().orElse(null);
                    return class_2561.method_43469(
                            "commands.thermoo.environment.temperature.success",
                            location.method_10263(),
                            location.method_10264(),
                            location.method_10260(),
                            biome == null ? "unknown" : biome.method_29177().toString(),
                            String.format("%.2f", temperature),
                            unit.getUnitSymbol()
                    );
                },
                false
        );

        return (int) (temperature * scale);
    }

    private static int executeRelativeHumidity(class_2168 source, class_2338 location, double scale) {
        double relativeHumidity = EnvironmentLookup.getInstance().findEnvironmentComponents(
                source.method_9225(), location
        ).method_58695(EnvironmentComponentTypes.RELATIVE_HUMIDITY, RelativeHumidityComponent.DEFAULT);
        double scaledHumidity = relativeHumidity * scale;

        source.method_9226(
                () -> {
                    class_5321<class_1959> biome = source.method_9225().method_23753(location).method_40230().orElse(null);
                    return class_2561.method_43469(
                            "commands.thermoo.environment.humidity.success",
                            location.method_10263(),
                            location.method_10264(),
                            location.method_10260(),
                            biome == null ? "unknown" : biome.method_29177().toString(),
                            String.format("%.2f", scaledHumidity)
                    );
                },
                false
        );

        return (int) (scaledHumidity);
    }
}
