package com.github.thedeathlycow.thermoo.api.command;

import com.github.thedeathlycow.thermoo.api.environment.EnvironmentLookup;
import com.github.thedeathlycow.thermoo.api.environment.component.EnvironmentComponentTypes;
import com.github.thedeathlycow.thermoo.api.environment.component.RelativeHumidityComponent;
import com.github.thedeathlycow.thermoo.api.environment.component.TemperatureRecordComponent;
import com.github.thedeathlycow.thermoo.api.environment.event.ServerPlayerEnvironmentTickEvents;
import com.github.thedeathlycow.thermoo.api.temperature.EnvironmentManager;
import com.github.thedeathlycow.thermoo.api.util.TemperatureConverter;
import com.github.thedeathlycow.thermoo.api.util.TemperatureUnit;
import com.github.thedeathlycow.thermoo.impl.LivingEntityTickUtil;
import com.github.thedeathlycow.thermoo.impl.Thermoo;
import com.github.thedeathlycow.thermoo.impl.environment.EnvironmentTickContextImpl;
import com.mojang.brigadier.arguments.DoubleArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import org.jetbrains.annotations.Contract;

import java.util.function.Supplier;
import net.minecraft.class_124;
import net.minecraft.class_1959;
import net.minecraft.class_2168;
import net.minecraft.class_2186;
import net.minecraft.class_2262;
import net.minecraft.class_2338;
import net.minecraft.class_243;
import net.minecraft.class_2561;
import net.minecraft.class_3222;
import net.minecraft.class_5321;

import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;

/**
 * Command relating to environment effects
 * <p>
 * Usage:
 * <p>
 * {@code thermoo environment checktemperature <args>}
 * <p>
 * {@code thermoo environment printcontroller}
 * <p>
 * {@code thermoo environment temperature <pos> [<unit>] [<scale>]}
 * <p>
 * {@code thermoo environment relativehumidity <pos> [<scale>]}
 */
public class EnvironmentCommand {

    /**
     * Supplier for creating a new environment command builder to be registered to the Minecraft server
     * <p>
     * Registered by the default implementation of this API.
     */
    public static final Supplier<LiteralArgumentBuilder<class_2168>> COMMAND_BUILDER = EnvironmentCommand::buildCommand;

    @Contract("->new")
    private static LiteralArgumentBuilder<class_2168> buildCommand() {

        var printController = method_9247("printcontroller")
                .executes(
                        context -> {
                            return printController(context.getSource());
                        }
                );

        var checkTemperature = method_9247("checktemperature")
                .executes(
                        context -> {
                            var pos = context.getSource().method_9222();
                            return executeCheckTemperature(
                                    context.getSource(),
                                    new class_2338((int) pos.field_1352, (int) pos.field_1351, (int) pos.field_1350)
                            );
                        }
                )
                .then(
                        method_9244("target", class_2186.method_9309())
                                .executes(
                                        context -> {
                                            return executeCheckTemperature(
                                                    context.getSource(),
                                                    class_2186.method_9313(
                                                            context,
                                                            "target"
                                                    ).method_24515()
                                            );
                                        }
                                )
                                .then(
                                        method_9244("unit", TemperatureUnitArgumentType.temperatureUnit())
                                                .executes(
                                                        context -> {
                                                            return executeCheckTemperature(
                                                                    context.getSource(),
                                                                    class_2186.method_9313(
                                                                            context,
                                                                            "target"
                                                                    ).method_24515(),
                                                                    TemperatureUnitArgumentType.getTemperatureUnit(
                                                                            context,
                                                                            "unit"
                                                                    )
                                                            );
                                                        }
                                                )
                                )
                )
                .then(
                        method_9244("location", class_2262.method_9698())
                                .executes(
                                        context -> {
                                            return executeCheckTemperature(
                                                    context.getSource(),
                                                    class_2262.method_9696(
                                                            context,
                                                            "location"
                                                    )
                                            );
                                        }
                                )
                                .then(
                                        method_9244("unit", TemperatureUnitArgumentType.temperatureUnit())
                                                .executes(
                                                        context -> {
                                                            return executeCheckTemperature(
                                                                    context.getSource(),
                                                                    class_2262.method_9696(
                                                                            context,
                                                                            "location"
                                                                    ),
                                                                    TemperatureUnitArgumentType.getTemperatureUnit(
                                                                            context,
                                                                            "unit"
                                                                    )
                                                            );
                                                        }
                                                )
                                )
                );

        final String location = "location";
        final String target = "target";
        final String unit = "unit";
        final String scale = "scale";
        final TemperatureUnit fallbackUnit = TemperatureUnit.CELSIUS;
        final double fallbackTempScale = 1.0;

        var temperature = method_9247("temperature")
                .then(method_9244(target, class_2186.method_9305())
                        .executes(
                                context -> executeEntityTemperature(
                                        context.getSource(),
                                        class_2186.method_9315(context, target)
                                )
                        )
                )
                .then(method_9244(location, class_2262.method_9698())
                        .executes(
                                context -> executeTemperature(
                                        context.getSource(),
                                        class_2262.method_9696(context, location),
                                        TemperatureUnit.CELSIUS,
                                        fallbackTempScale
                                )
                        )
                        .then(
                                method_9244(unit, TemperatureUnitArgumentType.temperatureUnit())
                                        .executes(
                                                context -> executeTemperature(
                                                        context.getSource(),
                                                        class_2262.method_9696(context, location),
                                                        TemperatureUnitArgumentType.getTemperatureUnit(context, unit),
                                                        fallbackTempScale
                                                )
                                        )
                                        .then(
                                                method_9244(scale, DoubleArgumentType.doubleArg(0))
                                                        .executes(
                                                                context -> executeTemperature(
                                                                        context.getSource(),
                                                                        class_2262.method_9696(context, location),
                                                                        TemperatureUnitArgumentType.getTemperatureUnit(context, unit),
                                                                        DoubleArgumentType.getDouble(context, scale)
                                                                )
                                                        )
                                        )
                        )
                );

        final double fallbackHumidityScale = 100.0;

        var relativeHumidity = method_9247("relativehumidity").then(
                method_9244(location, class_2262.method_9698())
                        .executes(
                                context -> executeRelativeHumidity(
                                        context.getSource(),
                                        class_2262.method_9696(context, location),
                                        fallbackHumidityScale
                                )
                        )
                        .then(
                                method_9244(scale, DoubleArgumentType.doubleArg(0))
                                        .executes(
                                                context -> executeRelativeHumidity(
                                                        context.getSource(),
                                                        class_2262.method_9696(context, location),
                                                        DoubleArgumentType.getDouble(context, scale)
                                                )
                                        )
                        )
        );

        return method_9247("thermoo").then(
                (method_9247("environment").requires((src) -> src.method_9259(2)))
                        .then(checkTemperature)
                        .then(printController)
                        .then(temperature)
                        .then(relativeHumidity)
        );
    }

    @Deprecated
    private static int printController(class_2168 source) {
        String controller = EnvironmentManager.INSTANCE.getController().toString();

        source.method_9226(() -> class_2561.method_48321(
                "commands.thermoo.environment.printcontroller.success",
                "Controller logged to console"
        ), false);
        source.method_9226(
                () -> class_2561.method_48321(
                        "commands.thermoo.environment.printcontroller.deprecation",
                        "This command is deprecated, the Environment Controller has been replaced with the Environment Datapack Registry."
                ).method_27692(class_124.field_1061),
                false
        );

        Thermoo.LOGGER.info("The current controller is: {}", controller);
        return 0;
    }

    private static int executeEntityTemperature(class_2168 source, class_3222 target) {
        class_2338 pos = LivingEntityTickUtil.getTemperatureTickPos(target);
        final EnvironmentTickContextImpl<class_3222> context = new EnvironmentTickContextImpl<>(
                target,
                target.method_51469(),
                pos,
                EnvironmentLookup.getInstance().findEnvironmentComponents(target.method_51469(), pos)
        );

        int tempChange = ServerPlayerEnvironmentTickEvents.GET_TEMPERATURE_CHANGE.invoker().addPointChange(context);
        double resistance = tempChange != 0
                ? tempChange > 0 ? target.thermoo$getEnvironmentHeatResistance() : target.thermoo$getEnvironmentColdResistance()
                : 0.0;

        if (resistance >= 0) {
            source.method_9226(
                    () -> class_2561.method_48322(
                            "commands.thermoo.environment.temperature.player.success",
                            "The environment temperature change of %s is %s (with a %s chance to dodge)",
                            target.method_5476(),
                            tempChange,
                            "%.2f%%".formatted(resistance * 100)
                    ),
                    false
            );
        } else {
            source.method_9226(
                    () -> class_2561.method_48322(
                            "commands.thermoo.environment.temperature.player.negative.success",
                            "The environment temperature change of %s is %s (with a %s chance of doubling)",
                            target.method_5476(),
                            tempChange,
                            "%.2f%%".formatted(resistance * -100)
                    ),
                    false
            );

        }

        return tempChange;
    }

    private static int executeTemperature(class_2168 source, class_2338 location, TemperatureUnit unit, double scale) {
        double temperature = EnvironmentLookup.getInstance().findEnvironmentComponents(
                        source.method_9225(), location
                ).method_57830(EnvironmentComponentTypes.TEMPERATURE, TemperatureRecordComponent.DEFAULT)
                .valueInUnit(unit);

        source.method_9226(
                () -> {
                    class_5321<class_1959> biome = source.method_9225().method_23753(location).method_40230().orElse(null);
                    return class_2561.method_48322(
                            "commands.thermoo.environment.temperature.success",
                            "The environment temperature at %s, %s, %s (%s) is %s°%s",
                            location.method_10263(),
                            location.method_10264(),
                            location.method_10260(),
                            biome == null ? "unknown" : biome.method_29177().toString(),
                            String.format("%.2f", temperature),
                            unit.getUnitSymbol()
                    );
                },
                false
        );

        return (int) (temperature * scale);
    }

    private static int executeRelativeHumidity(class_2168 source, class_2338 location, double scale) {
        double relativeHumidity = EnvironmentLookup.getInstance().findEnvironmentComponents(
                source.method_9225(), location
        ).method_57830(EnvironmentComponentTypes.RELATIVE_HUMIDITY, RelativeHumidityComponent.DEFAULT);
        double scaledHumidity = relativeHumidity * scale;

        source.method_9226(
                () -> {
                    class_5321<class_1959> biome = source.method_9225().method_23753(location).method_40230().orElse(null);
                    return class_2561.method_48322(
                            "commands.thermoo.environment.humidity.success",
                            "The environmental relative humidity at %s, %s, %s (%s) is %s%",
                            location.method_10263(),
                            location.method_10264(),
                            location.method_10260(),
                            biome == null ? "unknown" : biome.method_29177().toString(),
                            String.format("%.2f", scaledHumidity)
                    );
                },
                false
        );

        return (int) (scaledHumidity);
    }

    @Deprecated
    private static int executeCheckTemperature(class_2168 source, class_2338 location) {

        int temperatureChange = EnvironmentManager.INSTANCE.getController().getLocalTemperatureChange(
                source.method_9225(),
                location
        );


        var biome = source.method_9225().method_23753(location).method_40230().orElse(null);

        source.method_9226(
                () -> class_2561.method_48322(
                        "commands.thermoo.environment.checktemperature.success",
                        "The passive temperature change at %s, %s, %s (%s) is %s",
                        location.method_10263(),
                        location.method_10264(),
                        location.method_10260(),
                        biome == null ? "unknown" : biome.method_29177().toString(),
                        temperatureChange
                ),
                false
        );
        source.method_9226(
                () -> class_2561.method_48321(
                        "commands.thermoo.environment.checktemperature.deprecation",
                        "This command is deprecated, use /thermoo environment temperature <pos>"
                ).method_27692(class_124.field_1061),
                false
        );

        return temperatureChange;
    }

    @Deprecated
    private static int executeCheckTemperature(class_2168 source, class_2338 location, TemperatureUnit unit) {

        int temperatureTick = EnvironmentManager.INSTANCE.getController().getLocalTemperatureChange(
                source.method_9225(),
                location
        );


        var biome = source.method_9225().method_23753(location).method_40230().orElse(null);

        double temperature = TemperatureConverter.temperatureTickToAmbientTemperature(
                temperatureTick,
                new TemperatureConverter.Settings(unit, 1, 0)
        );

        source.method_9226(
                () -> class_2561.method_48322(
                        "commands.thermoo.environment.checktemperature.unit.success",
                        "The temperature at %s, %s, %s (%s) is %s°%s",
                        location.method_10263(),
                        location.method_10264(),
                        location.method_10260(),
                        biome == null ? "unknown" : biome.method_29177().toString(),
                        String.format("%.2f", temperature),
                        unit.getUnitSymbol()
                ),
                false
        );
        source.method_9226(
                () -> class_2561.method_48321(
                        "commands.thermoo.environment.checktemperature.deprecation",
                        "This command is deprecated, use /thermoo environment temperature <pos>"
                ).method_27692(class_124.field_1061),
                false
        );

        return (int) temperature;
    }
}
