package com.github.thedeathlycow.thermoo.api.environment;

import com.github.thedeathlycow.thermoo.api.environment.provider.EnvironmentProvider;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import net.minecraft.class_1959;
import net.minecraft.class_6880;
import net.minecraft.class_6885;
import net.minecraft.class_6895;
import net.minecraft.class_7924;
import org.jetbrains.annotations.Contract;

/**
 * Defines a biome's environmental temperature and relative humidity values. Must be defined in a datapack registry
 * in order to work.
 */
public final class EnvironmentDefinition {
    private static final int DEFAULT_PRIORITY = 1000;

    public static final Codec<EnvironmentDefinition> CODEC = RecordCodecBuilder.create(
            instance -> instance.group(
                    class_6895.method_40340(class_7924.field_41236)
                            .fieldOf("biomes")
                            .forGetter(EnvironmentDefinition::biomes),
                    class_6895.method_40340(class_7924.field_41236)
                            .optionalFieldOf("exclude_biomes", class_6885.method_58563())
                            .forGetter(EnvironmentDefinition::excludeBiomes),
                    EnvironmentProvider.ENTRY_CODEC
                            .fieldOf("provider")
                            .forGetter(EnvironmentDefinition::provider),
                    Codec.INT
                            .optionalFieldOf("priority", DEFAULT_PRIORITY)
                            .forGetter(EnvironmentDefinition::priority)
            ).apply(instance, EnvironmentDefinition::new)
    );

    private final class_6885<class_1959> biomes;

    private final class_6885<class_1959> excludeBiomes;

    private final class_6880<EnvironmentProvider> provider;

    private final int priority;

    private EnvironmentDefinition(
            class_6885<class_1959> biomes,
            class_6885<class_1959> excludeBiomes,
            class_6880<EnvironmentProvider> provider,
            int priority
    ) {
        this.biomes = biomes;
        this.excludeBiomes = excludeBiomes;
        this.provider = provider;
        this.priority = priority;
    }

    public static EnvironmentDefinition.Builder builder(
            class_6885<class_1959> biomes,
            class_6880<EnvironmentProvider> provider
    ) {
        return new Builder(biomes, provider);
    }

    /**
     * Creates an environment definition
     *
     * @param biomes   The biomes this definition provides for
     * @param provider The base value provider of this definition
     * @return Returns a new definition
     * @deprecated Use {@link #builder(class_6885, class_6880)}
     */
    @Contract("_,_->new")
    @Deprecated(since = "4.5")
    public static EnvironmentDefinition create(class_6885<class_1959> biomes, class_6880<EnvironmentProvider> provider) {
        return builder(biomes, provider).build();
    }

    /**
     * Creates an environment definition
     *
     * @param biomes        The biomes this definition provides for
     * @param excludeBiomes The biomes this definition has been blocked from providing for
     * @param provider      The base value provider of this definition
     * @return Returns a new definition
     * @deprecated Use {@link #builder(class_6885, class_6880)}
     */
    @Contract("_,_,_->new")
    @Deprecated(since = "4.5")
    public static EnvironmentDefinition create(
            class_6885<class_1959> biomes,
            class_6885<class_1959> excludeBiomes,
            class_6880<EnvironmentProvider> provider
    ) {
        return builder(biomes, provider).excludeBiomes(excludeBiomes).build();
    }

    /**
     * Checks that this definition can provide an environment for the given biome
     *
     * @param biome The biome to check
     * @return Returns {@code true} if the biome is in this definition's {@linkplain #biomes() biome list}, and NOT in
     * this definition's {@linkplain #excludeBiomes() excluded biome list}.
     */
    public boolean providesFor(class_6880<class_1959> biome) {
        return this.biomes().method_40241(biome) && !this.excludeBiomes().method_40241(biome);
    }

    /**
     * The biomes that this environment provides for
     *
     * @return The biomes that this definition provides an environment for
     */
    public class_6885<class_1959> biomes() {
        return this.biomes;
    }

    /**
     * The biomes that the environment has been blocked from providing for
     *
     * @return The biomes that this definition provides excludes
     */
    public class_6885<class_1959> excludeBiomes() {
        return this.excludeBiomes;
    }

    /**
     * @return The environment provider for this definition
     */
    public class_6880<EnvironmentProvider> provider() {
        return this.provider;
    }

    /**
     * Determines the priority for which this environment should be applied to a biome. Environments wither a HIGHER
     * priority will be applied FIRST, and environments with a LOWER priority will be applied LAST. Environments with the
     * same priority may be applied in any order.
     * <p>
     * The default priority is {@value DEFAULT_PRIORITY}.
     *
     * @return Returns this environment's priority.
     */
    public int priority() {
        return this.priority;
    }

    public static class Builder {
        private final class_6885<class_1959> biomes;
        private final class_6880<EnvironmentProvider> provider;
        private class_6885<class_1959> excludeBiomes = class_6885.method_58563();
        private int priority = DEFAULT_PRIORITY;

        private Builder(class_6885<class_1959> biomes, class_6880<EnvironmentProvider> provider) {
            this.biomes = biomes;
            this.provider = provider;
        }

        public Builder excludeBiomes(class_6885<class_1959> biomes) {
            this.excludeBiomes = biomes;
            return this;
        }

        public Builder withPriority(int priority) {
            this.priority = priority;
            return this;
        }

        public EnvironmentDefinition build() {
            return new EnvironmentDefinition(
                    this.biomes,
                    this.excludeBiomes,
                    this.provider,
                    this.priority
            );
        }
    }
}