/*
 * Decompiled with CFR 0.152.
 */
package com.frikinjay.villagerapi.villagerpack;

import com.frikinjay.villagerapi.VillagerAPI;
import com.frikinjay.villagerapi.villagerpack.VillagerPackConfig;
import com.frikinjay.villagerapi.villagerpack.VillagerPackLoader;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

public class VillagerPackValidator {
    public static ValidationResult validatePack(Path packPath) {
        ArrayList<String> errors = new ArrayList<String>();
        ArrayList<String> warnings = new ArrayList<String>();
        HashMap<String, Object> info = new HashMap<String, Object>();
        boolean criticalFailure = false;
        try {
            VillagerPackLoader.VillagerPack pack = new VillagerPackLoader.VillagerPack(packPath);
            if (!pack.hasConfig()) {
                errors.add("CRITICAL: No villagerapi_config.json found - this is not a valid villagerpack");
                pack.close();
                return new ValidationResult(false, true, errors, warnings, info);
            }
            criticalFailure = !VillagerPackValidator.validatePackStructure(pack, errors, warnings, info);
            VillagerPackValidator.validatePackMeta(pack, errors, warnings, info);
            VillagerPackValidator.validateVillagerData(pack, errors, warnings, info);
            VillagerPackValidator.validateDatapacks(pack, errors, warnings, info);
            VillagerPackValidator.validateResourcePacks(pack, errors, warnings, info);
            pack.close();
            boolean hasErrors = !errors.isEmpty();
            return new ValidationResult(!hasErrors, criticalFailure, errors, warnings, info);
        }
        catch (Exception e) {
            errors.add("CRITICAL: Failed to load pack: " + e.getMessage());
            return new ValidationResult(false, true, errors, warnings, info);
        }
    }

    private static boolean validatePackStructure(VillagerPackLoader.VillagerPack pack, List<String> errors, List<String> warnings, Map<String, Object> info) {
        Path root = pack.getRootPath();
        boolean hasVillagers = Files.exists(root.resolve("villagers"), new LinkOption[0]);
        boolean hasData = Files.exists(root.resolve("data"), new LinkOption[0]);
        boolean hasAssets = Files.exists(root.resolve("assets"), new LinkOption[0]);
        info.put("Has villagers folder", hasVillagers);
        info.put("Has data folder", hasData);
        info.put("Has assets folder", hasAssets);
        if (!hasVillagers && !hasData) {
            errors.add("CRITICAL: Pack has no villagers/ or data/ folders - nothing to load");
            return false;
        }
        if (!hasVillagers) {
            warnings.add("No villagers/ folder found - no custom villager content will be loaded");
        }
        if (!hasAssets) {
            warnings.add("No assets/ folder found - no custom textures/models will be loaded");
        }
        return true;
    }

    private static void validatePackMeta(VillagerPackLoader.VillagerPack pack, List<String> errors, List<String> warnings, Map<String, Object> info) {
        JsonObject meta;
        VillagerPackConfig config = pack.getConfig();
        info.put("Namespace", config.getNamespace());
        info.put("Display Name", config.getDisplayName());
        info.put("Version", config.getVersion());
        info.put("Author", config.getAuthor());
        if (!config.getDescription().isEmpty()) {
            info.put("Description", config.getDescription());
        }
        if ((meta = pack.getPackMeta()) == null) {
            warnings.add("No pack.mcmeta found - datapacks and resource packs may not load properly");
            return;
        }
        if (meta.has("pack")) {
            JsonObject packInfo = meta.getAsJsonObject("pack");
            if (packInfo.has("pack_format")) {
                int format = packInfo.get("pack_format").getAsInt();
                info.put("Pack format", format);
                if (format < 34) {
                    warnings.add("Pack format " + format + " is outdated for MC 1.21.1 (expected 34+ for resources, 48+ for data)");
                }
            } else {
                warnings.add("pack.mcmeta missing pack_format - may cause datapack/resource pack issues");
            }
        } else {
            warnings.add("pack.mcmeta missing 'pack' section");
        }
    }

    private static void validateVillagerData(VillagerPackLoader.VillagerPack pack, List<String> errors, List<String> warnings, Map<String, Object> info) {
        Path villagersPath = pack.getRootPath().resolve("villagers");
        if (!Files.exists(villagersPath, new LinkOption[0])) {
            return;
        }
        int workstations = VillagerPackValidator.countJsonFiles(villagersPath.resolve("workstations"));
        int poiTypes = VillagerPackValidator.countJsonFiles(villagersPath.resolve("poi_types"));
        int professions = VillagerPackValidator.countJsonFiles(villagersPath.resolve("professions"));
        int types = VillagerPackValidator.countJsonFiles(villagersPath.resolve("types"));
        int trades = VillagerPackValidator.countJsonFiles(villagersPath.resolve("trades"));
        int gifts = VillagerPackValidator.countJsonFiles(villagersPath.resolve("gifts"));
        int structureTags = VillagerPackValidator.countJsonFiles(villagersPath.resolve("structure_tags"));
        int biomeMappings = VillagerPackValidator.countJsonFiles(villagersPath.resolve("biome_mappings"));
        int biomeTrades = VillagerPackValidator.countJsonFiles(villagersPath.resolve("biome_trades"));
        info.put("Workstations", workstations);
        info.put("POI Types", poiTypes);
        info.put("Professions", professions);
        info.put("Villager Types", types);
        info.put("Trades", trades);
        info.put("Gifts", gifts);
        info.put("Structure Tags", structureTags);
        info.put("Biome Mappings", biomeMappings);
        info.put("Biome Trades", biomeTrades);
        if (professions > 0) {
            VillagerPackValidator.validateProfessionDependencies(villagersPath, errors, warnings);
        }
        if (structureTags > 0) {
            VillagerPackValidator.validateStructureTagDependencies(villagersPath, errors, warnings);
        }
        if (trades > 0) {
            VillagerPackValidator.validateTrades(villagersPath, errors, warnings);
        }
    }

    private static void validateProfessionDependencies(Path villagersPath, List<String> errors, List<String> warnings) {
        Path professionsPath = villagersPath.resolve("professions");
        Path poiTypesPath = villagersPath.resolve("poi_types");
        if (!Files.exists(professionsPath, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> files = Files.walk(professionsPath, new FileVisitOption[0]);){
            files.filter(p -> p.toString().endsWith(".json")).forEach(file -> {
                try {
                    JsonObject json = JsonParser.parseReader((Reader)new InputStreamReader(Files.newInputStream(file, new OpenOption[0]))).getAsJsonObject();
                    if (json.has("poi_type")) {
                        String poiType = json.get("poi_type").getAsString();
                        Path poiFile = poiTypesPath.resolve(poiType + ".json");
                        if (!Files.exists(poiFile, new LinkOption[0]) && !VillagerPackValidator.isVanillaPoiType(poiType)) {
                            errors.add("Profession '" + String.valueOf(file.getFileName()) + "' references POI type '" + poiType + "' which doesn't exist in the pack or vanilla - profession will not work!");
                        }
                    } else {
                        errors.add("Profession '" + String.valueOf(file.getFileName()) + "' missing required 'poi_type' field - profession will not work!");
                    }
                    if (!json.has("work_sound")) {
                        warnings.add("Profession '" + String.valueOf(file.getFileName()) + "' missing 'work_sound' field - will use default sound");
                    }
                    if (!json.has("namespace")) {
                        warnings.add("Profession '" + String.valueOf(file.getFileName()) + "' missing 'namespace' field - will use pack namespace");
                    }
                }
                catch (Exception e) {
                    errors.add("Failed to validate profession file " + String.valueOf(file.getFileName()) + ": " + e.getMessage());
                }
            });
        }
        catch (IOException e) {
            errors.add("Failed to read professions directory: " + e.getMessage());
        }
    }

    private static void validateStructureTagDependencies(Path villagersPath, List<String> errors, List<String> warnings) {
        Path structureTagsPath = villagersPath.resolve("structure_tags");
        if (!Files.exists(structureTagsPath, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> files = Files.walk(structureTagsPath, new FileVisitOption[0]);){
            files.filter(p -> p.toString().endsWith(".json")).forEach(file -> {
                try {
                    JsonObject json = JsonParser.parseReader((Reader)new InputStreamReader(Files.newInputStream(file, new OpenOption[0]))).getAsJsonObject();
                    if (!json.has("tag")) {
                        errors.add("Structure tag '" + String.valueOf(file.getFileName()) + "' missing required 'tag' field - map trades will fail!");
                    }
                    if (!json.has("map_decoration")) {
                        errors.add("Structure tag '" + String.valueOf(file.getFileName()) + "' missing required 'map_decoration' field - map trades will fail!");
                    }
                    if (!json.has("map_color")) {
                        errors.add("Structure tag '" + String.valueOf(file.getFileName()) + "' missing required 'map_color' field - map trades will fail!");
                    } else {
                        String color = json.get("map_color").getAsString();
                        if (!color.matches("^#[0-9A-Fa-f]{6}$")) {
                            errors.add("Structure tag '" + String.valueOf(file.getFileName()) + "' has invalid map_color format (should be #RRGGBB): " + color);
                        }
                    }
                }
                catch (Exception e) {
                    errors.add("Failed to validate structure tag file " + String.valueOf(file.getFileName()) + ": " + e.getMessage());
                }
            });
        }
        catch (IOException e) {
            errors.add("Failed to read structure_tags directory: " + e.getMessage());
        }
    }

    private static void validateTrades(Path villagersPath, List<String> errors, List<String> warnings) {
        Path tradesPath = villagersPath.resolve("trades");
        if (!Files.exists(tradesPath, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> files = Files.walk(tradesPath, new FileVisitOption[0]);){
            files.filter(p -> p.toString().endsWith(".json")).forEach(file -> {
                try {
                    JsonObject json = JsonParser.parseReader((Reader)new InputStreamReader(Files.newInputStream(file, new OpenOption[0]))).getAsJsonObject();
                    if (!json.has("profession")) {
                        errors.add("Trade file '" + String.valueOf(file.getFileName()) + "' missing required 'profession' field - trades will not load!");
                        return;
                    }
                    if (!json.has("levels")) {
                        errors.add("Trade file '" + String.valueOf(file.getFileName()) + "' missing required 'levels' field - trades will not load!");
                        return;
                    }
                    JsonObject levels = json.getAsJsonObject("levels");
                    if (levels.size() == 0) {
                        warnings.add("Trade file '" + String.valueOf(file.getFileName()) + "' has no level definitions");
                    }
                }
                catch (Exception e) {
                    errors.add("Failed to validate trade file " + String.valueOf(file.getFileName()) + ": " + e.getMessage());
                }
            });
        }
        catch (IOException e) {
            errors.add("Failed to read trades directory: " + e.getMessage());
        }
    }

    private static void validateDatapacks(VillagerPackLoader.VillagerPack pack, List<String> errors, List<String> warnings, Map<String, Object> info) {
        Path dataPath = pack.getRootPath().resolve("data");
        if (!Files.exists(dataPath, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> namespaces = Files.list(dataPath);){
            List<String> foundNamespaces = namespaces.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).map(p -> p.getFileName().toString()).toList();
            info.put("Data namespaces", String.join((CharSequence)", ", foundNamespaces));
            for (String namespace : foundNamespaces) {
                VillagerPackValidator.validateDataNamespace(dataPath.resolve(namespace), namespace, errors, warnings, info);
            }
        }
        catch (IOException e) {
            errors.add("Failed to read data directory: " + e.getMessage());
        }
    }

    private static void validateDataNamespace(Path namespacePath, String namespace, List<String> errors, List<String> warnings, Map<String, Object> info) {
        Path structureTagsPath = namespacePath.resolve("tags/worldgen/structure");
        if (Files.exists(structureTagsPath, new LinkOption[0])) {
            int tagCount = VillagerPackValidator.countJsonFiles(structureTagsPath);
            info.put("Structure tags in " + namespace, tagCount);
            try (Stream<Path> files = Files.walk(structureTagsPath, new FileVisitOption[0]);){
                files.filter(p -> p.toString().endsWith(".json")).forEach(file -> {
                    try {
                        JsonObject json = JsonParser.parseReader((Reader)new InputStreamReader(Files.newInputStream(file, new OpenOption[0]))).getAsJsonObject();
                        if (!json.has("values") || !json.get("values").isJsonArray()) {
                            errors.add("Structure tag " + namespace + ":" + structureTagsPath.relativize((Path)file).toString().replace(".json", "") + " missing or invalid 'values' array - treasure map trades will fail!");
                        } else {
                            JsonElement values = json.get("values");
                            if (values.getAsJsonArray().isEmpty()) {
                                warnings.add("Structure tag " + namespace + ":" + structureTagsPath.relativize((Path)file).toString().replace(".json", "") + " has empty values array - treasure map trades will not find structures");
                            }
                        }
                    }
                    catch (Exception e) {
                        errors.add("Failed to validate structure tag " + String.valueOf(file.getFileName()) + ": " + e.getMessage());
                    }
                });
            }
            catch (IOException e) {
                errors.add("Failed to read structure tags: " + e.getMessage());
            }
        }
    }

    private static void validateResourcePacks(VillagerPackLoader.VillagerPack pack, List<String> errors, List<String> warnings, Map<String, Object> info) {
        Path assetsPath = pack.getRootPath().resolve("assets");
        if (!Files.exists(assetsPath, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> namespaces = Files.list(assetsPath);){
            List<String> foundNamespaces = namespaces.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).map(p -> p.getFileName().toString()).toList();
            info.put("Asset namespaces", String.join((CharSequence)", ", foundNamespaces));
            for (String namespace : foundNamespaces) {
                VillagerPackValidator.validateResourceNamespace(assetsPath.resolve(namespace), namespace, warnings, info);
            }
        }
        catch (IOException e) {
            warnings.add("Failed to read assets directory: " + e.getMessage());
        }
    }

    private static void validateResourceNamespace(Path namespacePath, String namespace, List<String> warnings, Map<String, Object> info) {
        int langCount;
        int modelCount;
        int textureCount;
        Path texturesPath = namespacePath.resolve("textures");
        Path modelsPath = namespacePath.resolve("models");
        Path langPath = namespacePath.resolve("lang");
        if (Files.exists(texturesPath, new LinkOption[0]) && (textureCount = VillagerPackValidator.countFiles(texturesPath, ".png")) > 0) {
            info.put("Textures in " + namespace, textureCount);
        }
        if (Files.exists(modelsPath, new LinkOption[0]) && (modelCount = VillagerPackValidator.countJsonFiles(modelsPath)) > 0) {
            info.put("Models in " + namespace, modelCount);
        }
        if (Files.exists(langPath, new LinkOption[0]) && (langCount = VillagerPackValidator.countJsonFiles(langPath)) > 0) {
            info.put("Language files in " + namespace, langCount);
        }
        boolean hasTextures = Files.exists(texturesPath, new LinkOption[0]);
        boolean hasModels = Files.exists(modelsPath, new LinkOption[0]);
        if (hasTextures && !hasModels) {
            warnings.add("Resource pack has textures but no models in '" + namespace + "' - textures may not display");
        }
    }

    private static int countJsonFiles(Path path) {
        return VillagerPackValidator.countFiles(path, ".json");
    }

    private static int countFiles(Path path, String extension) {
        int n;
        block9: {
            if (!Files.exists(path, new LinkOption[0])) {
                return 0;
            }
            Stream<Path> files = Files.walk(path, new FileVisitOption[0]);
            try {
                n = (int)files.filter(p -> p.toString().endsWith(extension)).count();
                if (files == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (files != null) {
                        try {
                            files.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    return 0;
                }
            }
            files.close();
        }
        return n;
    }

    private static boolean isVanillaPoiType(String poiType) {
        Set<String> vanillaTypes = Set.of("armorer", "butcher", "cartographer", "cleric", "farmer", "fisherman", "fletcher", "leatherworker", "librarian", "stone_mason", "mason", "shepherd", "toolsmith", "weaponsmith", "home", "meeting", "beehive", "bee_nest", "nether_portal", "lodestone");
        String typeName = poiType.contains(":") ? poiType.substring(poiType.indexOf(":") + 1) : poiType;
        return vanillaTypes.contains(typeName);
    }

    public static void validateAllPacks(Path villagerPacksDir) {
        if (!Files.exists(villagerPacksDir, new LinkOption[0])) {
            VillagerAPI.LOGGER.info("No villagerpacks directory to validate");
            return;
        }
        VillagerAPI.LOGGER.info("=== Starting VillagerPack Validation ===");
        try (Stream<Path> packs = Files.list(villagerPacksDir);){
            List<Path> packPaths = packs.filter(path -> Files.isDirectory(path, new LinkOption[0]) || Files.isRegularFile(path, new LinkOption[0]) && path.toString().endsWith(".zip")).toList();
            int totalPacks = packPaths.size();
            int validPacks = 0;
            int invalidPacks = 0;
            int criticalFailures = 0;
            for (Path packPath : packPaths) {
                String packName = packPath.getFileName().toString();
                VillagerAPI.LOGGER.info("Validating pack: {}", (Object)packName);
                ValidationResult result = VillagerPackValidator.validatePack(packPath);
                result.logResults(packName);
                if (result.isCriticalFailure()) {
                    ++criticalFailures;
                    ++invalidPacks;
                } else if (result.isValid()) {
                    ++validPacks;
                } else {
                    ++invalidPacks;
                }
                VillagerAPI.LOGGER.info("---");
            }
            VillagerAPI.LOGGER.info("=== Validation Complete ===");
            VillagerAPI.LOGGER.info("Total packs: {}, Valid: {}, Invalid: {}, Critical failures: {}", new Object[]{totalPacks, validPacks, invalidPacks, criticalFailures});
            if (criticalFailures > 0) {
                VillagerAPI.LOGGER.error("{} pack(s) have critical failures and will not load!", (Object)criticalFailures);
            }
        }
        catch (IOException e) {
            VillagerAPI.LOGGER.error("Failed to validate packs: {}", (Object)e.getMessage());
        }
    }

    public static class ValidationResult {
        private final boolean valid;
        private final boolean criticalFailure;
        private final List<String> errors;
        private final List<String> warnings;
        private final Map<String, Object> info;

        public ValidationResult(boolean valid, boolean criticalFailure, List<String> errors, List<String> warnings, Map<String, Object> info) {
            this.valid = valid;
            this.criticalFailure = criticalFailure;
            this.errors = errors;
            this.warnings = warnings;
            this.info = info;
        }

        public boolean isValid() {
            return this.valid;
        }

        public boolean isCriticalFailure() {
            return this.criticalFailure;
        }

        public List<String> getErrors() {
            return this.errors;
        }

        public List<String> getWarnings() {
            return this.warnings;
        }

        public Map<String, Object> getInfo() {
            return this.info;
        }

        public void logResults(String packName) {
            if (this.criticalFailure) {
                VillagerAPI.LOGGER.error("=== CRITICAL VALIDATION FAILURE for pack '{}' - PACK WILL NOT LOAD ===", (Object)packName);
                this.errors.forEach(err -> VillagerAPI.LOGGER.error("  [CRITICAL] {}", err));
            } else if (!this.errors.isEmpty()) {
                VillagerAPI.LOGGER.error("Validation errors for pack '{}' ({} error(s)):", (Object)packName, (Object)this.errors.size());
                this.errors.forEach(err -> VillagerAPI.LOGGER.error("  [ERROR] {}", err));
            }
            if (!this.warnings.isEmpty()) {
                VillagerAPI.LOGGER.warn("Validation warnings for pack '{}' ({} warning(s)):", (Object)packName, (Object)this.warnings.size());
                this.warnings.forEach(warn -> VillagerAPI.LOGGER.warn("  [WARN] {}", warn));
            }
            if (this.valid || !this.criticalFailure) {
                VillagerAPI.LOGGER.info("Pack '{}' validation completed - Pack will load", (Object)packName);
                if (!this.info.isEmpty()) {
                    this.info.forEach((key, value) -> VillagerAPI.LOGGER.info("  - {}: {}", key, value));
                }
            }
        }
    }
}

