/*
 * Decompiled with CFR 0.152.
 */
package org.bxteam.quark.dependency;

import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;
import org.bxteam.quark.dependency.Dependency;
import org.bxteam.quark.dependency.model.DownloadResult;
import org.bxteam.quark.dependency.model.PomContext;
import org.bxteam.quark.dependency.model.ResolutionResult;
import org.bxteam.quark.dependency.model.ResolvedDependency;
import org.bxteam.quark.logger.Logger;
import org.bxteam.quark.pom.MetadataReader;
import org.bxteam.quark.pom.PomReader;
import org.bxteam.quark.pom.model.MavenMetadata;
import org.bxteam.quark.pom.model.ParentInfo;
import org.bxteam.quark.pom.model.PomInfo;
import org.bxteam.quark.repository.Repository;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class DependencyResolver {
    private final Logger logger;
    private final List<Repository> repositories;
    private final Path localRepository;
    private final PomReader pomReader;
    private final MetadataReader metadataReader;
    private final boolean includeDependencyManagement;
    private final int maxTransitiveDepth;
    private final Set<String> excludedGroupIds;
    private final List<Pattern> excludedArtifactPatterns;
    private final boolean skipOptionalDependencies;
    private final boolean skipTestDependencies;
    private final Map<String, PomInfo> pomCache = new ConcurrentHashMap<String, PomInfo>();
    private final Map<String, MavenMetadata> metadataCache = new ConcurrentHashMap<String, MavenMetadata>();
    private final Map<String, String> resolvedVersionCache = new ConcurrentHashMap<String, String>();
    private final Set<String> processedDependencies = ConcurrentHashMap.newKeySet();
    private final Map<String, Integer> dependencyDepths = new ConcurrentHashMap<String, Integer>();
    private final Map<String, String> globalDependencyManagement = new ConcurrentHashMap<String, String>();

    public DependencyResolver(@NotNull Logger logger, @NotNull List<Repository> repositories, @NotNull Path localRepository) {
        this(new Builder(logger, repositories, localRepository));
    }

    private DependencyResolver(Builder builder) {
        this.logger = Objects.requireNonNull(builder.logger, "Logger cannot be null");
        this.repositories = new ArrayList<Repository>((Collection)Objects.requireNonNull(builder.repositories, "Repositories cannot be null"));
        this.localRepository = Objects.requireNonNull(builder.localRepository, "Local repository cannot be null");
        this.pomReader = new PomReader();
        this.metadataReader = new MetadataReader();
        this.includeDependencyManagement = builder.includeDependencyManagement;
        this.maxTransitiveDepth = builder.maxTransitiveDepth;
        this.excludedGroupIds = new HashSet<String>(builder.excludedGroupIds);
        this.excludedArtifactPatterns = new ArrayList<Pattern>(builder.excludedArtifactPatterns);
        this.skipOptionalDependencies = builder.skipOptionalDependencies;
        this.skipTestDependencies = builder.skipTestDependencies;
    }

    @NotNull
    public ResolutionResult resolveDependencies(@NotNull Collection<Dependency> rootDependencies) {
        Objects.requireNonNull(rootDependencies, "Root dependencies cannot be null");
        this.logger.info("Resolving dependencies...");
        this.pomCache.clear();
        this.metadataCache.clear();
        this.resolvedVersionCache.clear();
        this.processedDependencies.clear();
        this.dependencyDepths.clear();
        this.globalDependencyManagement.clear();
        LinkedHashSet<Dependency> allDependencies = new LinkedHashSet<Dependency>();
        LinkedHashSet<Dependency> toProcess = new LinkedHashSet<Dependency>(rootDependencies);
        ArrayList<String> resolutionErrors = new ArrayList<String>();
        for (Dependency root : rootDependencies) {
            this.dependencyDepths.put(root.getCoordinates(), 0);
        }
        int iteration = 1;
        while (!toProcess.isEmpty()) {
            LinkedHashSet<Dependency> currentBatch = new LinkedHashSet<Dependency>(toProcess);
            toProcess.clear();
            this.logger.debug("=== Iteration " + iteration + " - Processing " + currentBatch.size() + " dependencies ===");
            for (Dependency dependency : currentBatch) {
                String dependencyKey = dependency.getCoordinates();
                if (this.processedDependencies.contains(dependencyKey)) continue;
                try {
                    if (this.shouldExcludeDependency(dependency)) {
                        this.logger.debug("Skipping excluded dependency: " + dependency.toShortString());
                        this.processedDependencies.add(dependencyKey);
                        continue;
                    }
                    int currentDepth = this.dependencyDepths.getOrDefault(dependencyKey, 0);
                    if (currentDepth > 0 && currentDepth > this.maxTransitiveDepth) {
                        this.logger.debug("Skipping dependency exceeding max depth: " + dependency.toShortString());
                        this.processedDependencies.add(dependencyKey);
                        continue;
                    }
                    Dependency resolvedDependency = this.resolveDependencyVersion(dependency);
                    this.logger.debug("Processing: " + resolvedDependency.toShortString());
                    allDependencies.add(resolvedDependency);
                    this.processedDependencies.add(dependencyKey);
                    PomContext pomContext = this.downloadAndProcessPom(resolvedDependency);
                    if (pomContext == null) continue;
                    this.logger.debug("Found " + pomContext.allDependencies().size() + " transitive dependencies for " + resolvedDependency.toShortString());
                    for (Dependency transitive : pomContext.allDependencies()) {
                        String transitiveKey = transitive.getCoordinates();
                        if (this.processedDependencies.contains(transitiveKey)) continue;
                        this.dependencyDepths.put(transitiveKey, currentDepth + 1);
                        toProcess.add(transitive);
                        this.logger.debug("  + " + transitive.toShortString() + " (depth " + (currentDepth + 1) + ")");
                    }
                }
                catch (Exception e) {
                    String errorMsg = "Failed to resolve dependency " + dependency.toShortString() + ": " + e.getMessage();
                    resolutionErrors.add(errorMsg);
                    this.logger.debug("Resolution error for " + dependency.toShortString() + ": " + e.getMessage());
                }
            }
            if (++iteration <= 50) continue;
            resolutionErrors.add("Maximum resolution iterations reached - possible circular dependencies");
            break;
        }
        this.logger.info("Resolved " + allDependencies.size() + " dependencies");
        ArrayList<ResolvedDependency> resolvedDependencies = new ArrayList<ResolvedDependency>();
        for (Dependency dependency : allDependencies) {
            try {
                DownloadResult downloadResult = this.downloadJar(dependency);
                resolvedDependencies.add(new ResolvedDependency(dependency, downloadResult.jarPath()));
                if (downloadResult.downloadedFrom() == null) continue;
                this.logger.info("Downloaded " + dependency.toShortString() + " from " + downloadResult.downloadedFrom());
            }
            catch (Exception e) {
                String errorMsg = "Failed to download JAR for " + dependency.toShortString() + ": " + e.getMessage();
                resolutionErrors.add(errorMsg);
                this.logger.debug("Download error for " + dependency.toShortString() + ": " + e.getMessage());
            }
        }
        return new ResolutionResult(resolvedDependencies, resolutionErrors);
    }

    private boolean shouldExcludeDependency(Dependency dependency) {
        if (this.excludedGroupIds.contains(dependency.getGroupId())) {
            return true;
        }
        String coordinates = dependency.getGroupArtifactId();
        for (Pattern pattern : this.excludedArtifactPatterns) {
            if (!pattern.matcher(coordinates).matches()) continue;
            return true;
        }
        return false;
    }

    @Nullable
    private PomContext downloadAndProcessPom(@NotNull Dependency dependency) throws Exception {
        PomInfo pomInfo = this.downloadAndParsePom(dependency);
        if (pomInfo == null) {
            this.logger.debug("No POM found for: " + dependency.toShortString());
            return null;
        }
        PomInfo mergedPomInfo = this.processParentHierarchy(pomInfo);
        List<Dependency> resolvedDependencies = this.resolveAllDependencies(mergedPomInfo);
        return new PomContext(mergedPomInfo, resolvedDependencies);
    }

    @NotNull
    private PomInfo processParentHierarchy(@NotNull PomInfo pomInfo) throws Exception {
        ArrayList<PomInfo> pomHierarchy = new ArrayList<PomInfo>();
        PomInfo currentPom = pomInfo;
        pomHierarchy.add(currentPom);
        while (currentPom.hasParent()) {
            ParentInfo parentInfo = currentPom.parentInfo();
            Dependency parentDependency = parentInfo.toDependency();
            this.logger.debug("Processing parent POM: " + parentDependency.toShortString());
            PomInfo parentPom = this.downloadAndParsePom(parentDependency);
            if (parentPom == null) {
                this.logger.debug("Could not download parent POM: " + parentDependency.toShortString());
                break;
            }
            pomHierarchy.add(parentPom);
            currentPom = parentPom;
            if (pomHierarchy.size() <= 10) continue;
            this.logger.warn("Maximum parent hierarchy depth reached (10) for " + pomInfo.artifactId());
            break;
        }
        return this.mergePomHierarchy(pomHierarchy);
    }

    @NotNull
    private PomInfo mergePomHierarchy(@NotNull List<PomInfo> pomHierarchy) {
        if (pomHierarchy.isEmpty()) {
            throw new IllegalArgumentException("POM hierarchy cannot be empty");
        }
        if (pomHierarchy.size() == 1) {
            PomInfo singlePom = pomHierarchy.get(0);
            this.globalDependencyManagement.putAll(singlePom.dependencyManagement());
            this.logger.debug("Added " + singlePom.dependencyManagement().size() + " entries to global dependency management from " + singlePom.artifactId());
            return singlePom;
        }
        HashMap<String, String> mergedDependencyManagement = new HashMap<String, String>();
        HashMap<String, String> mergedProperties = new HashMap<String, String>();
        for (int i = pomHierarchy.size() - 1; i >= 0; --i) {
            PomInfo pom = pomHierarchy.get(i);
            Map<String, String> pomProperties = pom.properties();
            for (Map.Entry<String, String> entry : pomProperties.entrySet()) {
                mergedProperties.putIfAbsent(entry.getKey(), entry.getValue());
            }
            Map<String, String> pomDependencyManagement = pom.dependencyManagement();
            for (Map.Entry<String, String> entry : pomDependencyManagement.entrySet()) {
                if (i == 0) {
                    mergedDependencyManagement.put(entry.getKey(), entry.getValue());
                    continue;
                }
                mergedDependencyManagement.putIfAbsent(entry.getKey(), entry.getValue());
            }
        }
        this.globalDependencyManagement.putAll(mergedDependencyManagement);
        this.logger.debug("Merged " + mergedDependencyManagement.size() + " dependency management entries from hierarchy");
        PomInfo childPom = pomHierarchy.get(0);
        return new PomInfo(childPom.groupId(), childPom.artifactId(), childPom.version(), childPom.dependencies(), mergedProperties, mergedDependencyManagement, childPom.parentInfo());
    }

    @NotNull
    private List<Dependency> resolveAllDependencies(@NotNull PomInfo pomInfo) {
        ArrayList<Dependency> resolved = new ArrayList<Dependency>();
        for (Dependency dependency : pomInfo.getRuntimeDependencies()) {
            try {
                Dependency resolvedDep;
                if (this.skipOptionalDependencies && dependency.isOptional()) {
                    this.logger.debug("Skipping optional dependency: " + dependency.toShortString());
                    continue;
                }
                if (this.skipTestDependencies && "test".equals(dependency.getScope())) {
                    this.logger.debug("Skipping test dependency: " + dependency.toShortString());
                    continue;
                }
                if (this.shouldExcludeDependency(dependency)) {
                    this.logger.debug("Skipping excluded dependency: " + dependency.toShortString());
                    continue;
                }
                if (dependency.getVersion() == null || dependency.getVersion().trim().isEmpty()) {
                    resolvedDep = this.resolveDependencyFromManagement(dependency, pomInfo);
                    this.logger.debug("Resolved version from dependencyManagement: " + resolvedDep.toShortString());
                } else {
                    resolvedDep = dependency;
                }
                if (resolvedDep.getVersion() == null || resolvedDep.getVersion().trim().isEmpty()) {
                    resolvedDep = this.resolveDependencyVersion(resolvedDep);
                }
                resolved.add(resolvedDep);
                this.logger.debug("Resolved transitive dependency: " + resolvedDep.toShortString());
            }
            catch (Exception e) {
                this.logger.debug("Could not resolve dependency: " + dependency.getGroupArtifactId() + " - " + e.getMessage());
                try {
                    Dependency withoutVersion = Dependency.builder().groupId(dependency.getGroupId()).artifactId(dependency.getArtifactId()).version("").build();
                    Dependency resolvedFromMetadata = this.resolveDependencyVersion(withoutVersion);
                    resolved.add(resolvedFromMetadata);
                    this.logger.debug("Resolved from metadata: " + resolvedFromMetadata.toShortString());
                }
                catch (Exception e2) {
                    this.logger.debug("Failed to resolve from metadata: " + dependency.getGroupArtifactId() + " - " + e2.getMessage());
                }
            }
        }
        if (this.includeDependencyManagement) {
            this.resolveDependenciesFromDependencyManagement(pomInfo, resolved);
        }
        return resolved;
    }

    private void resolveDependenciesFromDependencyManagement(@NotNull PomInfo pomInfo, @NotNull List<Dependency> resolvedList) {
        HashMap<String, String> dependencyManagement = new HashMap<String, String>(pomInfo.dependencyManagement());
        dependencyManagement.putAll(this.globalDependencyManagement);
        this.logger.debug("Processing " + dependencyManagement.size() + " entries from dependencyManagement");
        for (Map.Entry entry : dependencyManagement.entrySet()) {
            String dependencyKey = (String)entry.getKey();
            String version = (String)entry.getValue();
            try {
                String[] parts;
                boolean alreadyIncluded = resolvedList.stream().anyMatch(d -> d.getGroupArtifactId().equals(dependencyKey));
                if (alreadyIncluded || (parts = dependencyKey.split(":")).length != 2) continue;
                Dependency managementDep = Dependency.builder().groupId(parts[0]).artifactId(parts[1]).version(version).build();
                if (this.shouldExcludeDependency(managementDep)) {
                    this.logger.debug("Skipping excluded dependency from management: " + managementDep.toShortString());
                    continue;
                }
                if (managementDep.getGroupId() == null || managementDep.getArtifactId() == null || managementDep.getVersion() == null || managementDep.getVersion().isEmpty()) continue;
                resolvedList.add(managementDep);
                this.logger.debug("Added dependency from management: " + managementDep.toShortString());
            }
            catch (Exception e) {
                this.logger.debug("Failed to process dependencyManagement entry: " + dependencyKey + " - " + e.getMessage());
            }
        }
    }

    @NotNull
    private Dependency resolveDependencyFromManagement(@NotNull Dependency dependency, @NotNull PomInfo pomInfo) {
        if (dependency.getVersion() != null && !dependency.getVersion().trim().isEmpty()) {
            return dependency;
        }
        String dependencyKey = dependency.getGroupArtifactId();
        String version = null;
        if (pomInfo.dependencyManagement() != null && (version = pomInfo.dependencyManagement().get(dependencyKey)) != null && !version.trim().isEmpty()) {
            this.logger.debug("Found version in local dependency management for " + dependencyKey + ": " + version);
        }
        if (!(version != null && !version.trim().isEmpty() || this.globalDependencyManagement.isEmpty() || (version = this.globalDependencyManagement.get(dependencyKey)) == null || version.trim().isEmpty())) {
            this.logger.debug("Found version in global dependency management for " + dependencyKey + ": " + version);
        }
        if ((version == null || version.trim().isEmpty()) && (version = this.resolvedVersionCache.get(dependencyKey)) != null && !version.trim().isEmpty()) {
            this.logger.debug("Found version in resolved version cache for " + dependencyKey + ": " + version);
        }
        if (version != null && !version.trim().isEmpty()) {
            this.logger.debug("Resolved version for " + dependencyKey + " -> " + version);
            return dependency.withVersion(version);
        }
        this.logger.debug("Attempting to resolve version from metadata for " + dependencyKey);
        return this.resolveDependencyVersion(dependency);
    }

    @NotNull
    private Dependency resolveDependencyVersion(@NotNull Dependency dependency) {
        if (dependency.getVersion() != null && !dependency.getVersion().trim().isEmpty()) {
            return dependency;
        }
        String versionKey = dependency.getGroupId() + ":" + dependency.getArtifactId();
        String cachedVersion = this.resolvedVersionCache.get(versionKey);
        if (cachedVersion != null) {
            return dependency.withVersion(cachedVersion);
        }
        try {
            String bestVersion;
            MavenMetadata metadata = this.downloadAndParseMetadata(dependency);
            if (metadata != null && (bestVersion = metadata.getBestVersion()) != null) {
                this.resolvedVersionCache.put(versionKey, bestVersion);
                this.logger.debug("Resolved version from metadata: " + dependency.getGroupArtifactId() + " -> " + bestVersion);
                return dependency.withVersion(bestVersion);
            }
        }
        catch (Exception e) {
            this.logger.debug("Failed to resolve version for " + dependency.getGroupArtifactId() + ": " + e.getMessage());
        }
        throw new RuntimeException("Cannot resolve version for dependency: " + dependency.getGroupArtifactId());
    }

    @Nullable
    private MavenMetadata downloadAndParseMetadata(@NotNull Dependency dependency) throws Exception {
        String metadataKey = dependency.getGroupArtifactId();
        MavenMetadata cached = this.metadataCache.get(metadataKey);
        if (cached != null) {
            return cached;
        }
        for (Repository repository : this.repositories) {
            MavenMetadata mavenMetadata;
            block10: {
                if (repository.isLocal()) continue;
                URL metadataUrl = dependency.getMetadataUri(repository.getUrl()).toURL();
                this.logger.debug("Downloading metadata from: " + String.valueOf(metadataUrl));
                URLConnection connection = metadataUrl.openConnection();
                connection.setConnectTimeout(30000);
                connection.setReadTimeout(60000);
                connection.setRequestProperty("User-Agent", "Quark-LibraryManager/2.0");
                InputStream inputStream = connection.getInputStream();
                try {
                    MavenMetadata metadata = this.metadataReader.readMetadata(inputStream, metadataUrl.toString());
                    this.metadataCache.put(metadataKey, metadata);
                    this.logger.debug("Successfully parsed metadata for " + metadataKey);
                    mavenMetadata = metadata;
                    if (inputStream == null) break block10;
                }
                catch (Throwable throwable) {
                    try {
                        if (inputStream != null) {
                            try {
                                inputStream.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (Exception e) {
                        this.logger.debug("Failed to download metadata from " + repository.getUrl() + ": " + e.getMessage());
                    }
                }
                inputStream.close();
            }
            return mavenMetadata;
        }
        this.logger.debug("No metadata found for " + metadataKey);
        return null;
    }

    @Nullable
    private PomInfo downloadAndParsePom(@NotNull Dependency dependency) throws Exception {
        String pomKey = dependency.getCoordinates();
        PomInfo cached = this.pomCache.get(pomKey);
        if (cached != null) {
            return cached;
        }
        Path localPomPath = dependency.getPomPath(this.localRepository);
        if (Files.exists(localPomPath, new LinkOption[0]) && this.isValidPomFile(localPomPath)) {
            this.logger.debug("Using cached POM: " + dependency.toShortString());
        } else {
            try {
                this.downloadPomFile(dependency, localPomPath);
            }
            catch (Exception e) {
                this.logger.debug("Could not download POM for " + dependency.toShortString() + ": " + e.getMessage());
                return null;
            }
        }
        if (Files.exists(localPomPath, new LinkOption[0])) {
            try {
                PomInfo pomInfo = this.pomReader.readPom(localPomPath);
                this.pomCache.put(pomKey, pomInfo);
                this.logger.debug("Successfully parsed POM for " + dependency.toShortString());
                return pomInfo;
            }
            catch (Exception e) {
                this.logger.debug("Could not parse POM for " + dependency.toShortString() + ": " + e.getMessage());
            }
        }
        return null;
    }

    private void downloadPomFile(@NotNull Dependency dependency, @NotNull Path localPomPath) throws Exception {
        ArrayList<Exception> exceptions = new ArrayList<Exception>();
        Files.createDirectories(localPomPath.getParent(), new FileAttribute[0]);
        ArrayList<Repository> reposToTry = new ArrayList<Repository>(this.repositories);
        if (dependency.getFallbackRepository() != null) {
            reposToTry.add(Repository.of(dependency.getFallbackRepository()));
        }
        for (Repository repository : reposToTry) {
            if (repository.isLocal()) continue;
            try {
                URL pomUrl = dependency.getPomUri(repository.getUrl()).toURL();
                this.logger.debug("Downloading POM from: " + String.valueOf(pomUrl));
                URLConnection connection = pomUrl.openConnection();
                connection.setConnectTimeout(30000);
                connection.setReadTimeout(60000);
                connection.setRequestProperty("User-Agent", "Quark-LibraryManager/2.0");
                try (InputStream inputStream = connection.getInputStream();){
                    Files.copy(inputStream, localPomPath, new CopyOption[0]);
                    this.logger.debug("Successfully downloaded POM: " + dependency.toShortString());
                    return;
                }
            }
            catch (Exception e) {
                exceptions.add(e);
                this.logger.debug("Failed to download POM from " + repository.getUrl() + ": " + e.getMessage());
            }
        }
        RuntimeException lastException = exceptions.isEmpty() ? new RuntimeException("No repositories configured") : (Exception)exceptions.get(exceptions.size() - 1);
        throw new RuntimeException("Failed to download POM for " + dependency.toShortString(), lastException);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @NotNull
    private DownloadResult downloadJar(@NotNull Dependency dependency) throws Exception {
        Path localJarPath = dependency.getJarPath(this.localRepository);
        if (Files.exists(localJarPath, new LinkOption[0]) && this.isValidJarFile(localJarPath)) {
            return new DownloadResult(localJarPath, null);
        }
        ArrayList<Exception> exceptions = new ArrayList<Exception>();
        Files.createDirectories(localJarPath.getParent(), new FileAttribute[0]);
        ArrayList<Repository> reposToTry = new ArrayList<Repository>(this.repositories);
        if (dependency.getFallbackRepository() != null) {
            reposToTry.add(Repository.of(dependency.getFallbackRepository()));
        }
        for (Repository repository : reposToTry) {
            if (repository.isLocal()) continue;
            try {
                URL jarUrl = dependency.getJarUri(repository.getUrl()).toURL();
                URLConnection connection = jarUrl.openConnection();
                connection.setConnectTimeout(30000);
                connection.setReadTimeout(60000);
                connection.setRequestProperty("User-Agent", "Quark-LibraryManager/2.0");
                try (InputStream inputStream = connection.getInputStream();){
                    Files.copy(inputStream, localJarPath, new CopyOption[0]);
                    if (this.isValidJarFile(localJarPath)) {
                        DownloadResult downloadResult = new DownloadResult(localJarPath, repository.getUrl());
                        return downloadResult;
                    }
                    Files.deleteIfExists(localJarPath);
                    throw new RuntimeException("Downloaded JAR is invalid");
                }
            }
            catch (Exception e) {
                exceptions.add(e);
            }
        }
        RuntimeException lastException = exceptions.isEmpty() ? new RuntimeException("No repositories configured") : (Exception)exceptions.get(exceptions.size() - 1);
        throw new RuntimeException("Failed to download JAR for " + dependency.toShortString(), lastException);
    }

    private boolean isValidPomFile(@NotNull Path pomFile) {
        try {
            if (!Files.exists(pomFile, new LinkOption[0]) || !Files.isRegularFile(pomFile, new LinkOption[0]) || Files.size(pomFile) == 0L) {
                return false;
            }
            String content = Files.readString(pomFile).trim();
            return !content.isEmpty() && content.contains("<project") && !content.toLowerCase().contains("<html");
        }
        catch (Exception e) {
            return false;
        }
    }

    private boolean isValidJarFile(@NotNull Path jarFile) {
        try {
            return Files.exists(jarFile, new LinkOption[0]) && Files.isRegularFile(jarFile, new LinkOption[0]) && Files.size(jarFile) > 0L;
        }
        catch (Exception e) {
            return false;
        }
    }

    public static class Builder {
        private final Logger logger;
        private final List<Repository> repositories;
        private final Path localRepository;
        private boolean includeDependencyManagement = false;
        private int maxTransitiveDepth = Integer.MAX_VALUE;
        private final Set<String> excludedGroupIds = new HashSet<String>();
        private final List<Pattern> excludedArtifactPatterns = new ArrayList<Pattern>();
        private boolean skipOptionalDependencies = true;
        private boolean skipTestDependencies = true;

        public Builder(@NotNull Logger logger, @NotNull List<Repository> repositories, @NotNull Path localRepository) {
            this.logger = Objects.requireNonNull(logger, "Logger cannot be null");
            this.repositories = Objects.requireNonNull(repositories, "Repositories cannot be null");
            this.localRepository = Objects.requireNonNull(localRepository, "Local repository cannot be null");
        }

        public Builder includeDependencyManagement(boolean include) {
            this.includeDependencyManagement = include;
            return this;
        }

        public Builder maxTransitiveDepth(int depth) {
            this.maxTransitiveDepth = Math.max(0, depth);
            return this;
        }

        public Builder excludeGroupIds(String ... groupIds) {
            this.excludedGroupIds.addAll(Arrays.asList(groupIds));
            return this;
        }

        public Builder excludeArtifacts(String ... patterns) {
            for (String pattern : patterns) {
                String regex = pattern.replace(".", "\\.").replace("*", ".*");
                this.excludedArtifactPatterns.add(Pattern.compile(regex));
            }
            return this;
        }

        public Builder skipOptionalDependencies(boolean skip) {
            this.skipOptionalDependencies = skip;
            return this;
        }

        public Builder skipTestDependencies(boolean skip) {
            this.skipTestDependencies = skip;
            return this;
        }

        public DependencyResolver build() {
            return new DependencyResolver(this);
        }
    }
}

