package net.mcbrawls.blueprint.editor

import net.mcbrawls.blueprint.BlueprintMod
import net.minecraft.registry.RegistryKey
import net.minecraft.registry.RegistryKeys
import net.minecraft.server.MinecraftServer
import net.minecraft.server.network.ServerPlayerEntity
import net.minecraft.util.Identifier
import net.minecraft.util.math.BlockPos
import net.minecraft.util.math.Vec3d
import net.minecraft.world.GameMode
import net.minecraft.world.World
import net.minecraft.world.biome.BiomeKeys
import net.minecraft.world.dimension.DimensionType
import net.minecraft.world.dimension.DimensionTypes
import xyz.nucleoid.fantasy.Fantasy
import xyz.nucleoid.fantasy.RuntimeWorldConfig
import xyz.nucleoid.fantasy.RuntimeWorldHandle
import xyz.nucleoid.fantasy.util.VoidChunkGenerator

object BlueprintEditorHandler {
    private val handles: MutableMap<Identifier, RuntimeWorldHandle> = mutableMapOf()
    private val keys: MutableMap<RegistryKey<World>, Identifier> = mutableMapOf()

    private val DIMENSION_KEY: RegistryKey<DimensionType> = RegistryKey.of(RegistryKeys.DIMENSION_TYPE, Identifier.of(BlueprintMod.MOD_ID, "editor"))

    /**
     * Opens a Blueprint editor environment.
     * @return whether the blueprint opened is new
     */
    internal fun open(
        server: MinecraftServer,
        blueprintId: Identifier,
        player: ServerPlayerEntity?
    ): Boolean {
        var new = false

        // create world
        val fantasy = Fantasy.get(server)
        val (handle, world) = handles[blueprintId]?.let { handle -> handle to (handle.asWorld() as BlueprintEditorWorld) }
            ?: run {
                val handle = fantasy.openTemporaryWorld(
                    RuntimeWorldConfig()
                        .setMirrorOverworldGameRules(true)
                        .setMirrorOverworldDifficulty(true)
                        .setGenerator(VoidChunkGenerator(server, BiomeKeys.THE_VOID))
                        .setDimensionType(DimensionTypes.OVERWORLD)
                        .setWorldConstructor { server, key, config, _ -> BlueprintEditorWorld(blueprintId, server, key, config) }
                )

                // initialize world
                val world = handle.asWorld() as BlueprintEditorWorld
                new = world.initializeBlueprint()

                handle to world
            }

        // store
        handles[blueprintId] = handle

        keys[world.registryKey] = blueprintId

        // teleport player
        player?.also { player ->
            val minPos = world.minPos
            val maxPos = world.maxPos
            val posDiff = maxPos.subtract(minPos)
            val pos = BlockPos((minPos.x + posDiff.x / 2), (minPos.y + posDiff.y / 2), (minPos.z + posDiff.z / 2))
            val vec = Vec3d.ofBottomCenter(pos)
            player.teleport(world, vec.x, vec.y, vec.z, setOf(), 0.0f, 0.0f, true)

            player.changeGameMode(GameMode.SPECTATOR)
        }

        return new
    }

    internal fun close(world: BlueprintEditorWorld) {
        // get handle
        val key = world.registryKey
        val blueprintId = keys[key] ?: return
        val handle = handles[blueprintId] ?: return

        // delete world
        handle.delete()

        // remove keys
        keys.remove(key)
        handles.remove(blueprintId)
    }
}
