/*
 * Decompiled with CFR 0.152.
 */
package github.nighter.smartspawner.logging.discord;

import github.nighter.smartspawner.Scheduler;
import github.nighter.smartspawner.SmartSpawner;
import github.nighter.smartspawner.logging.SpawnerLogEntry;
import github.nighter.smartspawner.logging.discord.DiscordEmbed;
import github.nighter.smartspawner.logging.discord.DiscordEmbedBuilder;
import github.nighter.smartspawner.logging.discord.DiscordWebhookConfig;
import java.io.IOException;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;

public class DiscordWebhookLogger {
    private final SmartSpawner plugin;
    private final DiscordWebhookConfig config;
    private final ConcurrentLinkedQueue<SpawnerLogEntry> webhookQueue;
    private final AtomicBoolean isShuttingDown;
    private final AtomicLong lastWebhookTime;
    private final AtomicLong webhooksSentThisMinute;
    private Scheduler.Task webhookTask;
    private static final int MAX_REQUESTS_PER_MINUTE = 25;
    private static final long MINUTE_IN_MILLIS = 60000L;

    public DiscordWebhookLogger(SmartSpawner plugin, DiscordWebhookConfig config) {
        this.plugin = plugin;
        this.config = config;
        this.webhookQueue = new ConcurrentLinkedQueue();
        this.isShuttingDown = new AtomicBoolean(false);
        this.lastWebhookTime = new AtomicLong(System.currentTimeMillis());
        this.webhooksSentThisMinute = new AtomicLong(0L);
        if (config.isEnabled()) {
            this.startWebhookTask();
        }
    }

    public void queueWebhook(SpawnerLogEntry entry) {
        if (!this.config.isEnabled() || !this.config.isEventEnabled(entry.getEventType())) {
            return;
        }
        this.webhookQueue.offer(entry);
    }

    private void startWebhookTask() {
        this.webhookTask = Scheduler.runTaskTimerAsync(() -> {
            if (this.isShuttingDown.get()) {
                return;
            }
            this.processWebhookQueue();
        }, 40L, 40L);
    }

    private void processWebhookQueue() {
        if (this.webhookQueue.isEmpty()) {
            return;
        }
        long currentTime = System.currentTimeMillis();
        long timeSinceLastCheck = currentTime - this.lastWebhookTime.get();
        if (timeSinceLastCheck >= 60000L) {
            this.webhooksSentThisMinute.set(0L);
            this.lastWebhookTime.set(currentTime);
        }
        while (!this.webhookQueue.isEmpty() && this.webhooksSentThisMinute.get() < 25L) {
            SpawnerLogEntry entry = this.webhookQueue.poll();
            if (entry == null) continue;
            this.sendWebhook(entry);
            this.webhooksSentThisMinute.incrementAndGet();
        }
        if (this.webhookQueue.size() > 50) {
            this.plugin.getLogger().warning("Discord webhook queue is backing up: " + this.webhookQueue.size() + " entries pending");
        }
    }

    private void sendWebhook(SpawnerLogEntry entry) {
        String webhookUrl = this.config.getWebhookUrl();
        if (webhookUrl == null || webhookUrl.isEmpty()) {
            return;
        }
        try {
            DiscordEmbed embed = DiscordEmbedBuilder.buildEmbed(entry, this.config, this.plugin);
            String jsonPayload = embed.toJson();
            Scheduler.runTaskAsync(() -> {
                try {
                    this.sendHttpRequest(webhookUrl, jsonPayload);
                }
                catch (IOException e) {
                    this.plugin.getLogger().log(Level.WARNING, "Failed to send Discord webhook", e);
                }
            });
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.WARNING, "Error building Discord embed", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void sendHttpRequest(String webhookUrl, String jsonPayload) throws IOException {
        HttpURLConnection connection = null;
        try {
            URL url = new URL(webhookUrl);
            connection = (HttpURLConnection)url.openConnection();
            connection.setRequestMethod("POST");
            connection.setRequestProperty("Content-Type", "application/json");
            connection.setRequestProperty("User-Agent", "SmartSpawner-Logger/1.0");
            connection.setDoOutput(true);
            connection.setConnectTimeout(5000);
            connection.setReadTimeout(5000);
            try (OutputStream os = connection.getOutputStream();){
                byte[] input = jsonPayload.getBytes(StandardCharsets.UTF_8);
                os.write(input, 0, input.length);
            }
            int responseCode = connection.getResponseCode();
            if (responseCode == 429) {
                this.plugin.getLogger().warning("Discord webhook rate limited. Entries will retry.");
            } else if (responseCode < 200 || responseCode >= 300) {
                this.plugin.getLogger().warning("Discord webhook returned error code: " + responseCode);
            }
        }
        finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
    }

    public void shutdown() {
        this.isShuttingDown.set(true);
        if (this.webhookTask != null) {
            this.webhookTask.cancel();
        }
        int flushed = 0;
        while (!this.webhookQueue.isEmpty() && flushed < 10) {
            SpawnerLogEntry entry = this.webhookQueue.poll();
            if (entry == null) continue;
            this.sendWebhook(entry);
            ++flushed;
        }
        if (!this.webhookQueue.isEmpty()) {
            this.plugin.getLogger().warning("Discord webhook queue had " + this.webhookQueue.size() + " pending entries at shutdown (flushed " + flushed + ")");
        }
    }
}

