/*
 * Decompiled with CFR 0.152.
 */
package github.nighter.smartspawner.spawner.gui.synchronization;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.bukkit.ChatColor;
import org.bukkit.inventory.ItemStack;
import org.bukkit.inventory.meta.ItemMeta;

public class ItemUpdater {
    private static final Logger LOGGER = Logger.getLogger("SmartSpawner");
    private static final int MAX_CACHE_SIZE = 512;
    private static final long CACHE_EXPIRY_TIME_MS = TimeUnit.MINUTES.toMillis(30L);
    private static final AtomicInteger cacheHits = new AtomicInteger(0);
    private static final AtomicInteger cacheMisses = new AtomicInteger(0);
    private static final Map<String, CachedColorTranslation> COLOR_TRANSLATION_CACHE = new ConcurrentHashMap<String, CachedColorTranslation>();
    private static long lastCacheCleanup = System.currentTimeMillis();

    public static boolean updateLore(ItemStack item, List<String> newLore) {
        if (item == null || newLore == null) {
            return false;
        }
        try {
            ItemMeta meta = item.getItemMeta();
            if (meta == null) {
                return false;
            }
            List<String> coloredLore = ItemUpdater.colorLoreList(newLore);
            if (meta.hasLore() && ItemUpdater.areLoreListsEqual(meta.getLore(), coloredLore)) {
                return true;
            }
            meta.setLore(coloredLore);
            item.setItemMeta(meta);
            return true;
        }
        catch (Exception e) {
            LOGGER.log(Level.WARNING, "Failed to update lore", e);
            return false;
        }
    }

    public static boolean updateDisplayName(ItemStack item, String displayName) {
        if (item == null || displayName == null) {
            return false;
        }
        try {
            ItemMeta meta = item.getItemMeta();
            if (meta == null) {
                return false;
            }
            String coloredName = ItemUpdater.translateColorCodes(displayName);
            if (meta.hasDisplayName() && meta.getDisplayName().equals(coloredName)) {
                return true;
            }
            meta.setDisplayName(coloredName);
            item.setItemMeta(meta);
            return true;
        }
        catch (Exception e) {
            LOGGER.log(Level.WARNING, "Failed to update display name", e);
            return false;
        }
    }

    public static boolean updateItemMeta(ItemStack item, String displayName, List<String> lore) {
        if (item == null) {
            return false;
        }
        try {
            ItemMeta meta = item.getItemMeta();
            if (meta == null) {
                return false;
            }
            boolean needsUpdate = false;
            if (displayName != null) {
                String coloredName = ItemUpdater.translateColorCodes(displayName);
                if (!meta.hasDisplayName() || !meta.getDisplayName().equals(coloredName)) {
                    meta.setDisplayName(coloredName);
                    needsUpdate = true;
                }
            }
            if (lore != null) {
                List<String> coloredLore = ItemUpdater.colorLoreList(lore);
                if (!meta.hasLore() || !ItemUpdater.areLoreListsEqual(meta.getLore(), coloredLore)) {
                    meta.setLore(coloredLore);
                    needsUpdate = true;
                }
            }
            if (needsUpdate) {
                item.setItemMeta(meta);
            }
            return true;
        }
        catch (Exception e) {
            LOGGER.log(Level.WARNING, "Failed to update item meta", e);
            return false;
        }
    }

    public static boolean updateLoreLine(ItemStack item, int lineIndex, String newLine) {
        if (item == null || newLine == null || lineIndex < 0) {
            return false;
        }
        try {
            ArrayList<String> lore;
            ItemMeta meta = item.getItemMeta();
            if (meta == null) {
                return false;
            }
            ArrayList<String> arrayList = lore = meta.hasLore() ? new ArrayList<String>(meta.getLore()) : new ArrayList();
            while (lore.size() <= lineIndex) {
                lore.add("");
            }
            String coloredLine = ItemUpdater.translateColorCodes(newLine);
            if (((String)lore.get(lineIndex)).equals(coloredLine)) {
                return true;
            }
            lore.set(lineIndex, coloredLine);
            meta.setLore(lore);
            item.setItemMeta(meta);
            return true;
        }
        catch (Exception e) {
            LOGGER.log(Level.WARNING, "Failed to update lore line", e);
            return false;
        }
    }

    public static String translateColorCodes(String text) {
        if (text == null) {
            return null;
        }
        ItemUpdater.cleanupCacheIfNeeded();
        CachedColorTranslation cached = COLOR_TRANSLATION_CACHE.get(text);
        if (cached != null) {
            cached.incrementHits();
            cacheHits.incrementAndGet();
            return cached.translatedValue;
        }
        cacheMisses.incrementAndGet();
        String translated = ChatColor.translateAlternateColorCodes((char)'&', (String)text);
        COLOR_TRANSLATION_CACHE.put(text, new CachedColorTranslation(translated));
        if (COLOR_TRANSLATION_CACHE.size() > 512) {
            ItemUpdater.evictLeastValuableEntries();
        }
        return translated;
    }

    private static void cleanupCacheIfNeeded() {
        long currentTime = System.currentTimeMillis();
        if (currentTime - lastCacheCleanup < TimeUnit.MINUTES.toMillis(5L)) {
            return;
        }
        lastCacheCleanup = currentTime;
        COLOR_TRANSLATION_CACHE.entrySet().removeIf(entry -> currentTime - ((CachedColorTranslation)entry.getValue()).timestamp > CACHE_EXPIRY_TIME_MS);
    }

    private static void evictLeastValuableEntries() {
        int targetSize = 384;
        if (COLOR_TRANSLATION_CACHE.size() <= targetSize) {
            return;
        }
        long currentTime = System.currentTimeMillis();
        ArrayList<Map.Entry<String, CachedColorTranslation>> entries = new ArrayList<Map.Entry<String, CachedColorTranslation>>(COLOR_TRANSLATION_CACHE.entrySet());
        entries.sort((a, b) -> {
            double scoreA = ItemUpdater.calculateEntryValue((CachedColorTranslation)a.getValue(), currentTime);
            double scoreB = ItemUpdater.calculateEntryValue((CachedColorTranslation)b.getValue(), currentTime);
            return Double.compare(scoreA, scoreB);
        });
        int entriesToRemove = COLOR_TRANSLATION_CACHE.size() - targetSize;
        for (int i = 0; i < entriesToRemove && i < entries.size(); ++i) {
            COLOR_TRANSLATION_CACHE.remove(((Map.Entry)entries.get(i)).getKey());
        }
    }

    private static double calculateEntryValue(CachedColorTranslation entry, long currentTime) {
        double ageFactor = 1.0 - Math.min(1.0, (double)(currentTime - entry.timestamp) / (double)CACHE_EXPIRY_TIME_MS);
        int hits = entry.hitCount.get();
        return (double)hits * 0.7 + ageFactor * 0.3;
    }

    private static List<String> colorLoreList(List<String> lore) {
        if (lore == null) {
            return null;
        }
        ArrayList<String> coloredLore = new ArrayList<String>(lore.size());
        for (String line : lore) {
            coloredLore.add(ItemUpdater.translateColorCodes(line));
        }
        return coloredLore;
    }

    private static boolean areLoreListsEqual(List<String> lore1, List<String> lore2) {
        if (lore1 == lore2) {
            return true;
        }
        if (lore1 == null || lore2 == null) {
            return false;
        }
        int size = lore1.size();
        if (size != lore2.size()) {
            return false;
        }
        for (int i = 0; i < size; ++i) {
            String s2;
            String s1 = lore1.get(i);
            if (s1 == (s2 = lore2.get(i))) continue;
            if (s1 == null || s2 == null) {
                return false;
            }
            if (s1.equals(s2)) continue;
            return false;
        }
        return true;
    }

    public static boolean areItemsEqual(ItemStack item1, ItemStack item2) {
        if (item1 == item2) {
            return true;
        }
        if (item1 == null || item2 == null) {
            return false;
        }
        if (item1.getType() != item2.getType()) {
            return false;
        }
        ItemMeta meta1 = item1.getItemMeta();
        ItemMeta meta2 = item2.getItemMeta();
        if (meta1 == null && meta2 == null) {
            return true;
        }
        if (meta1 == null || meta2 == null) {
            return false;
        }
        if (meta1.hasDisplayName() != meta2.hasDisplayName()) {
            return false;
        }
        if (meta1.hasDisplayName() && !meta1.getDisplayName().equals(meta2.getDisplayName())) {
            return false;
        }
        return ItemUpdater.areLoreListsEqual(meta1.getLore(), meta2.getLore());
    }

    public static void clearCache() {
        COLOR_TRANSLATION_CACHE.clear();
        lastCacheCleanup = System.currentTimeMillis();
    }

    public static Map<String, Object> getCacheStats() {
        ConcurrentHashMap<String, Object> stats = new ConcurrentHashMap<String, Object>();
        stats.put("cacheSize", COLOR_TRANSLATION_CACHE.size());
        stats.put("hits", cacheHits.get());
        stats.put("misses", cacheMisses.get());
        stats.put("hitRatio", ItemUpdater.calculateHitRatio());
        stats.put("maxSize", 512);
        stats.put("lastCleanup", lastCacheCleanup);
        return stats;
    }

    private static double calculateHitRatio() {
        int hits = cacheHits.get();
        int total = hits + cacheMisses.get();
        return total > 0 ? (double)hits / (double)total * 100.0 : 0.0;
    }

    public static void resetStats() {
        cacheHits.set(0);
        cacheMisses.set(0);
    }

    private static class CachedColorTranslation {
        final String translatedValue;
        final long timestamp;
        final AtomicInteger hitCount;

        CachedColorTranslation(String translatedValue) {
            this.translatedValue = translatedValue;
            this.timestamp = System.currentTimeMillis();
            this.hitCount = new AtomicInteger(1);
        }

        void incrementHits() {
            this.hitCount.incrementAndGet();
        }
    }
}

