/*
 * Decompiled with CFR 0.152.
 */
package github.nighter.smartspawner.spawner.properties;

import github.nighter.smartspawner.Scheduler;
import github.nighter.smartspawner.SmartSpawner;
import github.nighter.smartspawner.spawner.properties.SpawnerData;
import github.nighter.smartspawner.spawner.utils.SpawnerFileHandler;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Logger;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.World;

public class SpawnerManager {
    private final SmartSpawner plugin;
    private final Map<String, SpawnerData> spawners = new HashMap<String, SpawnerData>();
    private final Map<LocationKey, SpawnerData> locationIndex = new HashMap<LocationKey, SpawnerData>();
    private final Map<String, Set<SpawnerData>> worldIndex = new HashMap<String, Set<SpawnerData>>();
    private final SpawnerFileHandler spawnerFileHandler;
    private final Logger logger;
    private final Set<String> confirmedGhostSpawners = ConcurrentHashMap.newKeySet();

    public SpawnerManager(SmartSpawner plugin) {
        this.plugin = plugin;
        this.logger = plugin.getLogger();
        this.spawnerFileHandler = plugin.getSpawnerFileHandler();
        this.loadSpawnerData();
    }

    public void addSpawner(String id, SpawnerData spawner) {
        this.spawners.put(id, spawner);
        this.locationIndex.put(new LocationKey(spawner.getSpawnerLocation()), spawner);
        String worldName = spawner.getSpawnerLocation().getWorld().getName();
        this.worldIndex.computeIfAbsent(worldName, k -> new HashSet()).add(spawner);
        this.spawnerFileHandler.queueSpawnerForSaving(id);
    }

    public void removeSpawner(String id) {
        SpawnerData spawner = this.spawners.get(id);
        if (spawner != null) {
            Location loc = spawner.getSpawnerLocation();
            Scheduler.runLocationTask(loc, spawner::removeHologram);
            this.locationIndex.remove(new LocationKey(spawner.getSpawnerLocation()));
            String worldName = spawner.getSpawnerLocation().getWorld().getName();
            Set<SpawnerData> worldSpawners = this.worldIndex.get(worldName);
            if (worldSpawners != null) {
                worldSpawners.remove(spawner);
                if (worldSpawners.isEmpty()) {
                    this.worldIndex.remove(worldName);
                }
            }
            this.spawners.remove(id);
        }
    }

    public int countSpawnersInWorld(String worldName) {
        Set<SpawnerData> worldSpawners = this.worldIndex.get(worldName);
        return worldSpawners != null ? worldSpawners.size() : 0;
    }

    public int countTotalSpawnersWithStacks(String worldName) {
        Set<SpawnerData> worldSpawners = this.worldIndex.get(worldName);
        if (worldSpawners == null) {
            return 0;
        }
        return worldSpawners.stream().mapToInt(SpawnerData::getStackSize).sum();
    }

    public SpawnerData getSpawnerByLocation(Location location) {
        return this.locationIndex.get(new LocationKey(location));
    }

    public SpawnerData getSpawnerById(String id) {
        return this.spawners.get(id);
    }

    public List<SpawnerData> getAllSpawners() {
        return new ArrayList<SpawnerData>(this.spawners.values());
    }

    public void loadSpawnerData() {
        this.spawners.clear();
        this.locationIndex.clear();
        this.worldIndex.clear();
        this.confirmedGhostSpawners.clear();
        Map<String, SpawnerData> loadedSpawners = this.spawnerFileHandler.loadAllSpawners();
        for (Map.Entry<String, SpawnerData> entry : loadedSpawners.entrySet()) {
            String spawnerId = entry.getKey();
            SpawnerData spawner = entry.getValue();
            this.spawners.put(spawnerId, spawner);
            this.locationIndex.put(new LocationKey(spawner.getSpawnerLocation()), spawner);
            World world = spawner.getSpawnerLocation().getWorld();
            if (world == null) continue;
            String worldName = world.getName();
            this.worldIndex.computeIfAbsent(worldName, k -> new HashSet()).add(spawner);
        }
        boolean removeOnStartup = this.plugin.getConfig().getBoolean("ghost_spawners.remove_on_startup", true);
        if (removeOnStartup) {
            Scheduler.runTaskLater(this::removeGhostSpawners, 100L);
        }
    }

    public void removeGhostSpawners() {
        if (this.spawners.isEmpty()) {
            return;
        }
        HashMap<String, SpawnerData> spawnersToCheck = new HashMap<String, SpawnerData>(this.spawners);
        AtomicInteger removedCount = new AtomicInteger(0);
        AtomicInteger checkedCount = new AtomicInteger(0);
        int totalToCheck = spawnersToCheck.size();
        ConcurrentHashMap.KeySetView ghostSpawnerIds = ConcurrentHashMap.newKeySet();
        ArrayList checks = new ArrayList();
        for (Map.Entry entry : spawnersToCheck.entrySet()) {
            String spawnerId = (String)entry.getKey();
            SpawnerData spawner = (SpawnerData)entry.getValue();
            Location loc = spawner.getSpawnerLocation();
            if (this.confirmedGhostSpawners.contains(spawnerId)) continue;
            CompletableFuture future = new CompletableFuture();
            checks.add(future);
            Scheduler.runLocationTask(loc, () -> {
                try {
                    boolean isGhost = false;
                    if (loc.getChunk().isLoaded() && loc.getBlock().getType() != Material.SPAWNER) {
                        ghostSpawnerIds.add(spawnerId);
                        this.confirmedGhostSpawners.add(spawnerId);
                        isGhost = true;
                    }
                    checkedCount.incrementAndGet();
                    if (checkedCount.get() % 100 == 0 || checkedCount.get() == totalToCheck) {
                        this.plugin.debug(String.format("Ghost spawner check progress: %d/%d", checkedCount.get(), totalToCheck));
                    }
                    future.complete(null);
                }
                catch (Exception e) {
                    this.plugin.getLogger().warning("Error checking spawner " + spawnerId + ": " + e.getMessage());
                    future.completeExceptionally(e);
                }
            });
        }
        CompletableFuture.allOf(checks.toArray(new CompletableFuture[0])).thenRunAsync(() -> {
            if (!ghostSpawnerIds.isEmpty()) {
                this.plugin.getLogger().info("Found " + ghostSpawnerIds.size() + " ghost spawners");
            }
            ArrayList removals = new ArrayList();
            for (String ghostId : ghostSpawnerIds) {
                SpawnerData spawner = this.spawners.get(ghostId);
                if (spawner != null) {
                    Location loc = spawner.getSpawnerLocation();
                    CompletableFuture removal = new CompletableFuture();
                    removals.add(removal);
                    Scheduler.runLocationTask(loc, () -> {
                        try {
                            spawner.removeHologram();
                            Scheduler.runTask(() -> {
                                this.removeSpawner(ghostId);
                                this.spawnerFileHandler.markSpawnerDeleted(ghostId);
                                removedCount.incrementAndGet();
                                removal.complete(null);
                            });
                        }
                        catch (Exception e) {
                            this.plugin.getLogger().warning("Error removing ghost spawner " + ghostId + ": " + e.getMessage());
                            removal.completeExceptionally(e);
                        }
                    });
                    continue;
                }
                CompletableFuture<Object> removal = new CompletableFuture<Object>();
                removals.add(removal);
                removal.complete(null);
            }
            CompletableFuture.allOf(removals.toArray(new CompletableFuture[0])).thenRunAsync(() -> {
                if (removedCount.get() > 0) {
                    this.spawnerFileHandler.flushChanges();
                    this.plugin.getLogger().info("Successfully removed " + removedCount.get() + " ghost spawners");
                }
            });
        });
    }

    public boolean isGhostSpawner(SpawnerData spawner) {
        if (spawner == null) {
            return false;
        }
        if (this.confirmedGhostSpawners.contains(spawner.getSpawnerId())) {
            return true;
        }
        Location loc = spawner.getSpawnerLocation();
        if (loc == null || loc.getWorld() == null) {
            return true;
        }
        if (!loc.getChunk().isLoaded()) {
            return false;
        }
        return loc.getBlock().getType() != Material.SPAWNER;
    }

    public void removeGhostSpawner(String spawnerId) {
        SpawnerData spawner = this.spawners.get(spawnerId);
        if (spawner != null) {
            Location loc = spawner.getSpawnerLocation();
            this.confirmedGhostSpawners.add(spawnerId);
            Scheduler.runLocationTask(loc, () -> {
                spawner.removeHologram();
                Scheduler.runTask(() -> {
                    this.removeSpawner(spawnerId);
                    this.spawnerFileHandler.markSpawnerDeleted(spawnerId);
                    this.plugin.debug("Removed ghost spawner " + spawnerId);
                });
            });
        }
    }

    public void markSpawnerModified(String spawnerId) {
        this.spawnerFileHandler.markSpawnerModified(spawnerId);
    }

    public void queueSpawnerForSaving(String spawnerId) {
        this.spawnerFileHandler.queueSpawnerForSaving(spawnerId);
    }

    public void refreshAllHolograms() {
        for (SpawnerData spawner : this.spawners.values()) {
            Location loc = spawner.getSpawnerLocation();
            Scheduler.runLocationTask(loc, spawner::refreshHologram);
        }
    }

    public void reloadAllHolograms() {
        if (this.plugin.getConfig().getBoolean("hologram.enabled", false)) {
            for (SpawnerData spawner : this.spawners.values()) {
                Location loc = spawner.getSpawnerLocation();
                Scheduler.runLocationTask(loc, spawner::reloadHologramData);
            }
        }
    }

    public void cleanupAllSpawners() {
        this.spawners.clear();
        this.locationIndex.clear();
        this.worldIndex.clear();
        this.confirmedGhostSpawners.clear();
    }

    public int getTotalSpawners() {
        return this.spawners.size();
    }

    private static class LocationKey {
        private final String world;
        private final int x;
        private final int y;
        private final int z;

        public LocationKey(Location location) {
            this.world = location.getWorld().getName();
            this.x = location.getBlockX();
            this.y = location.getBlockY();
            this.z = location.getBlockZ();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof LocationKey)) {
                return false;
            }
            LocationKey that = (LocationKey)o;
            return this.x == that.x && this.y == that.y && this.z == that.z && this.world.equals(that.world);
        }

        public int hashCode() {
            return Objects.hash(this.world, this.x, this.y, this.z);
        }
    }
}

