package me.basiqueevangelist.windowapi;

import com.mojang.blaze3d.systems.RenderSystem;
import me.basiqueevangelist.windowapi.context.CurrentWindowContext;
import me.basiqueevangelist.windowapi.context.WindowContext;
import me.basiqueevangelist.windowapi.util.GlUtil;
import net.fabricmc.fabric.api.event.Event;
import net.minecraft.class_1008;
import net.minecraft.class_1011;
import net.minecraft.class_276;
import net.minecraft.class_289;
import net.minecraft.class_308;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3545;
import net.minecraft.class_364;
import net.minecraft.class_3675;
import net.minecraft.class_4068;
import net.minecraft.class_6367;
import net.minecraft.class_8251;
import org.joml.Matrix4f;
import org.joml.Matrix4fStack;
import org.lwjgl.glfw.*;
import org.lwjgl.opengl.GL32;
import org.lwjgl.system.MemoryStack;
import org.lwjgl.system.MemoryUtil;
import org.lwjgl.system.NativeResource;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;

import static org.lwjgl.glfw.GLFW.*;
import static org.lwjgl.glfw.GLFW.glfwSetCharModsCallback;

public abstract class AltWindow extends SupportsFeaturesImpl<WindowContext> implements WindowContext, class_4068, class_364 {
    private static final boolean USE_GLOBAL_POS = glfwGetPlatform() != GLFW_PLATFORM_WAYLAND;

    private String title = "window api window";
    private int screenWidth = 854;
    private int screenHeight = 480;
    private WindowIcon icon = null;
    private final List<class_3545<Integer, Integer>> windowHints = new ArrayList<>();

    private int framebufferWidth;
    private int framebufferHeight;

    private long handle = 0;
    private class_276 framebuffer;
    private int localFramebuffer = 0;
    private final List<NativeResource> disposeList = new ArrayList<>();
    private final class_310 client = class_310.method_1551();

    private int scaleFactor;
    private int scaledWidth;
    private int scaledHeight;

    private int mouseX = -1;
    private int mouseY = -1;
    private int globalMouseX = -1;
    private int globalMouseY = -1;
    private int deltaX = 0;
    private int deltaY = 0;
    private int activeButton = -1;

    private boolean cursorLocked = false;

    private final int[] globalX = new int[1];
    private final int[] globalY = new int[1];

    private final Event<WindowFramebufferResized> framebufferResizedEvents = WindowFramebufferResized.newEvent();

    public AltWindow() {

    }

    //region User-implementable stuff
    protected abstract void build();

    protected abstract void resize(int newWidth, int newHeight);

    protected void lockedMouseMoved(double xDelta, double yDelta) { }
    //endregion

    public AltWindow size(int screenWidth, int screenHeight) {
        this.screenWidth = screenWidth;
        this.screenHeight = screenHeight;

        if (this.handle != 0) {
            glfwSetWindowSize(this.handle, screenWidth, screenHeight);
        }

        return this;
    }

    public AltWindow title(String title) {
        this.title = title;

        if (this.handle != 0) {
            glfwSetWindowTitle(this.handle, title);
        }

        return this;
    }

    public AltWindow icon(WindowIcon icon) {
        this.icon = icon;

        if (this.handle != 0) {
            applyIcon();
        }

        return this;
    }

    public AltWindow windowHint(int hint, int value) {
        if (this.handle != 0) {
            throw new IllegalStateException("Tried to add window hint after window was opened");
        }

        windowHints.add(new class_3545<>(hint, value));

        return this;
    }

    public void open() {
        try (var ignored = GlUtil.setContext(0)) {
            glfwDefaultWindowHints();
            glfwWindowHint(GLFW_CLIENT_API, GLFW_OPENGL_API);
            glfwWindowHint(GLFW_CONTEXT_CREATION_API, GLFW_NATIVE_CONTEXT_API);
            glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
            glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 2);
            glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);
            glfwWindowHint(GLFW_OPENGL_FORWARD_COMPAT, 1);

            for (var hint : windowHints) {
                glfwWindowHint(hint.method_15442(), hint.method_15441());
            }

            this.handle = glfwCreateWindow(this.screenWidth, this.screenHeight, this.title, 0, class_310.method_1551().method_22683().method_4490());

            if (this.handle == 0) {
                throw new IllegalStateException("OwoWindow creation failed due to GLFW error");
            }

            glfwMakeContextCurrent(this.handle);
            glfwSwapInterval(0);
        }

        applyIcon();

        int[] framebufferWidthArr = new int[1];
        int[] framebufferHeightArr = new int[1];
        glfwGetFramebufferSize(this.handle, framebufferWidthArr, framebufferHeightArr);
        this.framebufferWidth = framebufferWidthArr[0];
        this.framebufferHeight = framebufferHeightArr[0];

        this.framebuffer = new class_6367(this.framebufferWidth, this.framebufferHeight, true, class_310.field_1703);

        try (var ignored = GlUtil.setContext(this.handle)) {
            class_1008.method_4227(client.field_1690.field_1901, true);
        }

        initLocalFramebuffer();

        glfwSetWindowCloseCallback(handle, stowAndReturn(GLFWWindowCloseCallback.create(window -> {
            try (var ignored = CurrentWindowContext.setCurrent(this)) {
                this.close();
            }
        })));

        glfwSetWindowSizeCallback(handle, stowAndReturn(GLFWWindowSizeCallback.create((window, width, height) -> {
            this.screenWidth = width;
            this.screenHeight = height;
        })));

        glfwSetFramebufferSizeCallback(handle, stowAndReturn(GLFWFramebufferSizeCallback.create((window, width, height) -> {
            if (this.framebufferWidth == width && this.framebufferHeight == height) return;

            if (width == 0 || height == 0) return;

            this.framebufferWidth = width;
            this.framebufferHeight = height;

            try (var ignored = GlUtil.setContext(client.method_22683().method_4490())) {
                framebuffer.method_1238();

                this.framebuffer = new class_6367(width, height, true, class_310.field_1703);
            }

            initLocalFramebuffer();

            recalculateScale();

            try (var ignored = CurrentWindowContext.setCurrent(this)) {
                this.resize(scaledWidth(), scaledHeight());

                this.framebufferResizedEvents.invoker().onFramebufferResized(width, height);
            }
        })));

        glfwSetCursorPosCallback(handle, stowAndReturn(GLFWCursorPosCallback.create((window, xpos, ypos) -> {
            if (cursorLocked) {
                this.lockedMouseMoved(xpos - mouseX * scaleFactor, ypos - mouseY * scaleFactor);
                GLFW.glfwSetCursorPos(handle(), mouseX * scaleFactor, mouseY * scaleFactor);
                return;
            }


            int newX = (int) (xpos / scaleFactor);
            int newY = (int) (ypos / scaleFactor);

            if (!USE_GLOBAL_POS) {
                this.deltaX += newX - mouseX;
                this.deltaY += newY - mouseY;
            }

            this.mouseY = newY;
            this.mouseX = newX;

            if (USE_GLOBAL_POS) {
                glfwGetWindowPos(handle, this.globalX, this.globalY);
                int newGlobalX = (int) ((this.globalX[0] + xpos) / scaleFactor);
                int newGlobalY = (int) ((this.globalY[0] + ypos) / scaleFactor);

                this.deltaX += newGlobalX - this.globalMouseX;
                this.deltaY += newGlobalY - this.globalMouseY;

                this.globalMouseX = newGlobalX;
                this.globalMouseY = newGlobalY;
            }
        })));

        glfwSetMouseButtonCallback(handle, stowAndReturn(GLFWMouseButtonCallback.create((window, button, action, mods) -> {
            try (var ignored = CurrentWindowContext.setCurrent(this)) {
                if (action == GLFW_RELEASE) {
                    this.activeButton = -1;

                    this.method_25406(mouseX, mouseY, button);
                } else {
                    this.activeButton = button;

                    this.method_25402(mouseX, mouseY, button);
                }
            }
        })));

        glfwSetScrollCallback(handle, stowAndReturn(GLFWScrollCallback.create((window, xoffset, yoffset) -> {
            try (var ignored = CurrentWindowContext.setCurrent(this)) {
                double yAmount = (client.field_1690.method_42439().method_41753() ? Math.signum(yoffset) : yoffset)
                    * client.field_1690.method_41806().method_41753();
                double xAmount = (client.field_1690.method_42439().method_41753() ? Math.signum(xoffset) : xoffset)
                    * client.field_1690.method_41806().method_41753();
                this.method_25401(mouseX, mouseY, xAmount, yAmount);
            }
        })));

        glfwSetKeyCallback(handle, stowAndReturn(GLFWKeyCallback.create((window, key, scancode, action, mods) -> {
            try (var ignored = CurrentWindowContext.setCurrent(this)) {
                if (action == GLFW_RELEASE) {
                    this.method_16803(key, scancode, mods);
                } else {
                    this.method_25404(key, scancode, mods);
                }
            }
        })));

        glfwSetCharModsCallback(handle, stowAndReturn(GLFWCharModsCallback.create((window, codepoint, mods) -> {
            try (var ignored = CurrentWindowContext.setCurrent(this)) {
                this.method_25400((char) codepoint, mods);
            }
        })));

        recalculateScale();

        try (var ignored = CurrentWindowContext.setCurrent(this)) {
            build();
        }

        OpenWindows.add(this);
    }

    public boolean cursorLocked() {
        return cursorLocked;
    }

    public void lockCursor() {
        if (cursorLocked) return;

        this.cursorLocked = true;
        this.mouseX = scaledWidth / 2;
        this.mouseY = scaledHeight / 2;
        class_3675.method_15984(handle, class_3675.field_32005, this.mouseX * scaleFactor, this.mouseY * scaleFactor);
    }

    public void unlockCursor() {
        if (!cursorLocked) return;

        this.mouseX = scaledWidth / 2;
        this.mouseY = scaledHeight / 2;
        class_3675.method_15984(handle, class_3675.field_32006, this.mouseX * scaleFactor, this.mouseY * scaleFactor);
        this.cursorLocked = false;
    }

    private <T extends NativeResource> T stowAndReturn(T resource) {
        this.disposeList.add(resource);
        return resource;
    }

    private void applyIcon() {
        if (icon == null) return;

        List<class_1011> icons = icon.listIconImages();

        List<ByteBuffer> freeList = new ArrayList<>(icons.size());
        try (MemoryStack memoryStack = MemoryStack.stackPush()) {
            GLFWImage.Buffer buffer = GLFWImage.malloc(icons.size(), memoryStack);

            for (int i = 0; i < icons.size(); i++) {
                class_1011 icon = icons.get(i);
                ByteBuffer imgBuffer = MemoryUtil.memAlloc(icon.method_4307() * icon.method_4323() * 4);
                freeList.add(imgBuffer);
                imgBuffer.asIntBuffer().put(icon.method_48463());

                buffer
                    .position(i)
                    .width(icon.method_4307())
                    .height(icon.method_4323())
                    .pixels(imgBuffer);
            }

            GLFW.glfwSetWindowIcon(this.handle, buffer.position(0));
        } finally {
            freeList.forEach(MemoryUtil::memFree);

            if (icon.closeAfterUse())
                icons.forEach(class_1011::close);
        }
    }

    private void initLocalFramebuffer() {
        try (var ignored = GlUtil.setContext(this.handle)) {
            if (localFramebuffer != 0) {
                GL32.glDeleteFramebuffers(localFramebuffer);
            }

            this.localFramebuffer = GL32.glGenFramebuffers();
            GL32.glBindFramebuffer(GL32.GL_FRAMEBUFFER, this.localFramebuffer);
            GL32.glFramebufferTexture2D(GL32.GL_FRAMEBUFFER, GL32.GL_COLOR_ATTACHMENT0, GL32.GL_TEXTURE_2D, this.framebuffer.method_30277(), 0);

            int status = GL32.glCheckFramebufferStatus(GL32.GL_FRAMEBUFFER);
            if (status != GL32.GL_FRAMEBUFFER_COMPLETE)
                throw new IllegalStateException("Failed to create local framebuffer!");
        }
    }

    public void recalculateScale() {
        int guiScale = class_310.method_1551().field_1690.method_42474().method_41753();
        boolean forceUnicodeFont = class_310.method_1551().field_1690.method_42437().method_41753();

        int factor = 1;

        while (
            factor != guiScale
                && factor < this.framebufferWidth()
                && factor < this.framebufferHeight()
                && this.framebufferWidth() / (factor + 1) >= 320
                && this.framebufferHeight() / (factor + 1) >= 240
        ) {
            ++factor;
        }

        if (forceUnicodeFont && factor % 2 != 0) {
            ++factor;
        }

        this.scaleFactor = factor;
        this.scaledWidth = (int) Math.ceil((double) this.framebufferWidth() / scaleFactor);
        this.scaledHeight = (int) Math.ceil((double) this.framebufferHeight() / scaleFactor);
    }

    private void tickMouse() {
        if (deltaX == 0 && this.deltaY == 0) return;

        this.method_16014(mouseX, mouseY);

        if (activeButton != -1) this.method_25403(mouseX, mouseY, activeButton, deltaX, deltaY);

        deltaX = 0;
        deltaY = 0;
    }

    public void draw() {
        if (closed()) return;

        try (var ignored = CurrentWindowContext.setCurrent(this)) {
            tickMouse();

            framebuffer().method_1235(true);

            RenderSystem.clearColor(0, 0, 0, 0);
            RenderSystem.clear(GL32.GL_COLOR_BUFFER_BIT | GL32.GL_DEPTH_BUFFER_BIT, class_310.field_1703);

            Matrix4f matrix4f = new Matrix4f()
                .setOrtho(
                    0.0F,
                    scaledWidth(),
                    scaledHeight(),
                    0.0F,
                    1000.0F,
                    21000.0F
                );

            try (var ignored2 = GlUtil.setProjectionMatrix(matrix4f, class_8251.field_43361)) {
                Matrix4fStack matrixStack = RenderSystem.getModelViewStack();
                matrixStack.pushMatrix();
                matrixStack.identity();
                matrixStack.translate(0.0F, 0.0F, -11000.0F);
                RenderSystem.applyModelViewMatrix();
                class_308.method_24211();

                var consumers = client.method_22940().method_23000();
                this.method_25394(new class_332(client, consumers), mouseX, mouseY, client.method_60646().method_60637(false));
                consumers.method_22993();

                RenderSystem.getModelViewStack().popMatrix();
                RenderSystem.applyModelViewMatrix();
            }

            framebuffer.method_1240();
        }
    }

    void present() {
        if (closed()) return;

        GLFW.glfwMakeContextCurrent(this.handle);
        // This code intentionally doesn't use Minecraft's RenderSystem
        // class, as it caches GL state that is invalid on this context.
        GL32.glBindFramebuffer(GL32.GL_READ_FRAMEBUFFER, localFramebuffer);
        GL32.glBindFramebuffer(GL32.GL_DRAW_FRAMEBUFFER, 0);

        GL32.glClearColor(1, 1, 1, 1);
        GL32.glClear(GL32.GL_COLOR_BUFFER_BIT | GL32.GL_DEPTH_BUFFER_BIT);
        GL32.glBlitFramebuffer(0, 0, this.framebufferWidth, this.framebufferHeight, 0, 0, this.framebufferWidth, this.framebufferHeight, GL32.GL_COLOR_BUFFER_BIT, GL32.GL_NEAREST);

        // Intentionally doesn't poll events so that all events are on the main window
        class_289.method_1348().method_60828();
        GLFW.glfwSwapBuffers(this.handle);
    }

    @Override
    public long handle() {
        return handle;
    }

    @Override
    public Event<WindowFramebufferResized> framebufferResized() {
        return framebufferResizedEvents;
    }

    @Override
    public class_276 framebuffer() {
        return framebuffer;
    }

    @Override
    public int framebufferWidth() {
        return framebufferWidth;
    }

    @Override
    public int framebufferHeight() {
        return framebufferHeight;
    }

    @Override
    public double scaleFactor() {
        return scaleFactor;
    }

    @Override
    public int scaledWidth() {
        return scaledWidth;
    }

    @Override
    public int scaledHeight() {
        return scaledHeight;
    }

    public boolean closed() {
        return this.handle == 0;
    }

    public void close() {
        this.destroyFeatures();
        OpenWindows.remove(this);

        try (var ignored = GlUtil.setContext(this.handle)) {
            GL32.glDeleteFramebuffers(this.localFramebuffer);
        }

        this.framebuffer.method_1238();
        glfwDestroyWindow(this.handle);
        this.handle = 0;

        this.disposeList.forEach(NativeResource::free);
        this.disposeList.clear();
    }
}