package tools.redstone.redstonetools.features.commands;

import com.mojang.brigadier.Command;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.arguments.BoolArgumentType;
import com.mojang.brigadier.arguments.IntegerArgumentType;
import com.mojang.brigadier.arguments.StringArgumentType;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.SimpleCommandExceptionType;
import com.sk89q.worldedit.IncompleteRegionException;
import com.sk89q.worldedit.WorldEdit;
import com.sk89q.worldedit.WorldEditException;
import com.sk89q.worldedit.fabric.FabricAdapter;
import com.sk89q.worldedit.function.mask.Mask;
import com.sk89q.worldedit.function.mask.Mask2D;
import com.sk89q.worldedit.function.operation.ForwardExtentCopy;
import com.sk89q.worldedit.function.operation.Operations;
import com.sk89q.worldedit.math.BlockVector3;
import com.sk89q.worldedit.regions.Region;
import com.sk89q.worldedit.util.Direction;
import org.jetbrains.annotations.Nullable;
import tools.redstone.redstonetools.utils.ArgumentUtils;
import tools.redstone.redstonetools.utils.DirectionArgument;

import java.util.Objects;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2561;
import net.minecraft.class_7157;

import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;
import static tools.redstone.redstonetools.utils.DirectionUtils.directionToBlock;
import static tools.redstone.redstonetools.utils.DirectionUtils.matchDirection;

public class RStackFeature {
	public static final RStackFeature INSTANCE = new RStackFeature();

	protected RStackFeature() {
	}

	public void registerCommand(CommandDispatcher<class_2168> dispatcher, class_7157 registryAccess, class_2170.class_5364 registrationEnvironment) {
			dispatcher.register(
				method_9247("/rstack")
					.requires(source -> source.method_9259(2))
					.executes(getCommandForArgumentCount(0))
					.then(method_9244("count", IntegerArgumentType.integer())
						.executes(getCommandForArgumentCount(1))
						.then(method_9244("direction", StringArgumentType.string()).suggests(ArgumentUtils.DIRECTION_SUGGESTION_PROVIDER)
							.executes(getCommandForArgumentCount(2))
							.then(method_9244("offset", IntegerArgumentType.integer())
								.executes(getCommandForArgumentCount(3))
								.then(method_9244("moveSelection", BoolArgumentType.bool())
									.executes(getCommandForArgumentCount(4)))))));
	}

	protected Command<class_2168> getCommandForArgumentCount(int argNum) {
		return context -> execute(context, argNum);
	}

	protected int execute(CommandContext<class_2168> context, int argCount) throws CommandSyntaxException {
		int count = argCount >= 1 ? IntegerArgumentType.getInteger(context, "count") : 1;
		DirectionArgument direction = argCount >= 2 ? ArgumentUtils.parseDirection(context, "direction") : DirectionArgument.ME;
		int offset = argCount >= 3 ? IntegerArgumentType.getInteger(context, "offset") : 2;
		boolean moveSelection = argCount >= 4 && BoolArgumentType.getBool(context, "moveSelection");
		return execute(context, count, offset, direction, moveSelection);
	}

	protected int execute(CommandContext<class_2168> context, int count, int offset, DirectionArgument direction, boolean moveSelection) throws CommandSyntaxException {
		var actor = FabricAdapter.adaptPlayer(Objects.requireNonNull(context.getSource().method_44023()));

		var localSession = WorldEdit.getInstance()
				.getSessionManager()
				.get(actor);

		final var selectionWorld = localSession.getSelectionWorld();
		assert selectionWorld != null;

		final Region selection;
		try {
			selection = localSession.getSelection(selectionWorld);
		} catch (IncompleteRegionException ex) {
			throw new SimpleCommandExceptionType(class_2561.method_43470("Please make a selection with WorldEdit first.")).create();
		}

		final Mask airFilter = new Mask() {
			@Override
			public boolean test(BlockVector3 vector) {
				return !"minecraft:air".equals(selectionWorld.getBlock(vector).getBlockType().id());
			}

			@Nullable
			@Override
			public Mask2D toMask2D() {
				return null;
			}
		};

		var playerFacing = actor.getLocation().getDirectionEnum();
		Direction stackDirection;
		try {
			stackDirection = matchDirection(direction, playerFacing);
		} catch (Exception e) {
			throw new SimpleCommandExceptionType(class_2561.method_43470(e.getMessage().formatted(e))).create();
		}
		var stackVector = directionToBlock(stackDirection);


		try (var editSession = localSession.createEditSession(actor)) {
			for (var i = 1; i <= count; i++) {
				BlockVector3 offsetVector = Objects.requireNonNull(stackVector).multiply(i * offset);
				var copy = new ForwardExtentCopy(
						editSession,
						selection,
						editSession,
						selection.getMinimumPoint().add(offsetVector)
				);
				copy.setSourceMask(airFilter);
				copy.setSourceFunction(position -> false);
				Operations.complete(copy);
				if (i == count && moveSelection) {
					selection.shift(offsetVector);
				}
			}
			localSession.remember(editSession);
		} catch (WorldEditException e) {
			throw new RuntimeException(e);
		}

		context.getSource().method_45068(class_2561.method_43470("Stacked %s time(s).".formatted(count)));
		return 1;
	}
}
