package com.github.thedeathlycow.frostiful.block;

import com.github.thedeathlycow.frostiful.Frostiful;
import com.github.thedeathlycow.frostiful.config.FrostifulConfig;
import com.github.thedeathlycow.frostiful.entity.damage.FDamageSources;
import com.github.thedeathlycow.frostiful.mixins.entity.FallingBlockEntityAccessor;
import com.github.thedeathlycow.frostiful.registry.FBlocks;
import com.github.thedeathlycow.frostiful.registry.tag.FBlockTags;
import com.github.thedeathlycow.thermoo.api.temperature.HeatingModes;
import net.minecraft.block.*;
import net.minecraft.class_10225;
import net.minecraft.class_1282;
import net.minecraft.class_1297;
import net.minecraft.class_1309;
import net.minecraft.class_1540;
import net.minecraft.class_1676;
import net.minecraft.class_1750;
import net.minecraft.class_1922;
import net.minecraft.class_1936;
import net.minecraft.class_1937;
import net.minecraft.class_1959;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2398;
import net.minecraft.class_243;
import net.minecraft.class_259;
import net.minecraft.class_265;
import net.minecraft.class_2680;
import net.minecraft.class_2689;
import net.minecraft.class_2741;
import net.minecraft.class_2746;
import net.minecraft.class_2754;
import net.minecraft.class_3218;
import net.minecraft.class_3610;
import net.minecraft.class_3612;
import net.minecraft.class_3726;
import net.minecraft.class_3737;
import net.minecraft.class_3965;
import net.minecraft.class_4538;
import net.minecraft.class_5688;
import net.minecraft.class_5689;
import net.minecraft.class_5691;
import net.minecraft.class_5819;
import net.minecraft.class_6017;
import net.minecraft.class_6019;
import org.jetbrains.annotations.Nullable;

import java.util.Optional;
import java.util.function.BiPredicate;
import java.util.function.Predicate;

/**
 * Icicles are growths of ice that sometimes will fall and
 * damage players.
 * Largely based on code from {@link class_5689}.
 */
public class IcicleBlock extends class_2248 implements class_5688, class_3737 {

    /**
     * Icicles can point up and down
     */
    public static final class_2754<class_2350> VERTICAL_DIRECTION = class_2741.field_28062;
    /**
     * Icicles have varying levels of thickness
     */
    public static final class_2754<class_5691> THICKNESS = class_2741.field_28063;
    /**
     * Icicles can be waterlogged
     */
    public static final class_2746 WATERLOGGED = class_2741.field_12508;
    /**
     * An unstable icicle is about to fall
     */
    public static final class_2746 UNSTABLE = class_2741.field_12539;

    private static final class_265 TIP_MERGE_SHAPE = class_2248.method_9541(5.0D, 0.0D, 5.0D, 11.0D, 16.0D, 11.0D);
    private static final class_265 UP_TIP_SHAPE = class_2248.method_9541(5.0D, 0.0D, 5.0D, 11.0D, 11.0D, 11.0D);
    private static final class_265 DOWN_TIP_SHAPE = class_2248.method_9541(5.0D, 5.0D, 5.0D, 11.0D, 16.0D, 11.0D);
    private static final class_265 BASE_SHAPE = class_2248.method_9541(4.0D, 0.0D, 4.0D, 12.0D, 16.0D, 12.0D);
    private static final class_265 FRUSTUM_SHAPE = class_2248.method_9541(3.0D, 0.0D, 3.0D, 13.0D, 16.0D, 13.0D);
    private static final class_265 MIDDLE_SHAPE = class_2248.method_9541(2.0D, 0.0D, 2.0D, 14.0D, 16.0D, 14.0D);

    private static final class_6017 UNSTABLE_TICKS_BEFORE_FALL = class_6019.method_35017(40, 80);

    public IcicleBlock(class_2251 settings) {
        super(settings);
        this.method_9590(this.field_10647.method_11664()
                .method_11657(VERTICAL_DIRECTION, class_2350.field_11036)
                .method_11657(THICKNESS, class_5691.field_28065)
                .method_11657(WATERLOGGED, false)
                .method_11657(UNSTABLE, false)
        );
    }

    @Override
    protected void method_9515(class_2689.class_2690<class_2248, class_2680> builder) {
        builder.method_11667(VERTICAL_DIRECTION, THICKNESS, WATERLOGGED, UNSTABLE);
    }

    /**
     * Determine if the state can legally be placed as an icicle
     *
     * @param state State of the icicle to test
     * @param world World the icicle is in
     * @param pos   The position of the icicle
     * @return Returns true if the icicle can be placed, false otherwise
     */
    @Override
    protected boolean method_9558(class_2680 state, class_4538 world, class_2338 pos) {
        return canPlaceAtWithDirection(world, pos, state.method_11654(VERTICAL_DIRECTION));
    }

    /**
     * Icicles collapse when hit by any projectile
     *
     * @param world      The world the icicle is in
     * @param state      The state of the icicle
     * @param hit        How the projectile hit the icicle
     * @param projectile The projectile that hit the icicle
     */
    @Override
    protected void method_19286(class_1937 world, class_2680 state, class_3965 hit, class_1676 projectile) {
        class_2338 blockPos = hit.method_17777();
        if (world instanceof class_3218 serverWorld && projectile.method_36971(serverWorld, blockPos) && projectile.method_18798().method_1033() > 0.6D) {
            world.method_22352(blockPos, true);
        }
    }

    /**
     * Hurt entities when they fall on icicles
     *
     * @param world        The world the icicle is in
     * @param state        The state of the icicle
     * @param pos          The position of the icicle
     * @param entity       The entity that fell
     * @param fallDistance How far the entity fell
     */
    @Override
    public void method_9554(class_1937 world, class_2680 state, class_2338 pos, class_1297 entity, double fallDistance) {
        if (state.method_11654(VERTICAL_DIRECTION) == class_2350.field_11036) {
            class_1282 damageSource = FDamageSources.getDamageSources(world).frostiful$icicle();
            boolean tookDamage = entity.method_5747(fallDistance + 2.0, 2.0f, damageSource);
            if (tookDamage && entity instanceof class_1309 livingEntity) {
                FrostifulConfig config = Frostiful.getConfig();
                livingEntity.thermoo$addTemperature(
                        -config.icicleConfig.getIcicleCollisionFreezeAmount(),
                        HeatingModes.ACTIVE
                );
            }
        } else {
            super.method_9554(world, state, pos, entity, fallDistance);
        }
    }

    /**
     * Get the state of an icicle to be placed
     *
     * @param ctx Context for placement
     * @return Returns the {@link class_2680} to be placed
     */
    @Override
    @Nullable
    public class_2680 method_9605(class_1750 ctx) {
        class_1936 worldAccess = ctx.method_8045();
        class_2338 blockPos = ctx.method_8037();
        class_2350 lookingDirection = ctx.method_32760().method_10153();
        class_2350 direction = getDirectionToPlaceAt(worldAccess, blockPos, lookingDirection);
        if (direction == null) {
            return null;
        }

        class_5691 thickness = getThickness(worldAccess, blockPos, direction, !ctx.method_8046());

        boolean unstable = false;
        if (direction == class_2350.field_11033) {
            class_2680 state = worldAccess.method_8320(blockPos.method_10084());
            unstable = isUnstable(state) && isHeldByIcicle(state, worldAccess, blockPos);
        }
        return thickness == null ? null : this.method_9564()
                .method_11657(VERTICAL_DIRECTION, direction)
                .method_11657(THICKNESS, thickness)
                .method_11657(WATERLOGGED, worldAccess.method_8316(blockPos).method_15772() == class_3612.field_15910)
                .method_11657(UNSTABLE, unstable);
    }

    /**
     * Gets an updated block state to change the icicle to when a neighbour is updated.
     * Handles fluid ticking when water logged.
     * Schedules the icicle to be broken if its support has been broken or if its support
     * is unstable.
     *
     * @param state         the state of this block
     * @param direction     the direction from this block to the neighbor
     * @param neighborState the state of the updated neighbor block
     * @param world         the world
     * @param pos           the position of this block
     * @param neighborPos   the position of the neighbor block
     * @return Returns the updated {@link class_2680} of the icicle
     */
    @Override
    protected class_2680 method_9559(
            class_2680 state,
            class_4538 world,
            class_10225 tickView,
            class_2338 pos,
            class_2350 direction,
            class_2338 neighborPos,
            class_2680 neighborState,
            class_5819 random
    ) {
        if (state.method_11654(WATERLOGGED)) {
            tickView.method_64312(pos, class_3612.field_15910, class_3612.field_15910.method_15789(world));
        }

        if (direction != class_2350.field_11036 && direction != class_2350.field_11033) {
            return state;
        } else {
            class_2350 pointingIn = state.method_11654(VERTICAL_DIRECTION);
            if (pointingIn == class_2350.field_11033 && tickView.method_8397().method_8674(pos, this)) {
                return state;
            } else if (direction == pointingIn.method_10153() && !this.method_9558(state, world, pos)) {
                if (pointingIn == class_2350.field_11033) {
                    tickView.method_64310(pos, this, 2);
                } else {
                    tickView.method_64310(pos, this, 1);
                }

                return state;
            } else {
                boolean tryMerge = state.method_11654(THICKNESS) == class_5691.field_28064;
                class_5691 thickness = getThickness(world, pos, pointingIn, tryMerge);

                boolean makeUnstable = isUnstable(state)
                        || (isHeldByIcicle(state, world, pos)
                        && isUnstable(neighborState)
                        && direction == class_2350.field_11036);

                return state.method_11657(THICKNESS, thickness).method_11657(UNSTABLE, makeUnstable);
            }
        }
    }

    @Override
    protected void method_9588(class_2680 state, class_3218 world, class_2338 pos, class_5819 random) {
        if (isPointingUp(state) && !this.method_9558(state, world, pos)) {
            world.method_22352(pos, true);
        } else {
            spawnFallingBlock(state, world, pos);
        }
    }

    @Override
    protected void method_9514(class_2680 state, class_3218 world, class_2338 pos, class_5819 random) {
        if (isPointingDown(state)) {
            FrostifulConfig config = Frostiful.getConfig();
            if (random.method_43057() < config.icicleConfig.getBecomeUnstableChance() && isHeldByIcicleFallable(state, world, pos)) { // fall
                this.tryFall(state, world, pos, random);
            }
            final double growChance = this.getGrowChance(world);
            if (random.method_43057() < growChance) { // grow
                this.tryGrowIcicle(state, world, pos, random);
            }
        }
    }

    @Override
    public void method_9496(class_2680 state, class_1937 world, class_2338 pos, class_5819 random) {
        if (isUnstable(state)) {
            createUnstableParticle(world, pos, state);
        }
    }

    @Override
    public class_1282 method_32898(class_1297 attacker) {
        return FDamageSources.getDamageSources(attacker.method_73183()).frostiful$fallingIcicle(attacker);
    }

    @Override
    protected class_3610 method_9545(class_2680 state) {
        return Boolean.TRUE.equals(state.method_11654(WATERLOGGED))
                ? class_3612.field_15910.method_15729(false)
                : super.method_9545(state);
    }

    @Override
    protected class_265 method_9571(class_2680 state) {
        return class_259.method_1073();
    }

    @Override
    protected class_265 method_9530(class_2680 state, class_1922 world, class_2338 pos, class_3726 context) {
        class_5691 thickness = state.method_11654(THICKNESS);
        class_265 voxelShape;
        if (thickness == class_5691.field_28064) {
            voxelShape = TIP_MERGE_SHAPE;
        } else if (thickness == class_5691.field_28065) {
            if (state.method_11654(VERTICAL_DIRECTION) == class_2350.field_11033) {
                voxelShape = DOWN_TIP_SHAPE;
            } else {
                voxelShape = UP_TIP_SHAPE;
            }
        } else if (thickness == class_5691.field_28066) {
            voxelShape = BASE_SHAPE;
        } else if (thickness == class_5691.field_28067) {
            voxelShape = FRUSTUM_SHAPE;
        } else {
            voxelShape = MIDDLE_SHAPE;
        }

        class_243 vec3d = state.method_26226(pos);
        return voxelShape.method_1096(vec3d.field_1352, 0.0D, vec3d.field_1350);
    }

    @Override
    protected boolean method_37403(class_2680 state, class_1922 world, class_2338 pos) {
        return false;
    }

    @Override
    protected float method_32913() {
        return 0.125F;
    }

    private Double getGrowChance(class_3218 world) {
        FrostifulConfig config = Frostiful.getConfig();
        if (world.method_8546()) {
            return config.icicleConfig.getGrowChanceDuringThunder();
        } else if (world.method_8419()) {
            return config.icicleConfig.getGrowChanceDuringRain();
        } else {
            return config.icicleConfig.getGrowChance();
        }
    }

    private void tryFall(class_2680 state, class_3218 world, class_2338 pos, class_5819 random) {

        if (isUnstable(state) || isPointingUp(state)) {
            return;
        }

        class_2338 tipPos = getTipPos(state, world, pos, 25, false);
        if (tipPos != null) {
            world.method_8501(pos, state.method_11657(UNSTABLE, true));
            world.method_64310(pos, this, UNSTABLE_TICKS_BEFORE_FALL.method_35008(random));
        }
    }

    private void tryGrowIcicle(class_2680 state, class_3218 world, class_2338 pos, class_5819 random) {

        if (isUnstable(state) || isPointingUp(state)) {
            return;
        }

        class_2338 anchorPos = pos.method_10084();
        class_2680 anchor = world.method_8320(anchorPos);
        if (canGrowIcicleOnAnchor(world, anchorPos, anchor)) {
            class_2338 tipPos = getTipPos(state, world, pos, 7, false);
            if (tipPos != null) {
                class_2680 tipState = world.method_8320(tipPos);
                if (!isUnstable(tipState) && isTipDown(tipState) && canGrow(tipState, world, tipPos)) {
                    tryGrow(world, tipPos, class_2350.field_11033);
                }
            }
        }
    }

    private static void tryGrowGroundIcicle(class_3218 world, class_2338 pos) {
        class_2338.class_2339 mutable = pos.method_25503();

        for (int i = 0; i < 10; ++i) {
            mutable.method_10098(class_2350.field_11033);
            class_2680 blockState = world.method_8320(mutable);
            if (!blockState.method_26227().method_15769()) {
                return;
            }

            if (isTip(blockState, class_2350.field_11036) && canGrow(blockState, world, mutable)) {
                tryGrow(world, mutable, class_2350.field_11036);
                return;
            }

            if (canPlaceAtWithDirection(world, mutable, class_2350.field_11036)) {
                tryGrow(world, mutable.method_10074(), class_2350.field_11036);
                return;
            }
        }
    }

    private static void tryGrow(class_3218 world, class_2338 pos, class_2350 direction) {
        class_2338 posGrowingInto = pos.method_10093(direction);
        class_2680 stateGrowingInto = world.method_8320(posGrowingInto);
        if (isTip(stateGrowingInto, direction.method_10153())) {
            growMerged(stateGrowingInto, world, posGrowingInto);
        } else if (stateGrowingInto.method_26215() || stateGrowingInto.method_27852(class_2246.field_10382)) {
            place(world, posGrowingInto, direction, class_5691.field_28065);
        }
    }

    private static void place(class_1936 world, class_2338 pos, class_2350 direction, class_5691 thickness) {
        class_2680 blockState = FBlocks.ICICLE.method_9564().method_11657(VERTICAL_DIRECTION, direction).method_11657(THICKNESS, thickness).method_11657(WATERLOGGED, world.method_8316(pos).method_15772() == class_3612.field_15910);
        world.method_8652(pos, blockState, 3);
    }

    private static void growMerged(class_2680 state, class_1936 world, class_2338 pos) {
        class_2338 upperPos;
        class_2338 lowerPos;
        if (state.method_11654(VERTICAL_DIRECTION) == class_2350.field_11036) {
            lowerPos = pos;
            upperPos = pos.method_10084();
        } else {
            upperPos = pos;
            lowerPos = pos.method_10074();
        }

        place(world, upperPos, class_2350.field_11033, class_5691.field_28064);
        place(world, lowerPos, class_2350.field_11036, class_5691.field_28064);
    }

    @Nullable
    private static class_2338 getTipPos(class_2680 state, class_1936 world, class_2338 pos, int range, boolean allowMerged) {
        if (isTip(state, allowMerged)) {
            return pos;
        } else {
            class_2350 direction = state.method_11654(VERTICAL_DIRECTION);
            BiPredicate<class_2338, class_2680> continuePredicate = (posx, statex) -> {
                return statex.method_27852(FBlocks.ICICLE) && statex.method_11654(VERTICAL_DIRECTION) == direction;
            };
            Predicate<class_2680> stopPredicate = (statex) -> {
                return isTip(statex, allowMerged);
            };
            return searchInDirection(world, pos, direction.method_10171(), continuePredicate, stopPredicate, range).orElse(null);
        }
    }

    private static boolean isTipDown(class_2680 state) {
        return isPointingDown(state) && state.method_11654(THICKNESS) == class_5691.field_28065;
    }

    private static boolean canGrow(class_2680 state, class_3218 world, class_2338 pos) {
        class_2350 direction = state.method_11654(VERTICAL_DIRECTION);
        class_2338 blockPos = pos.method_10093(direction);
        class_2680 blockState = world.method_8320(blockPos);
        if (!blockState.method_26227().method_15769()) {
            return false;
        } else {
            return blockState.method_26215() || isTip(blockState, direction.method_10153());
        }
    }

    private static boolean canGrowIcicleOnAnchor(class_3218 world, class_2338 anchorPos, class_2680 anchorState) {

        if (world.method_8419()) {
            class_1959 biome = world.method_23753(anchorPos).comp_349();
            return biome.method_33599(anchorPos, world.method_8615());
        }

        return anchorState.method_26164(FBlockTags.ICICLE_GROWABLE);
    }

    private static Optional<class_2338> searchInDirection(class_1936 world, class_2338 pos, class_2350.class_2352 direction, BiPredicate<class_2338, class_2680> continuePredicate, Predicate<class_2680> stopPredicate, int range) {
        class_2350 toMove = class_2350.method_10156(direction, class_2350.class_2351.field_11052);
        class_2338.class_2339 current = pos.method_25503();

        for (int i = 1; i < range; i++) {
            current.method_10098(toMove);
            class_2680 blockState = world.method_8320(current);
            if (stopPredicate.test(blockState)) {
                return Optional.of(current.method_10062());
            }

            if (world.method_31601(current.method_10264()) || !continuePredicate.test(current, blockState)) {
                return Optional.empty();
            }
        }

        return Optional.empty();
    }

    private static boolean canPlaceAtWithDirection(class_4538 world, class_2338 pos, class_2350 direction) {
        class_2338 candidateAnchorPos = pos.method_10093(direction.method_10153());
        class_2680 candidateAnchorState = world.method_8320(candidateAnchorPos);
        return candidateAnchorState.method_26206(world, candidateAnchorPos, direction)
                || isIcicleFacingDirection(candidateAnchorState, direction);
    }

    @Nullable
    private static class_2350 getDirectionToPlaceAt(class_4538 world, class_2338 pos, class_2350 direction) {
        class_2350 toPlace;
        if (canPlaceAtWithDirection(world, pos, direction)) {
            toPlace = direction;
        } else {
            if (!canPlaceAtWithDirection(world, pos, direction.method_10153())) {
                return null;
            }
            toPlace = direction.method_10153();
        }

        return toPlace;
    }

    private static class_5691 getThickness(class_4538 world, class_2338 pos, class_2350 direction, boolean tryMerge) {
        class_2350 direction2 = direction.method_10153();
        class_2680 blockState = world.method_8320(pos.method_10093(direction));
        if (isIcicleFacingDirection(blockState, direction2)) {
            return !tryMerge && blockState.method_11654(THICKNESS) != class_5691.field_28064 ? class_5691.field_28065 : class_5691.field_28064;
        } else if (!isIcicleFacingDirection(blockState, direction)) {
            return class_5691.field_28065;
        } else {
            class_5691 thickness = blockState.method_11654(THICKNESS);
            if (thickness != class_5691.field_28065 && thickness != class_5691.field_28064) {
                class_2680 blockState2 = world.method_8320(pos.method_10093(direction2));
                return !isIcicleFacingDirection(blockState2, direction) ? class_5691.field_28068 : class_5691.field_28067;
            } else {
                return class_5691.field_28066;
            }
        }
    }

    private static boolean isUnstable(class_2680 state) {
        return state.method_27852(FBlocks.ICICLE) && state.method_11654(UNSTABLE);
    }

    private static boolean isPointingUp(class_2680 state) {
        return isIcicleFacingDirection(state, class_2350.field_11036);
    }

    private static boolean isPointingDown(class_2680 state) {
        return isIcicleFacingDirection(state, class_2350.field_11033);
    }

    private static boolean isIcicleFacingDirection(class_2680 state, class_2350 direction) {
        return state.method_27852(FBlocks.ICICLE) && state.method_11654(VERTICAL_DIRECTION) == direction;
    }

    private static boolean isTip(class_2680 state, class_2350 direction) {
        return isTip(state, false) && state.method_11654(VERTICAL_DIRECTION) == direction;
    }

    private static boolean isTip(class_2680 state, boolean allowMerged) {
        if (!state.method_27852(FBlocks.ICICLE)) {
            return false;
        } else {
            class_5691 thickness = state.method_11654(THICKNESS);
            return thickness == class_5691.field_28065 || (allowMerged && thickness == class_5691.field_28064);
        }
    }

    private static void spawnFallingBlock(class_2680 state, class_3218 world, class_2338 pos) {
        class_2338.class_2339 current = pos.method_25503();

        for (class_2680 blockState = state; isPointingDown(blockState); blockState = world.method_8320(current)) {
            class_1540 fallingBlockEntity = class_1540.method_40005(world, current, blockState);
            fallingBlockEntity.field_7193 = false;
            ((FallingBlockEntityAccessor) fallingBlockEntity).frostiful$setDestroyOnLanding(true);
            if (isTip(blockState, true)) {
                float fallHurtAmount = Math.max(1 + pos.method_10264() - current.method_10264(), 6);
                fallingBlockEntity.method_6965(fallHurtAmount, 40);
            }

            current.method_10098(class_2350.field_11033);
        }
    }

    private static boolean isHeldByIcicleFallable(class_2680 state, class_4538 world, class_2338 pos) {
        return !(world.method_8320(pos.method_10084()).method_26164(FBlockTags.ICICLE_GROWABLE)) || isHeldByIcicle(state, world, pos);
    }

    private static boolean isHeldByIcicle(class_2680 state, class_4538 world, class_2338 pos) {
        return isPointingDown(state) && world.method_8320(pos.method_10084()).method_27852(FBlocks.ICICLE);
    }

    private static void createUnstableParticle(class_1937 world, class_2338 pos, class_2680 state) {
        class_243 vec3d = state.method_26226(pos);
        double xOffset = (double) pos.method_10263() + 0.5D + vec3d.field_1352;
        double yOffset = (double) ((float) (pos.method_10264() + 1) - 0.6875F) - 0.0625D;
        double zOffset = (double) pos.method_10260() + 0.5D + vec3d.field_1350;
        world.method_8406(class_2398.field_28013, xOffset, yOffset, zOffset, 0.0D, 0.0D, 0.0D);
    }
}
