package com.github.thedeathlycow.frostiful.server.world.gen.feature;

import com.github.thedeathlycow.frostiful.block.IcicleHelper;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import net.minecraft.class_1936;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_3031;
import net.minecraft.class_3037;
import net.minecraft.class_3532;
import net.minecraft.class_4538;
import net.minecraft.class_5281;
import net.minecraft.class_5721;
import net.minecraft.class_5819;
import net.minecraft.class_5821;
import net.minecraft.class_5861;
import net.minecraft.class_5863;
import net.minecraft.class_5866;
import net.minecraft.class_6017;
import net.minecraft.class_6019;
import net.minecraft.util.valueproviders.*;
import java.util.Optional;
import java.util.OptionalInt;

public class IcicleFeature extends class_3031<IcicleFeature.IcicleFeatureConfig> {

    public IcicleFeature(Codec<IcicleFeatureConfig> configCodec) {
        super(configCodec);
    }

    @Override
    public boolean method_13151(class_5821<IcicleFeatureConfig> context) {

        // set up variables
        class_5281 worldAccess = context.method_33652();
        class_2338 origin = context.method_33655();
        class_5819 random = context.method_33654();
        IcicleFeatureConfig config = context.method_33656();

        // read radii from config
        int xRadius = config.radius.method_35008(random);
        int zRadius = config.radius.method_35008(random);

        float density = config.density.method_33920(random);
        int icicleHeight = config.icicleHeight.method_35008(random);

        // generate
        generate(worldAccess, random, origin, xRadius, zRadius, density, icicleHeight, config);

        return true;
    }

    private void generate(
            class_1936 world,
            class_5819 random,
            class_2338 origin,
            int xRadius,
            int zRadius,
            float density,
            int icicleHeight,
            IcicleFeatureConfig config
    ) {
        class_2338 pos;

        for (int x = -xRadius; x < xRadius; x++) {
            for (int z = -zRadius; z < zRadius; z++) {
                pos = origin.method_10069(x, 0, z);
                double icicleChance = this.icicleChance(xRadius, zRadius, x, z, config);
                generateColumn(world, random, pos, x, z, density, icicleChance, icicleHeight, config);
            }
        }
    }

    private void generateColumn(
            class_1936 world,
            class_5819 random,
            class_2338 position,
            int localX, int localZ,
            float density,
            double icicleChance,
            int icicleHeight,
            IcicleFeatureConfig config
    ) {

        // get surface heights
        Optional<class_5721> caveSurfaceResult = class_5721.method_32982(
                world,
                position,
                config.floorToCeilingSearchRange,
                IcicleHelper::canGenerate,
                IcicleHelper::canReplace
        );
        if (caveSurfaceResult.isEmpty()) {
            return;
        }
        class_5721 surface = caveSurfaceResult.get();
        OptionalInt ceilingHeight = surface.method_32985();
        OptionalInt floorHeight = surface.method_32987();
        if (ceilingHeight.isEmpty() && floorHeight.isEmpty()) {
            return;
        }

        boolean shouldGenerateCeiling = random.method_43058() < icicleChance;
        boolean shouldGenerateFloor = random.method_43058() < icicleChance;


        int floorIcicleLength = 0;
        int ceilingIcicleLength = 0;


        // build ice layers
        if (ceilingHeight.isPresent() && shouldGenerateCeiling && !this.isLava(world, position.method_33096(ceilingHeight.getAsInt()))) {
            int thickness = config.packedIceBlockLayerThickness.method_35008(random);
            this.placePackedIceBlocks(world, position.method_33096(ceilingHeight.getAsInt()), thickness, class_2350.field_11036);
            ceilingIcicleLength = getHeight(
                    random,
                    localX, localZ,
                    density,
                    icicleHeight,
                    config
            );
        }

        if (floorHeight.isPresent() && shouldGenerateFloor && !this.isLava(world, position.method_33096(floorHeight.getAsInt()))) {
            int thickness = config.packedIceBlockLayerThickness.method_35008(random);
            this.placePackedIceBlocks(world, position.method_33096(floorHeight.getAsInt()), thickness, class_2350.field_11033);
            floorIcicleLength = getHeight(
                    random,
                    localX, localZ,
                    density,
                    icicleHeight,
                    config
            );
        }

        // place icicles
        boolean merge = false;

        // adjust icicle heights if they would intersect with each other or terrain
        if (floorHeight.isPresent() && ceilingHeight.isPresent()) {
            int floorY = floorHeight.getAsInt();
            int ceilY = ceilingHeight.getAsInt();

            // number of air blocks between floor and ceiling
            // the -1 removes the stone blocks at the extrema
            int span = ceilY - floorY - 1;

            int iciclesLength = floorIcicleLength + ceilingIcicleLength;
            // icicles will intersect if and only if their summed lengths exceed the available space
            if (iciclesLength > span) {
                floorIcicleLength = span / 2;
                ceilingIcicleLength = floorIcicleLength + (span % 2);
                merge = floorIcicleLength > 0 && ceilingIcicleLength > 0 && random.method_43056();
            }
        }

        // place icicles
        if (ceilingHeight.isPresent() && shouldGenerateCeiling && ceilingIcicleLength > 0) {
            IcicleHelper.generateIcicle(
                    world,
                    position.method_33096(ceilingHeight.getAsInt() - 1),
                    class_2350.field_11033,
                    ceilingIcicleLength,
                    merge
            );
        }
        if (floorHeight.isPresent() && shouldGenerateFloor && floorIcicleLength > 0) {
            IcicleHelper.generateIcicle(
                    world,
                    position.method_33096(floorHeight.getAsInt() + 1),
                    class_2350.field_11036,
                    floorIcicleLength,
                    merge
            );
        }
    }

    private boolean isLava(class_4538 world, class_2338 pos) {
        return world.method_8320(pos).method_27852(class_2246.field_10164);
    }

    private void placePackedIceBlocks(class_1936 world, class_2338 pos, int thickness, class_2350 direction) {
        class_2338.class_2339 position = pos.method_25503();
        for (int i = 0; i < thickness; i++) {
            if (!IcicleHelper.generateIceBaseBlock(world, position)) {
                return;
            }
            position.method_10098(direction);
        }
    }

    private int getHeight(class_5819 random, int localX, int localZ, float density, int maxHeight, IcicleFeatureConfig config) {
        if (random.method_43057() > density) {
            return 0;
        }
        int absoluteDistanceManhattan = Math.abs(localX) + Math.abs(localZ);
        float mean = class_3532.method_37958(
                absoluteDistanceManhattan,
                1.0f,
                config.maxDistanceFromCenterAffectingHeightBias,
                maxHeight / 2.0f,
                1.0f
        );
        return (int) clampedGaussian(random, 1.0f, maxHeight, mean, config.heightDeviation);
    }

    private static float clampedGaussian(class_5819 random, float min, float max, float mean, float deviation) {
        return class_5861.method_33903(random, mean, deviation, min, max);
    }

    private double icicleChance(int radiusX, int radiusZ, int localX, int localZ, IcicleFeatureConfig config) {
        int distanceToEdgeX = radiusX - Math.abs(localX);
        int distanceToEdgeZ = radiusZ - Math.abs(localZ);
        int minDistanceToEdge = Math.min(distanceToEdgeX, distanceToEdgeZ);
        return class_3532.method_32854(
                minDistanceToEdge,
                0.0, IcicleFeatureConfig.MAX_DIST_FROM_CENTER_AFFECTING_COLUMN_CHANCE,
                IcicleFeatureConfig.CHANCE_OF_COLUMN_AT_MAX_DISTANCE, 1.0
        );
    }

    public record IcicleFeatureConfig(
            class_6017 icicleHeight,
            class_6017 radius,
            class_5863 density,
            int floorToCeilingSearchRange,
            class_6017 packedIceBlockLayerThickness,
            int heightDeviation,
            int maxDistanceFromCenterAffectingHeightBias
    ) implements class_3037 {

        private static final double MAX_DIST_FROM_CENTER_AFFECTING_COLUMN_CHANCE = 3.0;
        private static final double CHANCE_OF_COLUMN_AT_MAX_DISTANCE = 0.1;

        public static final Codec<IcicleFeatureConfig> CODEC = RecordCodecBuilder.create(
                instance -> instance.group(
                        class_6017.method_35004(0, 10)
                                .fieldOf("icicle_height")
                                .orElse(class_6019.method_35017(1, 6))
                                .forGetter(config -> config.icicleHeight),
                        class_6017.method_35004(1, 32)
                                .fieldOf("radius")
                                .orElse(class_6019.method_35017(2, 8))
                                .forGetter(config -> config.radius),
                        class_5863.method_33916(0f, 1f)
                                .fieldOf("density")
                                .orElse(class_5866.method_33934(0.1f, 0.4f))
                                .forGetter(config -> config.density),
                        Codec.intRange(1, 32)
                                .fieldOf("floor_to_ceiling_search_range")
                                .orElse(12)
                                .forGetter(config -> config.floorToCeilingSearchRange),
                        class_6017.method_35004(1, 32)
                                .fieldOf("packed_ice_layer_thickness")
                                .orElse(class_6019.method_35017(1, 4))
                                .forGetter(config -> config.packedIceBlockLayerThickness),
                        Codec.intRange(1, 64)
                                .fieldOf("height_deviation")
                                .orElse(3)
                                .forGetter(config -> config.heightDeviation),
                        Codec.intRange(1, 64)
                                .fieldOf("max_distance_from_center_affecting_height_bias")
                                .orElse(8)
                                .forGetter(config -> config.maxDistanceFromCenterAffectingHeightBias)
                ).apply(instance, IcicleFeatureConfig::new)
        );

    }

}
