package com.lowdragmc.lowdraglib.client.model.forge;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.lowdragmc.lowdraglib.client.model.custommodel.CustomBakedModel;
import com.lowdragmc.lowdraglib.client.renderer.IBlockRendererProvider;
import com.lowdragmc.lowdraglib.client.renderer.IRenderer;
import com.mojang.datafixers.util.Pair;
import net.minecraft.MethodsReturnNonnullByDefault;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.RenderType;
import net.minecraft.client.renderer.block.model.BakedQuad;
import net.minecraft.client.renderer.block.model.ItemOverrides;
import net.minecraft.client.renderer.texture.MissingTextureAtlasSprite;
import net.minecraft.client.renderer.texture.TextureAtlas;
import net.minecraft.client.renderer.texture.TextureAtlasSprite;
import net.minecraft.client.resources.model.*;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.util.RandomSource;
import net.minecraft.world.level.BlockAndTintGetter;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraftforge.client.model.data.ModelData;
import net.minecraftforge.client.model.data.ModelProperty;
import net.minecraftforge.client.model.geometry.IGeometryBakingContext;
import net.minecraftforge.client.model.geometry.IGeometryLoader;
import net.minecraftforge.client.model.geometry.IUnbakedGeometry;
import org.jetbrains.annotations.NotNull;
import javax.annotation.Nullable;

import javax.annotation.ParametersAreNonnullByDefault;
import java.util.*;
import java.util.function.Function;

/**
 * @author KilaBash
 * @date 2022/05/28
 * @implNote LDLModel, use vanilla way to improve model rendering
 */
@ParametersAreNonnullByDefault
@MethodsReturnNonnullByDefault
public class LDLRendererModel implements IUnbakedGeometry<LDLRendererModel> {
    public static final LDLRendererModel INSTANCE = new LDLRendererModel();

    private LDLRendererModel() {}

    @Override
    public BakedModel bake(IGeometryBakingContext iGeometryBakingContext, ModelBaker arg, Function<Material, TextureAtlasSprite> function, ModelState arg2, ItemOverrides arg3, ResourceLocation arg4) {
        return new RendererBakedModel();
    }

    public static final class RendererBakedModel implements BakedModel {

        @Override
        public List<BakedQuad> m_213637_(@Nullable BlockState state, @Nullable Direction direction, RandomSource random) {
            return Collections.emptyList();
        }

        @Override
        public boolean m_7541_() {
            return false;
        }

        @Override
        public boolean m_7539_() {
            return true;
        }

        @Override
        public boolean m_7547_() {
            return false;
        }

        @Override
        public boolean m_7521_() {
            return false;
        }

        @Override
        public TextureAtlasSprite m_6160_() {
            return Minecraft.m_91087_().m_91258_(TextureAtlas.f_118259_).apply(MissingTextureAtlasSprite.m_118071_());
        }

        @Override
        public ItemOverrides m_7343_() {
            return ItemOverrides.f_111734_;
        }

        // forge
        public static final ModelProperty<IRenderer> IRENDERER = new ModelProperty<>();
        public static final ModelProperty<BlockAndTintGetter> WORLD = new ModelProperty<>();
        public static final ModelProperty<BlockPos> POS = new ModelProperty<>();
        public static final ModelProperty<ModelData> MODEL_DATA = new ModelProperty<>();

        public static final ThreadLocal<ModelData> CURRENT_MODEL_DATA = new ThreadLocal<>();
        public static final ThreadLocal<RenderType> CURRENT_RENDER_TYPE = new ThreadLocal<>();

        @Override
        public @NotNull List<BakedQuad> getQuads(@Nullable BlockState state, @Nullable Direction side, @NotNull RandomSource rand, @NotNull ModelData data, @Nullable RenderType renderType) {
            IRenderer renderer = data.get(IRENDERER);
            BlockAndTintGetter world = data.get(WORLD);
            BlockPos pos = data.get(POS);
            if (renderer != null) {
                CURRENT_MODEL_DATA.set(data);
                CURRENT_RENDER_TYPE.set(renderType);
                var quads = renderer.renderModel(world, pos, state, side, rand);
                if (renderer.reBakeCustomQuads() && state != null && world != null && pos != null) {
                    return CustomBakedModel.reBakeCustomQuads(quads, world, pos, state, side, renderer.reBakeCustomQuadsOffset());
                }
                CURRENT_MODEL_DATA.remove();
                CURRENT_RENDER_TYPE.remove();
                return quads;
            }
            return Collections.emptyList();
        }

        @Override
        public boolean useAmbientOcclusion(BlockState state) {
            if (state.m_60734_() instanceof IBlockRendererProvider rendererProvider) {
                IRenderer renderer = rendererProvider.getRenderer(state);
                if (renderer != null) {
                    return renderer.useAO(state);
                }
            }
            return m_7541_();
        }


        @Override
        public @NotNull ModelData getModelData(@NotNull BlockAndTintGetter level, @NotNull BlockPos pos, @NotNull BlockState state, @NotNull ModelData modelData) {
            if (state.m_60734_() instanceof IBlockRendererProvider rendererProvider) {
                IRenderer renderer = rendererProvider.getRenderer(state);
                if (renderer != null) {
                    modelData = ModelData.builder()
                            .with(IRENDERER, renderer)
                            .with(WORLD, level)
                            .with(POS, pos)
                            .with(MODEL_DATA, modelData)
                            .build();
                }
            }
            return modelData;
        }

        @Override
        public TextureAtlasSprite getParticleIcon(@NotNull ModelData data) {
            IRenderer renderer = data.get(IRENDERER);
            if (renderer != null) {
                CURRENT_MODEL_DATA.set(data);
                var texture = renderer.getParticleTexture();
                CURRENT_MODEL_DATA.remove();
                return texture;
            }
            return BakedModel.super.getParticleIcon(data);
        }

    }

    public static final class Loader implements IGeometryLoader<LDLRendererModel> {

        public static final Loader INSTANCE = new Loader();
        private Loader() {}

        @Override
        public LDLRendererModel read(JsonObject jsonObject, JsonDeserializationContext jsonDeserializationContext) throws JsonParseException {
            return LDLRendererModel.INSTANCE;
        }
    }
}
