package com.lowdragmc.lowdraglib.client.scene;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import com.google.common.collect.Queues;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.BufferBuilder;
import com.mojang.blaze3d.vertex.PoseStack;
import com.mojang.blaze3d.vertex.Tesselator;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.client.Camera;
import net.minecraft.client.Minecraft;
import net.minecraft.client.particle.Particle;
import net.minecraft.client.particle.ParticleRenderType;
import net.minecraft.client.renderer.GameRenderer;
import net.minecraft.client.renderer.texture.TextureManager;
import net.minecraft.world.level.Level;

import java.util.*;

/**
 * @author KilaBash
 * @date 2022/06/05
 * @implNote ParticleManager, for LParticle
 */
@Environment(EnvType.CLIENT)
public class ParticleManager {
    private static final List<ParticleRenderType> RENDER_ORDER = ImmutableList.of(ParticleRenderType.f_107429_, ParticleRenderType.f_107430_, ParticleRenderType.f_107432_, ParticleRenderType.f_107431_, ParticleRenderType.f_107433_);
    private final Queue<Particle> waitToAdded = Queues.newArrayDeque();
    private final Map<ParticleRenderType, Queue<Particle>> particles = Maps.newTreeMap(makeParticleRenderTypeComparator(RENDER_ORDER));
    private final TextureManager textureManager = Minecraft.m_91087_().m_91097_();

    public Level level;

    public void setLevel(Level level) {
        this.level = level;
    }

    public void clearAllParticles() {
        synchronized (waitToAdded) {
            waitToAdded.clear();
            particles.clear();
        }
    }

    public void addParticle(Particle particle) {
        synchronized (waitToAdded) {
            waitToAdded.add(particle);
        }
    }

    public int getParticleAmount() {
        int amount = waitToAdded.size();
        amount += particles.values().stream().mapToInt(Collection::size).sum();
        return amount;
    }

    public void tick() {
        if (!waitToAdded.isEmpty()) {
            synchronized (waitToAdded) {
                for (var particle : waitToAdded) {
                    particles.computeIfAbsent(particle.m_7556_(), type -> Queues.newArrayDeque()).add(particle);
                }
                waitToAdded.clear();
            }
        }
        this.particles.forEach((particleRenderType, particleQueue) -> this.tickParticleList(particleQueue));
    }

    private void tickParticleList(Collection<Particle> pParticles) {
        if (!pParticles.isEmpty()) {
            var iterator = pParticles.iterator();
            while(iterator.hasNext()) {
                var particle = iterator.next();
                particle.m_5989_();
                if (!particle.m_107276_()) {
                    iterator.remove();
                }
            }
        }

    }

    public void render(PoseStack pMatrixStack, Camera pActiveRenderInfo, float pPartialTicks) {
        Minecraft.m_91087_().f_91063_.m_109154_().m_109896_();
        RenderSystem.enableDepthTest();
        RenderSystem.activeTexture(org.lwjgl.opengl.GL13.GL_TEXTURE2);
        RenderSystem.activeTexture(org.lwjgl.opengl.GL13.GL_TEXTURE0);
        PoseStack posestack = RenderSystem.getModelViewStack();
        posestack.m_85836_();
        posestack.m_252931_(pMatrixStack.m_85850_().m_252922_());
        RenderSystem.applyModelViewMatrix();

        for(ParticleRenderType particlerendertype : this.particles.keySet()) {
            if (particlerendertype == ParticleRenderType.f_107434_) continue;
            var iterable = this.particles.get(particlerendertype);
            if (iterable != null) {
                RenderSystem.setShader(GameRenderer::m_172829_);
                RenderSystem.setShaderColor(1.0F, 1.0F, 1.0F, 1.0F);
                Tesselator tesselator = Tesselator.m_85913_();
                BufferBuilder bufferbuilder = tesselator.m_85915_();
                particlerendertype.m_6505_(bufferbuilder, this.textureManager);

                for(var particle : iterable) {
                    try {
                        particle.m_5744_(bufferbuilder, pActiveRenderInfo, pPartialTicks);
                    } catch (Throwable throwable) {
                        throw throwable;
                    }
                }

                particlerendertype.m_6294_(tesselator);
            }
        }

        posestack.m_85849_();
        RenderSystem.applyModelViewMatrix();
        RenderSystem.depthMask(true);
        RenderSystem.disableBlend();
        Minecraft.m_91087_().f_91063_.m_109154_().m_109891_();
    }

    public static Comparator<ParticleRenderType> makeParticleRenderTypeComparator(List<ParticleRenderType> renderOrder) {
        Comparator<ParticleRenderType> vanillaComparator = Comparator.comparingInt(renderOrder::indexOf);
        return (typeOne, typeTwo) ->
        {
            boolean vanillaOne = renderOrder.contains(typeOne);
            boolean vanillaTwo = renderOrder.contains(typeTwo);

            if (vanillaOne && vanillaTwo)
            {
                return vanillaComparator.compare(typeOne, typeTwo);
            }
            if (!vanillaOne && !vanillaTwo)
            {
                return Integer.compare(System.identityHashCode(typeOne), System.identityHashCode(typeTwo));
            }
            return vanillaOne ? -1 : 1;
        };
    }

}
