package com.lowdragmc.lowdraglib.client.scene;

import com.lowdragmc.lowdraglib.Platform;
import com.lowdragmc.lowdraglib.client.shader.management.ShaderManager;
import com.lowdragmc.lowdraglib.client.utils.glu.Project;
import com.lowdragmc.lowdraglib.utils.*;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.*;
import com.mojang.blaze3d.vertex.BufferBuilder.RenderedBuffer;
import dev.architectury.injectables.annotations.ExpectPlatform;
import dev.architectury.injectables.annotations.PlatformOnly;
import lombok.Getter;
import lombok.Setter;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.client.Camera;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.screens.LoadingOverlay;
import net.minecraft.client.renderer.*;
import net.minecraft.client.renderer.MultiBufferSource.BufferSource;
import net.minecraft.client.renderer.block.BlockRenderDispatcher;
import net.minecraft.client.renderer.block.ModelBlockRenderer;
import net.minecraft.client.renderer.blockentity.BlockEntityRenderer;
import net.minecraft.client.renderer.entity.EntityRenderDispatcher;
import net.minecraft.client.renderer.texture.OverlayTexture;
import net.minecraft.core.BlockPos;
import net.minecraft.util.Mth;
import net.minecraft.util.RandomSource;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.level.BlockAndTintGetter;
import net.minecraft.world.level.ClipContext;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.material.FluidState;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraft.world.phys.Vec3;
import org.joml.Matrix4f;
import org.joml.Vector3f;
import org.joml.Vector4f;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.opengl.GL11;

import javax.annotation.Nonnull;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import static net.minecraft.world.level.block.RenderShape.INVISIBLE;

import F;


/**
 * @author KilaBash
 * @date 2022/05/25
 * @implNote  render a scene, through VBO compilation scene, greatly optimize rendering performance.
 */
@SuppressWarnings("ALL")
@Environment(EnvType.CLIENT)
public abstract class WorldSceneRenderer {
    protected static final FloatBuffer MODELVIEW_MATRIX_BUFFER = ByteBuffer.allocateDirect(16 * 4).order(ByteOrder.nativeOrder()).asFloatBuffer();
    protected static final FloatBuffer PROJECTION_MATRIX_BUFFER = ByteBuffer.allocateDirect(16 * 4).order(ByteOrder.nativeOrder()).asFloatBuffer();
    protected static final IntBuffer VIEWPORT_BUFFER = ByteBuffer.allocateDirect(16 * 4).order(ByteOrder.nativeOrder()).asIntBuffer();
    protected static final FloatBuffer PIXEL_DEPTH_BUFFER = ByteBuffer.allocateDirect(4).order(ByteOrder.nativeOrder()).asFloatBuffer();
    protected static final FloatBuffer OBJECT_POS_BUFFER = ByteBuffer.allocateDirect(3 * 4).order(ByteOrder.nativeOrder()).asFloatBuffer();
    enum CacheState {
        UNUSED,
        NEED,
        COMPILING,
        COMPILED
    }

    public final Level world;
    public final Map<Collection<BlockPos>, ISceneBlockRenderHook> renderedBlocksMap;
    protected VertexBuffer[] vertexBuffers;
    protected Set<BlockPos> tileEntities;
    protected boolean useCache;
    @Getter @Setter
    protected boolean endBatchLast = false;// if true, endBatch will be called after all rendering
    protected boolean ortho;
    protected AtomicReference<CacheState> cacheState;
    protected int maxProgress;
    protected int progress;
    protected Thread thread;
    protected ParticleManager particleManager;
    protected Camera camera;
    protected CameraEntity cameraEntity;
    private Consumer<WorldSceneRenderer> beforeRender;
    private Consumer<WorldSceneRenderer> afterRender;
    @Setter @Nullable
    private BiConsumer<MultiBufferSource, Float> beforeBatchEnd;
    private Consumer<BlockHitResult> onLookingAt;
    @Setter @Nullable
    private ISceneEntityRenderHook sceneEntityRenderHook;
    protected int clearColor;
    @Getter
    private Vector3f lastHit;
    @Getter
    private BlockHitResult lastTraceResult;
    private Set<BlockPos> blocked;
    private Vector3f eyePos = new Vector3f(0, 0, 10f);
    private Vector3f lookAt = new Vector3f(0, 0, 0);
    private Vector3f worldUp = new Vector3f(0, 1, 0);
    @Getter
    private float fov = 60f;
    private float minX, maxX, minY, maxY, minZ, maxZ;

    public WorldSceneRenderer(Level world) {
        this.world = world;
        renderedBlocksMap = new LinkedHashMap<>();
        cacheState = new AtomicReference<>(CacheState.UNUSED);
    }

    public WorldSceneRenderer setParticleManager(ParticleManager particleManager) {
        if (particleManager == null) {
            this.particleManager = null;
            this.camera = null;
            this.cameraEntity = null;
            return this;
        }
        this.particleManager = particleManager;
        if (this.world instanceof DummyWorld dummyWorld) {
            dummyWorld.setParticleManager(particleManager);
        }
        this.particleManager.setLevel(world);
        this.camera = new Camera();
        this.cameraEntity = new CameraEntity(world);
        setCameraLookAt(eyePos, lookAt, worldUp);
        return this;
    }

    public ParticleManager getParticleManager() {
        return particleManager;
    }

    public WorldSceneRenderer useCacheBuffer(boolean useCache) {
        if (this.useCache || !Minecraft.m_91087_().m_18695_()) return this;
        deleteCacheBuffer();
        if (useCache) {
            List<RenderType> layers = RenderType.m_110506_();
            this.vertexBuffers = new VertexBuffer[layers.size()];
            for (int j = 0; j < layers.size(); ++j) {
                this.vertexBuffers[j] = new VertexBuffer(VertexBuffer.Usage.STATIC);
            }
            if (cacheState.get() == CacheState.COMPILING && thread != null) {
                thread.interrupt();
                thread = null;
            }
            cacheState.set(CacheState.NEED);
        }
        this.useCache = useCache;
        return this;
    }

    public WorldSceneRenderer useOrtho(boolean ortho) {
        this.ortho = ortho;
        return this;
    }

    public WorldSceneRenderer deleteCacheBuffer() {
        if (useCache) {
            for (int i = 0; i < RenderType.m_110506_().size(); ++i) {
                if (this.vertexBuffers[i] != null) {
                    this.vertexBuffers[i].close();
                }
            }
            if (cacheState.get() == CacheState.COMPILING && thread != null) {
                thread.interrupt();
                thread = null;
            }
        }
        this.tileEntities = null;
        useCache = false;
        cacheState.set(CacheState.UNUSED);
        return this;
    }

    public WorldSceneRenderer needCompileCache() {
        if (cacheState.get() == CacheState.COMPILING && thread != null) {
            thread.interrupt();
            thread = null;
        }
        cacheState.set(CacheState.NEED);
        return this;
    }

    public WorldSceneRenderer setBeforeWorldRender(Consumer<WorldSceneRenderer> callback) {
        this.beforeRender = callback;
        return this;
    }

    public WorldSceneRenderer setAfterWorldRender(Consumer<WorldSceneRenderer> callback) {
        this.afterRender = callback;
        return this;
    }

    public WorldSceneRenderer addRenderedBlocks(Collection<BlockPos> blocks, ISceneBlockRenderHook renderHook) {
        if (blocks != null) {
            this.renderedBlocksMap.put(blocks, renderHook);
        }
        return this;
    }

    public WorldSceneRenderer setBlocked(Set<BlockPos> blocked) {
        this.blocked = blocked;
        return this;
    }

    public WorldSceneRenderer setOnLookingAt(Consumer<BlockHitResult> onLookingAt) {
        this.onLookingAt = onLookingAt;
        return this;
    }

    public boolean isUseCache() {
        return useCache;
    }

    public void setClearColor(int clearColor) {
        this.clearColor = clearColor;
    }

    public void render(@Nonnull PoseStack poseStack, float x, float y, float width, float height, int mouseX, int mouseY) {
        // do not render if the minecraft is reloading
        if (Minecraft.m_91087_().m_91265_() instanceof LoadingOverlay) {
            return;
        }
        // setupCamera
        var pose = poseStack.m_85850_().m_252922_();
        Vector4f pos = new Vector4f(x, y, 0, 1.0F);
        pos = pose.transform(pos);
        Vector4f size = new Vector4f(x + width, y + height, 0, 1.0F);
        size = pose.transform(size);
        x = pos.x();
        y = pos.y();
        width = size.x() - x;
        height = size.y() - y;
        PositionedRect viewport = getPositionedRect((int)x, (int)y, (int)width, (int)height);
        var topLeft = poseStack.m_85850_().m_252922_().transformPosition(new Vector3f(0.0f, 0.0f, 0.0f));
        PositionedRect mouse = getPositionedRect((int) (mouseX + topLeft.x), (int) (mouseY + topLeft.y), 0, 0);
        mouseX = mouse.position.x;
        mouseY = mouse.position.y;
        setupCamera(viewport);
        // render TrackedDummyWorld
        drawWorld();
        // check lookingAt
        this.lastTraceResult = null;
        this.lastHit = unProject(mouseX, mouseY);
        if (onLookingAt != null && mouseX > viewport.position.x && mouseX < viewport.position.x + viewport.size.width
                && mouseY > viewport.position.y && mouseY < viewport.position.y + viewport.size.height) {
            BlockHitResult result = rayTrace(lastHit);
            if (result != null) {
                this.lastTraceResult = null;
                this.lastTraceResult = result;
                onLookingAt.accept(result);
            }
        }
        // resetCamera
        resetCamera();
    }

    public Vector3f getEyePos() {
        return eyePos;
    }

    public Vector3f getLookAt() {
        return lookAt;
    }

    public Vector3f getWorldUp() {
        return worldUp;
    }

    public void setCameraLookAt(Vector3f eyePos, Vector3f lookAt, Vector3f worldUp) {
        this.eyePos = eyePos;
        this.lookAt = lookAt;
        this.worldUp = worldUp;
        if (camera != null) {
            Vector3f xzProduct = new Vector3f(lookAt.x() - eyePos.x(), 0, lookAt.z() - eyePos.z());
            double angleYaw = Math.toDegrees(xzProduct.angle(new Vector3f(0, 0, 1)));
            if (xzProduct.angle(new Vector3f(1, 0, 0)) < Math.PI / 2) {
                angleYaw = -angleYaw;
            }
            double anglePitch = Math.toDegrees(new Vector3f(lookAt).sub(new Vector3f(eyePos)).angle(new Vector3f(0, 1, 0))) - 90;
            cameraEntity.m_6034_(eyePos.x(), eyePos.y() - cameraEntity.m_20192_(), eyePos.z());
            cameraEntity.f_19854_ = cameraEntity.m_20185_();
            cameraEntity.f_19855_ = cameraEntity.m_20186_();
            cameraEntity.f_19856_ = cameraEntity.m_20189_();
            cameraEntity.m_146922_((float) angleYaw);
            cameraEntity.m_146926_((float) anglePitch);
            cameraEntity.f_19859_ = cameraEntity.m_146908_();
            cameraEntity.f_19860_ = cameraEntity.m_146909_();
        }
    }

    public void setCameraLookAt(Vector3f lookAt, double radius, double rotationPitch, double rotationYaw) {
        Vector3f vecX = new Vector3f((float) Math.cos(rotationPitch), (float) 0, (float) Math.sin(rotationPitch));
        Vector3f vecY = new Vector3f(0, (float) (Math.tan(rotationYaw) * vecX.length()),0);
        Vector3f pos = new Vector3f(vecX).add(vecY).normalize().mul((float) radius);
        setCameraLookAt(pos.add(lookAt.x(), lookAt.y(), lookAt.z()), lookAt, worldUp);
    }

    public void setFov(float fov) {
        this.fov = fov;
    }

    public void setCameraOrtho(float x, float y, float z) {
        this.minX = -x;
        this.maxX = x;
        this.minY = -y;
        this.maxY = y;
        this.minZ = -z;
        this.maxZ = z;
    }

    public void setCameraOrtho(float minX, float maxX, float minY, float maxY, float minZ, float maxZ) {
        this.minX = minX;
        this.maxX = maxX;
        this.minY = minY;
        this.maxY = maxY;
        this.minZ = minZ;
        this.maxZ = maxZ;
    }

    public PositionedRect getPositionedRect(int x, int y, int width, int height) {
        return new PositionedRect(new Position(x, y), new Size(width, height));
    }

    public PositionedRect getPositionRectRevert(int windowX, int windowY, int windowWidth, int windowHeight) {
        return new PositionedRect(new Position(windowX, windowY), new Size(windowWidth, windowHeight));
    }

    protected void setupCamera(PositionedRect viewport) {
        int x = viewport.getPosition().x;
        int y = viewport.getPosition().y;
        int width = viewport.getSize().width;
        int height = viewport.getSize().height;

        RenderSystem.enableDepthTest();
        RenderSystem.enableBlend();

        //setup viewport and clear GL buffers
        RenderSystem.viewport(x, y, width, height);

        RenderSystem.depthMask(true);
        clearView(x, y, width, height);

        //setup projection matrix to perspective
        RenderSystem.backupProjectionMatrix();

        float aspectRatio = width / (height * 1.0f);
        if (ortho) {
            RenderSystem.setProjectionMatrix(new Matrix4f().setOrtho(minX, maxX, minY / aspectRatio, maxY / aspectRatio, minZ, maxZ), VertexSorting.m_276997_(camera.m_90583_().m_252839_()));
        } else {
            RenderSystem.setProjectionMatrix(new Matrix4f().setPerspective(fov * 0.01745329238474369F, aspectRatio, 0.1f, 10000.0f), VertexSorting.m_276997_(camera.m_90583_().m_252839_()));
        }

        //setup modelview matrix
        PoseStack posesStack = RenderSystem.getModelViewStack();
        posesStack.m_85836_();
        posesStack.m_166856_();
        Project.gluLookAt(posesStack, eyePos.x(), eyePos.y(), eyePos.z(), lookAt.x(), lookAt.y(), lookAt.z(), worldUp.x(), worldUp.y(), worldUp.z());
        RenderSystem.applyModelViewMatrix();

        RenderSystem.activeTexture(org.lwjgl.opengl.GL13.GL_TEXTURE0);

        Minecraft mc = Minecraft.m_91087_();
        RenderSystem.enableCull();

        if (camera != null) {
            camera.m_90575_(world, cameraEntity, false, false, mc.m_91296_());
        }
        ShaderManager.getInstance().setViewPort(viewport);

    }

    protected void clearView(int x, int y, int width, int height) {
        int i = (clearColor & 0xFF0000) >> 16;
        int j = (clearColor & 0xFF00) >> 8;
        int k = (clearColor & 0xFF);
        RenderSystem.clearColor(i / 255.0f, j / 255.0f, k / 255.0f, (clearColor >> 24) / 255.0f);
        RenderSystem.clear(GL11.GL_COLOR_BUFFER_BIT | GL11.GL_DEPTH_BUFFER_BIT, Minecraft.f_91002_);
    }

    protected void resetCamera() {
        RenderSystem.clear(GL11.GL_DEPTH_BUFFER_BIT, Minecraft.f_91002_);

        //reset viewport
        Minecraft minecraft = Minecraft.m_91087_();
        RenderSystem.viewport(0, 0, minecraft.m_91268_().m_85441_(), minecraft.m_91268_().m_85442_());

        //reset projection matrix
        RenderSystem.restoreProjectionMatrix();

        //reset modelview matrix
        PoseStack posesStack = RenderSystem.getModelViewStack();
        posesStack.m_85849_();
        RenderSystem.applyModelViewMatrix();
        
        RenderSystem.depthMask(false);
        RenderSystem.disableDepthTest();
        RenderSystem.enableBlend();
        ShaderManager.getInstance().clearViewPort();

    }

    protected void drawWorld() {
        if (beforeRender != null) {
            beforeRender.accept(this);
        }

        Minecraft mc = Minecraft.m_91087_();

        float particleTicks = mc.m_91296_();
        var buffers = mc.m_91269_().m_110104_();
        if (useCache) {
            renderCacheBuffer(mc, buffers, particleTicks);
        } else {
            BlockRenderDispatcher blockrendererdispatcher = mc.m_91289_();
            try { // render com.lowdragmc.lowdraglib.test.block in each layer
                renderedBlocksMap.forEach((renderedBlocks, hook) -> {
                    for (RenderType layer : RenderType.m_110506_()) {
                        layer.m_110185_();
                        Random random = new Random();
                        PoseStack poseStack = new PoseStack();

                        if (layer == RenderType.m_110466_()) { // render tesr before translucent
                            if (hook != null) {
                                hook.apply(true, layer);
                            } else {
                                setDefaultRenderLayerState(null);
                            }

                            renderTESR(renderedBlocks, poseStack, buffers, hook, particleTicks);

                            if (hook != null || !endBatchLast) {
                                buffers.m_109911_();
                            }
                        }

                        if (hook != null) {
                            hook.apply(false, layer);
                        } else {
                            setDefaultRenderLayerState(layer);
                        }

                        var buffer = buffers.m_6299_(layer);

                        renderBlocks(poseStack, blockrendererdispatcher, layer, new VertexConsumerWrapper(buffer), renderedBlocks, hook, particleTicks);

                        if (!endBatchLast) {
                            buffers.m_109911_();
                        }
                        layer.m_110188_();
                    }
                });
            } finally {
            }
        }

        if (world instanceof TrackedDummyWorld level) {
            PoseStack poseStack = new PoseStack();
            renderEntities(level, poseStack, buffers, sceneEntityRenderHook, particleTicks);
        }

        if (beforeBatchEnd != null) {
            beforeBatchEnd.accept(buffers, particleTicks);
        }

        buffers.m_109911_();

        if (particleManager != null) {
            @Nonnull PoseStack poseStack = new PoseStack();
            poseStack.m_166856_();
            poseStack.m_85837_(cameraEntity.m_20185_(), cameraEntity.m_20186_(), cameraEntity.m_20189_());
            particleManager.render(poseStack, camera, particleTicks);
        }

        if (afterRender != null) {
            afterRender.accept(this);
        }
    }

    public boolean isCompiling() {
        return cacheState.get() == CacheState.COMPILING;
    }

    public double getCompileProgress() {
        if (maxProgress > 1000) {
            return progress * 1. / maxProgress;
        }
        return 0;
    }

    private void renderCacheBuffer(Minecraft mc, MultiBufferSource.BufferSource buffers, float particleTicks) {
        List<RenderType> layers = RenderType.m_110506_();
        if (cacheState.get() == CacheState.NEED) {
            progress = 0;
            maxProgress = renderedBlocksMap.keySet().stream().map(Collection::size).reduce(0, Integer::sum) * (layers.size() + 1);
            thread = new Thread(()->{
                cacheState.set(CacheState.COMPILING);
                BlockRenderDispatcher blockrendererdispatcher = mc.m_91289_();
                try { // render com.lowdragmc.lowdraglib.test.block in each layer
                    ModelBlockRenderer.m_111000_();
                    PoseStack matrixstack = new PoseStack();
                    for (int i = 0; i < layers.size(); i++) {
                        if (Thread.interrupted())
                            return;
                        RenderType layer = layers.get(i);
                        BufferBuilder buffer = new BufferBuilder(layer.m_110507_());
                        buffer.m_166779_(VertexFormat.Mode.QUADS, DefaultVertexFormat.f_85811_);
                        renderedBlocksMap.forEach((renderedBlocks, hook) -> {
                            renderBlocks(matrixstack, blockrendererdispatcher, layer, new VertexConsumerWrapper(buffer), renderedBlocks, hook, 0);
                        });
                        var builder = buffer.m_231175_();

                        var vertexBuffer = vertexBuffers[i];
                        Runnable toUpload = () -> {
                            if (!vertexBuffer.m_231230_()) {
                                vertexBuffer.m_85921_();
                                vertexBuffer.m_231221_(builder);
                                VertexBuffer.m_85931_();
                            }
                        };
                        CompletableFuture.runAsync(toUpload, runnable -> {
                            RenderSystem.recordRenderCall(runnable::run);
                        });

                    }
                    ModelBlockRenderer.m_111077_();
                } finally {
                }
                Set<BlockPos> poses = new HashSet<>();
                renderedBlocksMap.forEach((renderedBlocks, hook) -> {
                    for (BlockPos pos : renderedBlocks) {
                        progress++;
                        if (Thread.interrupted())
                            return;
                        BlockEntity tile = world.m_7702_(pos);
                        if (tile != null) {
                            if (mc.m_167982_().m_112265_(tile) != null) {
                                poses.add(pos);
                            }
                        }
                    }
                });
                if (Thread.interrupted())
                    return;
                tileEntities = poses;
                cacheState.set(CacheState.COMPILED);
                thread = null;
                maxProgress = -1;
            });
            thread.start();
        } else {
            PoseStack matrixstack = new PoseStack();
            for (int i = 0; i < layers.size(); i++) {
                VertexBuffer vertexbuffer = vertexBuffers[i];
                if (vertexbuffer.m_231230_() || vertexbuffer.m_166892_() == null) continue;

                RenderType layer = layers.get(i);
                if (layer == RenderType.m_110466_() && tileEntities != null) { // render tesr before translucent
                    if (world instanceof TrackedDummyWorld level) {
                        renderEntities(level, matrixstack, buffers, sceneEntityRenderHook, particleTicks);
                    }
                    renderTESR(tileEntities, matrixstack, mc.m_91269_().m_110104_(), null, particleTicks);
                    if (!endBatchLast) {
                        buffers.m_109911_();
                    }
                }

                layer.m_110185_();

                matrixstack.m_85836_();

                ShaderInstance shaderInstance = RenderSystem.getShader();

                for(int j = 0; j < 12; ++j) {
                    int k = RenderSystem.getShaderTexture(j);
                    shaderInstance.m_173350_("Sampler" + j, k);
                }

                // setup shader uniform
                if (shaderInstance.f_173308_ != null) {
                    shaderInstance.f_173308_.m_5679_(RenderSystem.getModelViewMatrix());
                }

                if (shaderInstance.f_173309_ != null) {
                    shaderInstance.f_173309_.m_5679_(RenderSystem.getProjectionMatrix());
                }

                if (shaderInstance.f_173312_ != null) {
                    shaderInstance.f_173312_.m_5941_(RenderSystem.getShaderColor());
                }

                if (shaderInstance.f_173315_ != null) {
                    shaderInstance.f_173315_.m_5985_(RenderSystem.getShaderFogStart());
                }

                if (shaderInstance.f_173316_ != null) {
                    shaderInstance.f_173316_.m_5985_(RenderSystem.getShaderFogEnd());
                }

                if (shaderInstance.f_173317_ != null) {
                    shaderInstance.f_173317_.m_5941_(RenderSystem.getShaderFogColor());
                }

                if (shaderInstance.f_202432_ != null) {
                    shaderInstance.f_202432_.m_142617_(RenderSystem.getShaderFogShape().m_202324_());
                }

                if (shaderInstance.f_173310_ != null) {
                    shaderInstance.f_173310_.m_5679_(RenderSystem.getTextureMatrix());
                }

                if (shaderInstance.f_173319_ != null) {
                    shaderInstance.f_173319_.m_5985_(RenderSystem.getShaderGameTime());
                }

                RenderSystem.setupShaderLights(shaderInstance);
                shaderInstance.m_173363_();

                setDefaultRenderLayerState(layer);

                RenderSystem.setShaderColor(1.0F, 1.0F, 1.0F, 1.0F);

                vertexbuffer.m_85921_();
                vertexbuffer.m_166882_();

                matrixstack.m_85849_();

                shaderInstance.m_173362_();
                VertexBuffer.m_85931_();
                layer.m_110188_();
            }
        }
    }

    private void renderBlocks(PoseStack poseStack, BlockRenderDispatcher blockrendererdispatcher, RenderType layer, VertexConsumerWrapper wrapperBuffer, Collection<BlockPos> renderedBlocks, @Nullable ISceneBlockRenderHook hook, float partialTicks) {
        for (BlockPos pos : renderedBlocks) {
            if (blocked != null && blocked.contains(pos)) {
                continue;
            }
            BlockState state = world.m_8055_(pos);
            FluidState fluidState = state.m_60819_();
            Block block = state.m_60734_();
            BlockEntity te = world.m_7702_(pos);

            if (hook != null) {
                hook.applyVertexConsumerWrapper(world, pos, state, wrapperBuffer, layer, partialTicks);
            }

            if (block == Blocks.f_50016_) continue;
            if (state.m_60799_() != INVISIBLE && canRenderInLayer(state, layer)) {
                poseStack.m_85836_();
                poseStack.m_252880_(pos.m_123341_(), pos.m_123342_(), pos.m_123343_());
                if (Platform.isForge()) {
                    renderBlocksForge(blockrendererdispatcher, state, pos, world, poseStack, wrapperBuffer, world.f_46441_, layer);
                } else {
                    blockrendererdispatcher.m_234355_(state, pos, world, poseStack, wrapperBuffer, false, world.f_46441_);
                }
                poseStack.m_85849_();
            }
            if (!fluidState.m_76178_() && ItemBlockRenderTypes.m_109287_(fluidState) == layer) { // I dont want to do this fxxk wrapper
                wrapperBuffer.addOffset((pos.m_123341_() - (pos.m_123341_() & 15)), (pos.m_123342_() - (pos.m_123342_() & 15)), (pos.m_123343_() - (pos.m_123343_() & 15)));
                blockrendererdispatcher.m_234363_(pos, world, wrapperBuffer, state, fluidState);
            }
            wrapperBuffer.clerOffset();
            wrapperBuffer.clearColor();
            if (maxProgress > 0) {
                progress++;
            }
        }
    }

    @ExpectPlatform
    public static boolean canRenderInLayer(BlockState state, RenderType renderType) {
        throw new AssertionError();
    }

    @ExpectPlatform
    @PlatformOnly(PlatformOnly.FORGE)
    public static void renderBlocksForge(BlockRenderDispatcher blockRenderDispatcher, BlockState state, BlockPos pos, BlockAndTintGetter level, @Nonnull PoseStack poseStack, VertexConsumer consumer, RandomSource random, RenderType renderType) {
        throw new AssertionError();
    }


    private void renderTESR(Collection<BlockPos> poses, PoseStack poseStack, MultiBufferSource.BufferSource buffers, @Nullable ISceneBlockRenderHook hook, float partialTicks) {
        for (BlockPos pos : poses) {
            BlockEntity tile = world.m_7702_(pos);
            if (tile != null) {
                poseStack.m_85836_();
                poseStack.m_252880_(pos.m_123341_(), pos.m_123342_(), pos.m_123343_());
                BlockEntityRenderer<BlockEntity> tileentityrenderer = Minecraft.m_91087_().m_167982_().m_112265_(tile);
                if (tileentityrenderer != null) {
                    if (tile.m_58898_() && tile.m_58903_().m_155262_(tile.m_58900_())) {
                        Level world = tile.m_58904_();

                        if (hook != null) {
                            hook.applyBESR(world, pos, tile, poseStack, partialTicks);
                        }

                        tileentityrenderer.m_6922_(tile, partialTicks, poseStack, buffers, 0xF000F0, OverlayTexture.f_118083_);
                    }
                }
                poseStack.m_85849_();
            }
        }
    }

    private void renderEntities(TrackedDummyWorld level, PoseStack poseStack, MultiBufferSource buffer, @Nullable ISceneEntityRenderHook hook, float partialTicks) {
        for (Entity entity : level.getAllEntities()) {
            poseStack.m_85836_();
            if (entity.f_19797_ == 0) {
                entity.f_19790_ = entity.m_20185_();
                entity.f_19791_ = entity.m_20186_();
                entity.f_19792_ = entity.m_20189_();
            }
            double d0 = Mth.m_14139_((double) partialTicks, entity.f_19790_, entity.m_20185_());
            double d1 = Mth.m_14139_((double) partialTicks, entity.f_19791_, entity.m_20186_());
            double d2 = Mth.m_14139_((double) partialTicks, entity.f_19792_, entity.m_20189_());
            float f = Mth.m_14179_(partialTicks, entity.f_19859_, entity.m_146908_());
            var renderManager = Minecraft.m_91087_().m_91290_();
            int light = renderManager.m_114382_(entity).m_114505_(entity, partialTicks);
            if (hook != null) {
                hook.applyEntity(world, entity, poseStack, partialTicks);
            }
            renderManager.m_114384_(entity, d0, d1, d2, f, partialTicks, poseStack, buffer, light);
            poseStack.m_85849_();
        }
    }


    public static void setDefaultRenderLayerState(RenderType layer) {
        RenderSystem.setShaderColor(1, 1, 1, 1);
        if (layer == RenderType.m_110466_()) { // TRANSLUCENT
            RenderSystem.enableBlend();
            RenderSystem.blendFunc(GL11.GL_SRC_ALPHA, GL11.GL_ONE_MINUS_SRC_ALPHA);
            RenderSystem.depthMask(false);
        } else { // SOLID
            RenderSystem.enableDepthTest();
            RenderSystem.disableBlend();
            RenderSystem.depthMask(true);
        }
    }

    public BlockHitResult rayTrace(Vector3f hitPos) {
        Vec3 startPos = new Vec3(this.eyePos.x(), this.eyePos.y(), this.eyePos.z());
        if (ortho) {
            startPos = startPos.m_82549_(new Vec3(startPos.f_82479_ - lookAt.x(), startPos.f_82480_ - lookAt.y(), startPos.f_82481_ - lookAt.z()).m_82542_(500, 500, 500));
        }
        hitPos = hitPos.mul(2, new Vector3f()); // Double view range to ensure pos can be seen.
        Vec3 endPos = new Vec3((hitPos.x() - startPos.f_82479_), (hitPos.y() - startPos.f_82480_), (hitPos.z() - startPos.f_82481_));
        try {
            return this.world.m_45547_(new ClipContext(startPos, endPos, ClipContext.Block.OUTLINE, ClipContext.Fluid.NONE, cameraEntity));
        } catch (Exception e) {
            return null;
        }
    }

    public Vector3f project(Vector3f pos) {
        //read current rendering parameters
        RenderSystem.getModelViewMatrix().get(MODELVIEW_MATRIX_BUFFER);
        RenderSystem.getProjectionMatrix().get(PROJECTION_MATRIX_BUFFER);
        GL11.glGetIntegerv(GL11.GL_VIEWPORT, VIEWPORT_BUFFER);

        //rewind buffers after write by OpenGL glGet calls
        MODELVIEW_MATRIX_BUFFER.rewind();
        PROJECTION_MATRIX_BUFFER.rewind();
        VIEWPORT_BUFFER.rewind();

        //call gluProject with retrieved parameters
        Project.gluProject(pos.x(), pos.y(), pos.z(), MODELVIEW_MATRIX_BUFFER, PROJECTION_MATRIX_BUFFER, VIEWPORT_BUFFER, OBJECT_POS_BUFFER);

        //rewind buffers after read by gluProject
        VIEWPORT_BUFFER.rewind();
        PROJECTION_MATRIX_BUFFER.rewind();
        MODELVIEW_MATRIX_BUFFER.rewind();

        //rewind buffer after write by gluProject
        OBJECT_POS_BUFFER.rewind();

        //obtain position in Screen
        float winX = OBJECT_POS_BUFFER.get();
        float winY = OBJECT_POS_BUFFER.get();
        float winZ = OBJECT_POS_BUFFER.get();

        //rewind buffer after read
        OBJECT_POS_BUFFER.rewind();

        return new Vector3f(winX, winY, winZ);
    }

    public Vector3f unProject(int mouseX, int mouseY) {
        return unProject(mouseX, mouseY, true);
    }

    public Vector3f unProject(int mouseX, int mouseY, boolean checkDepth) {
        var pixelDepth = 0.999f;
        if (checkDepth) {
            //read depth of pixel under mouse
            GL11.glReadPixels(mouseX, mouseY, 1, 1, GL11.GL_DEPTH_COMPONENT, GL11.GL_FLOAT, PIXEL_DEPTH_BUFFER);

            //rewind buffer after write by glReadPixels
            PIXEL_DEPTH_BUFFER.rewind();

            //retrieve depth from buffer (0.0-1.0f)
            pixelDepth = PIXEL_DEPTH_BUFFER.get();
        }

        //rewind buffer after read
        PIXEL_DEPTH_BUFFER.rewind();

        //read current rendering parameters
        RenderSystem.getModelViewMatrix().get(MODELVIEW_MATRIX_BUFFER);
        RenderSystem.getProjectionMatrix().get(PROJECTION_MATRIX_BUFFER);
        GL11.glGetIntegerv(GL11.GL_VIEWPORT, VIEWPORT_BUFFER);

        //rewind buffers after write by OpenGL glGet calls
        MODELVIEW_MATRIX_BUFFER.rewind();
        PROJECTION_MATRIX_BUFFER.rewind();
        VIEWPORT_BUFFER.rewind();

        //call gluUnProject with retrieved parameters
        Project.gluUnProject(mouseX, mouseY, pixelDepth, MODELVIEW_MATRIX_BUFFER, PROJECTION_MATRIX_BUFFER, VIEWPORT_BUFFER, OBJECT_POS_BUFFER);

        //rewind buffers after read by gluUnProject
        VIEWPORT_BUFFER.rewind();
        PROJECTION_MATRIX_BUFFER.rewind();
        MODELVIEW_MATRIX_BUFFER.rewind();

        //rewind buffer after write by gluUnProject
        OBJECT_POS_BUFFER.rewind();

        //obtain absolute position in world
        float posX = OBJECT_POS_BUFFER.get();
        float posY = OBJECT_POS_BUFFER.get();
        float posZ = OBJECT_POS_BUFFER.get();

        //rewind buffer after read
        OBJECT_POS_BUFFER.rewind();

        return new Vector3f(posX, posY, posZ);
    }

    /***
     * For better performance, You'd better handle the event {@link #setOnLookingAt(Consumer)} or {@link #getLastTraceResult()}
     * @param mouseX xPos in Texture
     * @param mouseY yPos in Texture
     * @return RayTraceResult Hit
     */
    protected BlockHitResult screenPos2BlockPosFace(int mouseX, int mouseY, int x, int y, int width, int height) {
        // render a frame
        RenderSystem.enableDepthTest();
        setupCamera(getPositionedRect(x, y, width, height));

        drawWorld();

        Vector3f hitPos = this.lastHit == null ? unProject(mouseX, mouseY) : this.lastHit;
        BlockHitResult result = rayTrace(hitPos);

        resetCamera();

        return result;
    }

    /***
     * For better performance, You'd better do project in {@link #setAfterWorldRender(Consumer)}
     * @param pos BlockPos
     * @param depth should pass Depth Test
     * @return x, y, z
     */
    protected Vector3f blockPos2ScreenPos(BlockPos pos, boolean depth, int x, int y, int width, int height){
        // render a frame
        RenderSystem.enableDepthTest();
        setupCamera(getPositionedRect(x, y, width, height));

        drawWorld();
        Vector3f winPos = project(new Vector3f(pos.m_123341_() + 0.5f, pos.m_123342_() + 0.5f, pos.m_123343_() + 0.5f));

        resetCamera();

        return winPos;
    }

    public static class VertexConsumerWrapper implements VertexConsumer {
        final VertexConsumer builder;
        @Setter
        double offsetX, offsetY, offsetZ;
        float r = 1, g = 1, b = 1, a = 1;

        public VertexConsumerWrapper(VertexConsumer builder) {
            this.builder = builder;
        }

        public void addOffset(double offsetX, double offsetY, double offsetZ) {
            this.offsetX += offsetX;
            this.offsetY += offsetY;
            this.offsetZ += offsetZ;
        }

        public void setColor(float r, float g, float b, float a) {
            this.r = r;
            this.g = g;
            this.b = b;
            this.a = a;
        }

        public void clerOffset() {
            this.offsetX = 0;
            this.offsetY = 0;
            this.offsetZ = 0;
        }

        public void clearColor() {
            this.r = 1;
            this.g = 1;
            this.b = 1;
            this.a = 1;
        }

        @Override
        public VertexConsumer m_5483_(double x, double y, double z) {
            return builder.m_5483_(x + offsetX, y + offsetY, z + offsetZ);
        }

        @Override
        public VertexConsumer m_6122_(int red, int green, int blue, int alpha) {
            return builder.m_6122_((int)(red * r), (int)(green * g), (int)(blue * b), (int)(alpha * a));
        }

        @Override
        public VertexConsumer m_7421_(float u, float v) {
            return builder.m_7421_(u, v);
        }

        @Override
        public VertexConsumer m_7122_(int u, int v) {
            return builder.m_7122_(u, v);
        }

        @Override
        public VertexConsumer m_7120_(int u, int v) {
            return builder.m_7120_(u, v);
        }

        @Override
        public VertexConsumer m_5601_(float x, float y, float z) {
            return builder.m_5601_(x, y, z);
        }

        @Override
        public void m_5752_() {
            builder.m_5752_();
        }

        @Override
        public void m_7404_(int defaultR, int defaultG, int defaultB, int defaultA) {
            builder.m_7404_(defaultR, defaultG, defaultB, defaultA);
        }

        @Override
        public void m_141991_() {
            builder.m_141991_();
        }
    }
}
