package com.lowdragmc.lowdraglib.client.utils;

import com.mojang.blaze3d.platform.Window;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.*;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.renderer.GameRenderer;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.util.Mth;
import org.joml.Matrix4f;
import org.joml.Quaternionf;
import org.joml.Vector3f;
import org.joml.Vector4f;
import org.lwjgl.opengl.GL11;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Stack;

@Environment(EnvType.CLIENT)
public class RenderUtils {

    private static final Stack<int[]> scissorFrameStack = new Stack<>();

    @Deprecated
    public static void useScissor(int x, int y, int width, int height, Runnable codeBlock) {
        pushScissorFrame(x, y, width, height);
        try {
            codeBlock.run();
        } finally {
            popScissorFrame();
        }
    }

    /**
     * use a scissor for the current screen. it maintains a stack for deeper scissors.
     * @param poseStack current stack state
     * @param x screen pos x
     * @param y screen pos y
     * @param width screen pos width
     * @param height screen pos height
     * @param codeBlock inner rendering logic
     */
    @Deprecated
    public static void useScissor(@Nonnull PoseStack poseStack, int x, int y, int width, int height, Runnable codeBlock) {
        var pose = poseStack.m_85850_().m_252922_();
        Vector4f pos = pose.transform(new Vector4f(x, y, 0, 1.0F));
        Vector4f size = pose.transform(new Vector4f(x + width, y + height, 0, 1.0F));

        x = (int) pos.x();
        y = (int) pos.y();
        width = (int) (size.x() - x);
        height = (int) (size.y() - y);

        pushScissorFrame(x, y, width, height);
        try {
            codeBlock.run();
        } finally {
            popScissorFrame();
        }
    }

    private static int[] peekFirstScissorOrFullScreen() {
        int[] currentTopFrame = scissorFrameStack.isEmpty() ? null : scissorFrameStack.peek();
        if (currentTopFrame == null) {
            Window window = Minecraft.m_91087_().m_91268_();
            return new int[]{0, 0, window.m_85441_(), window.m_85442_()};
        }
        return currentTopFrame;
    }

    private static void pushScissorFrame(int x, int y, int width, int height) {
        int[] parentScissor = peekFirstScissorOrFullScreen();
        int parentX = parentScissor[0];
        int parentY = parentScissor[1];
        int parentWidth = parentScissor[2];
        int parentHeight = parentScissor[3];

        boolean pushedFrame = false;
        if (x <= parentX + parentWidth && y <= parentY + parentHeight) {
            int newX = Math.max(x, parentX);
            int newY = Math.max(y, parentY);
            int newWidth = width - (newX - x);
            int newHeight = height - (newY - y);
            if (newWidth > 0 && newHeight > 0) {
                int maxWidth = parentWidth - (x - parentX);
                int maxHeight = parentHeight - (y - parentY);
                newWidth = Math.min(maxWidth, newWidth);
                newHeight = Math.min(maxHeight, newHeight);
                applyScissor(newX, newY, newWidth, newHeight);
                //finally, push applied scissor on top of scissor stack
                if (scissorFrameStack.isEmpty()) {
                    GL11.glEnable(GL11.GL_SCISSOR_TEST);
                }
                scissorFrameStack.push(new int[]{newX, newY, newWidth, newHeight});
                pushedFrame = true;
            }
        }
        if (!pushedFrame) {
            if (scissorFrameStack.isEmpty()) {
                GL11.glEnable(GL11.GL_SCISSOR_TEST);
            }
            scissorFrameStack.push(new int[]{parentX, parentY, parentWidth, parentHeight});
        }
    }

    private static void popScissorFrame() {
        scissorFrameStack.pop();
        int[] parentScissor = peekFirstScissorOrFullScreen();
        int parentX = parentScissor[0];
        int parentY = parentScissor[1];
        int parentWidth = parentScissor[2];
        int parentHeight = parentScissor[3];
        applyScissor(parentX, parentY, parentWidth, parentHeight);
        if (scissorFrameStack.isEmpty()) {
            GL11.glDisable(GL11.GL_SCISSOR_TEST);
        }
    }

    private static void applyScissor(int x, int y, int w, int h) {
        //translate upper-left to bottom-left
        Window window = Minecraft.m_91087_().m_91268_();
        double s = window.m_85449_();
        int translatedY = window.m_85446_() - y - h;
        GL11.glScissor((int)(x * s), (int)(translatedY * s), (int)(w * s), (int)(h * s));
    }

    /***
     * used to render pixels in stencil mask. (e.g. Restrict rendering results to be displayed only in Monitor Screens)
     * if you want to do the similar things in Gui(2D) not World(3D), plz consider using the {@link #useScissor(int, int, int, int, Runnable)}
     * that you don't need to draw mask to build a rect mask easily.
     * @param mask draw mask
     * @param renderInMask rendering in the mask
     * @param shouldRenderMask should mask be rendered too
     */
    public static void useStencil(Runnable mask, Runnable renderInMask, boolean shouldRenderMask) {
        GL11.glStencilMask(0xFF);
        GL11.glClearStencil(0);
        GL11.glClear(GL11.GL_STENCIL_BUFFER_BIT);
        GL11.glEnable(GL11.GL_STENCIL_TEST);

        GL11.glStencilFunc(GL11.GL_ALWAYS, 1, 0xFF);
        GL11.glStencilOp(GL11.GL_KEEP, GL11.GL_KEEP, GL11.GL_REPLACE);

        if (!shouldRenderMask) {
            GL11.glColorMask(false, false, false, false);
            GL11.glDepthMask(false);
        }

        mask.run();

        if (!shouldRenderMask) {
            GL11.glColorMask(true, true, true, true);
            GL11.glDepthMask(true);
        }

        GL11.glStencilMask(0x00);
        GL11.glStencilFunc(GL11.GL_EQUAL, 1, 0xFF);
        GL11.glStencilOp(GL11.GL_KEEP, GL11.GL_KEEP, GL11.GL_KEEP);

        renderInMask.run();

        GL11.glDisable(GL11.GL_STENCIL_TEST);
    }

    public static void renderBlockOverLay(@Nonnull PoseStack poseStack, BlockPos pos, float r, float g, float b, float scale) {
        if (pos == null) return;
        RenderSystem.enableBlend();
        RenderSystem.blendFunc(GL11.GL_SRC_ALPHA, GL11.GL_ONE);

        poseStack.m_85836_();
        poseStack.m_85837_((pos.m_123341_() + 0.5), (pos.m_123342_() + 0.5), (pos.m_123343_() + 0.5));
        poseStack.m_85841_(scale, scale, scale);

        Tesselator tessellator = Tesselator.m_85913_();
        BufferBuilder buffer = tessellator.m_85915_();
        RenderSystem.setShader(GameRenderer::m_172811_);
        buffer.m_166779_(VertexFormat.Mode.QUADS, DefaultVertexFormat.f_85815_);
        RenderUtils.renderCubeFace(poseStack, buffer, -0.5f, -0.5f, -0.5f, 0.5f, 0.5f, 0.5f, r, g, b, 1);
        tessellator.m_85914_();

        poseStack.m_85849_();

        RenderSystem.blendFunc(GL11.GL_SRC_ALPHA, GL11.GL_ONE_MINUS_SRC_ALPHA);
        RenderSystem.setShaderColor(1, 1, 1, 1);
    }

    public static void renderCubeFace(PoseStack poseStack, BufferBuilder buffer, float minX, float minY, float minZ, float maxX, float maxY, float maxZ, float r, float g, float b, float a) {
        Matrix4f mat = poseStack.m_85850_().m_252922_();
        buffer.m_252986_(mat, minX, minY, minZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, minX, minY, maxZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, minX, maxY, maxZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, minX, maxY, minZ).m_85950_(r, g, b, a).m_5752_();

        buffer.m_252986_(mat, maxX, minY, minZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, maxX, maxY, minZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, maxX, maxY, maxZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, maxX, minY, maxZ).m_85950_(r, g, b, a).m_5752_();

        buffer.m_252986_(mat, minX, minY, minZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, maxX, minY, minZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, maxX, minY, maxZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, minX, minY, maxZ).m_85950_(r, g, b, a).m_5752_();

        buffer.m_252986_(mat, minX, maxY, minZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, minX, maxY, maxZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, maxX, maxY, maxZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, maxX, maxY, minZ).m_85950_(r, g, b, a).m_5752_();

        buffer.m_252986_(mat, minX, minY, minZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, minX, maxY, minZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, maxX, maxY, minZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, maxX, minY, minZ).m_85950_(r, g, b, a).m_5752_();

        buffer.m_252986_(mat, minX, minY, maxZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, maxX, minY, maxZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, maxX, maxY, maxZ).m_85950_(r, g, b, a).m_5752_();
        buffer.m_252986_(mat, minX, maxY, maxZ).m_85950_(r, g, b, a).m_5752_();
    }

    public static void moveToFace(PoseStack poseStack, double x, double y, double z, Direction face) {
        poseStack.m_85837_(x + 0.5 + face.m_122429_() * 0.5, y + 0.5 + face.m_122430_() * 0.5, z + 0.5 + face.m_122431_() * 0.5);
    }

    public static void rotateToFace(PoseStack poseStack, Direction face, @Nullable Direction spin) {
        float angle = spin == Direction.EAST ? Mth.f_144831_ : spin == Direction.SOUTH ? Mth.f_144830_ : spin == Direction.WEST ? -Mth.f_144831_ : 0;
        switch (face) {
            case UP -> {
                poseStack.m_85841_(1.0f, -1.0f, 1.0f);
                poseStack.m_252781_(new Quaternionf().rotateAxis(Mth.f_144831_, new Vector3f(1, 0, 0)));
                poseStack.m_252781_(new Quaternionf().rotateAxis(angle, new Vector3f(0, 0, 1)));
            }
            case DOWN -> {
                poseStack.m_85841_(1.0f, -1.0f, 1.0f);
                poseStack.m_252781_(new Quaternionf().rotateAxis(-Mth.f_144831_, new Vector3f(1, 0, 0)));
                poseStack.m_252781_(new Quaternionf().rotateAxis(spin == Direction.EAST ? Mth.f_144831_ : spin == Direction.NORTH ? Mth.f_144830_ : spin == Direction.WEST ? -Mth.f_144831_ : 0, new Vector3f(0, 0, 1)));
            }
            case EAST -> {
                poseStack.m_85841_(-1.0f, -1.0f, -1.0f);
                poseStack.m_252781_(new Quaternionf().rotateAxis(-Mth.f_144831_, new Vector3f(0, 1, 0)));
                poseStack.m_252781_(new Quaternionf().rotateAxis(angle, new Vector3f(0, 0, 1)));
            }
            case WEST -> {
                poseStack.m_85841_(-1.0f, -1.0f, -1.0f);
                poseStack.m_252781_(new Quaternionf().rotateAxis(Mth.f_144831_, new Vector3f(0, 1, 0)));
                poseStack.m_252781_(new Quaternionf().rotateAxis(angle, new Vector3f(0, 0, 1)));
            }
            case NORTH -> {
                poseStack.m_85841_(-1.0f, -1.0f, -1.0f);
                poseStack.m_252781_(new Quaternionf().rotateAxis(angle, new Vector3f(0, 0, 1)));
            }
            case SOUTH -> {
                poseStack.m_85841_(-1.0f, -1.0f, -1.0f);
                poseStack.m_252781_(new Quaternionf().rotateAxis(Mth.f_144830_, new Vector3f(0, 1, 0)));
                poseStack.m_252781_(new Quaternionf().rotateAxis(angle, new Vector3f(0, 0, 1)));
            }
            default -> {
            }
        }
    }
}
