package com.lowdragmc.lowdraglib.gui.compass.component.animation;

import ;
import F;
import I;
import com.lowdragmc.lowdraglib.client.scene.ISceneBlockRenderHook;
import com.lowdragmc.lowdraglib.client.scene.ISceneEntityRenderHook;
import com.lowdragmc.lowdraglib.client.scene.WorldSceneRenderer;
import com.lowdragmc.lowdraglib.client.utils.RenderUtils;
import com.lowdragmc.lowdraglib.gui.animation.Transform;
import com.lowdragmc.lowdraglib.gui.compass.CompassManager;
import com.lowdragmc.lowdraglib.gui.compass.component.CompassComponent;
import com.lowdragmc.lowdraglib.gui.editor.ColorPattern;
import com.lowdragmc.lowdraglib.gui.editor.Icons;
import com.lowdragmc.lowdraglib.gui.texture.IGuiTexture;
import com.lowdragmc.lowdraglib.gui.texture.ItemStackTexture;
import com.lowdragmc.lowdraglib.gui.texture.ProgressTexture;
import com.lowdragmc.lowdraglib.gui.texture.ResourceTexture;
import com.lowdragmc.lowdraglib.gui.util.ClickData;
import com.lowdragmc.lowdraglib.gui.util.DrawerHelper;
import com.lowdragmc.lowdraglib.gui.widget.*;
import com.lowdragmc.lowdraglib.utils.*;
import com.lowdragmc.lowdraglib.utils.XmlUtils.SizedIngredient;
import com.lowdragmc.lowdraglib.utils.interpolate.Eases;
import com.mojang.blaze3d.platform.Window;
import com.mojang.blaze3d.vertex.PoseStack;
import com.mojang.blaze3d.vertex.Tesselator;
import lombok.Getter;
import lombok.val;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.player.RemotePlayer;
import net.minecraft.client.renderer.MultiBufferSource;
import net.minecraft.client.renderer.RenderType;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.network.chat.Component;
import net.minecraft.util.Mth;
import net.minecraft.util.Tuple;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.phys.Vec2;
import net.minecraft.world.phys.Vec3;
import org.apache.commons.lang3.tuple.MutableTriple;
import org.jetbrains.annotations.NotNull;
import org.joml.Vector3f;

import javax.annotation.Nullable;
import java.util.*;
import java.util.Map.Entry;

/**
 * @author KilaBash
 * @date 2023/7/28
 * @implNote AnimationScene
 */
@Environment(EnvType.CLIENT)
public class CompassScene extends WidgetGroup implements ISceneBlockRenderHook, ISceneEntityRenderHook {
    @Getter
    protected final SceneWidget sceneWidget;
    @Getter
    protected final WidgetGroup headerGroup;
    @Getter
    protected final int minX, minY = 0, minZ, maxX, maxY = 0, maxZ;
    @Getter
    protected final List<AnimationFrame> frames;
    protected final boolean useScene, tickScene;
    protected TrackedDummyWorld world;
    //runtime
    private final Map<BlockPos, Tuple<BlockAnima, Integer>> addedBlocks = new HashMap<>();
    private final Map<BlockPos, Tuple<BlockAnima, Integer>> removedBlocks = new HashMap<>();
    private final Map<BlockPosFace, Integer> highlightBlocks = new HashMap<>();
    private final Map<Vec3, MutableTriple<Tuple<XmlUtils.SizedIngredient, List<Component>>, Vec2, Integer>> tooltipBlocks = new HashMap<>();
    private final Map<Vec3, Vec2> tooltipPos = new HashMap<>();
    private int currentFrame = -1;
    private int frameTick = 0;
    private boolean isPause;

    public CompassScene(int width, CompassComponent component) {
        super(0, 0, width, component.getHeight());
        this.frames = component.getFrames();
        this.useScene = component.isUseScene();
        this.tickScene = component.isTickScene();
        this.minX = -component.getRange();
        this.minZ = -component.getRange();
        this.maxX = component.getRange();
        this.maxZ = component.getRange();
        var height = component.getHeight();
        var headerHeight = 80;
        var sceneHeight = height - headerHeight;
        if (useScene) {
            int sw = (sceneHeight * 4);
            sceneWidget = new SceneWidget((width - sw) / 2, headerHeight, sw, sceneHeight, world = new TrackedDummyWorld());
            sceneWidget.setHoverTips(true)
                    .useOrtho(component.isOrtho())
                    .setOrthoRange(0.5f)
                    .setScalable(component.isScalable())
                    .setDraggable(false)
                    .setRenderFacing(false)
                    .setRenderSelect(false);

            sceneWidget.getRenderer().setFov(30);
            sceneWidget.setRenderedCore(List.of(BlockPos.f_121853_), this);
            sceneWidget.getRenderer().setSceneEntityRenderHook(this);
            if (component.getZoom() > 0) {
                sceneWidget.setZoom(component.getZoom());
            } else {
                sceneWidget.setZoom((9 * Mth.m_14116_(maxX - minX)));
            }

            sceneWidget.setBeforeWorldRender(this::renderBeforeWorld);
            sceneWidget.setAfterWorldRender(this::renderAfterWorld);
            sceneWidget.setCameraYawAndPitch(component.getYaw(), sceneWidget.getRotationPitch());
            addWidget(sceneWidget);
        } else {
            sceneWidget = null;
        }
        addWidget(headerGroup = new WidgetGroup(0, 0, width, useScene ? headerHeight : (height - 25)));
        addWidget(new ButtonWidget((width - 12) / 2 + 20, height - 20, 12, 12, Icons.REPLAY, this::replay).setHoverTexture(Icons.REPLAY.copy().setColor(ColorPattern.GREEN.color)));
        addWidget(new ButtonWidget((width - 12) / 2, height - 20, 12, 12, Icons.ROTATION, this::rotation).setHoverTexture(Icons.ROTATION.copy().setColor(ColorPattern.GREEN.color)));
        addWidget(new ButtonWidget((width - 12) / 2 + 40, height - 20, 12, 12, Icons.borderText(0, "+", -1), cd -> zoom(-1)).setHoverTexture(Icons.borderText(0, "+", ColorPattern.GREEN.color)));
        addWidget(new ButtonWidget((width - 12) / 2 - 40, height - 20, 12, 12, Icons.borderText(0, "-", -1), cd -> zoom(1)).setHoverTexture(Icons.borderText(0, "-", ColorPattern.GREEN.color)));
        addWidget(new SwitchWidget((width - 12) / 2 - 20, height - 20, 12, 12, this::playPause)
                .setSupplier(() -> isPause)
                .setTexture(Icons.PLAY_PAUSE, Icons.PLAY_PAUSE.copy().setColor(ColorPattern.GREEN.color))
                .setClientSideWidget());
        if (!frames.isEmpty()) {
            int progressWidth = (width - 2) / frames.size();
            for (int i = 0; i < frames.size(); i++) {
                val frame = frames.get(i);
                val frameIndex = i;
                addWidget(new ProgressWidget(() -> {
                    if (currentFrame < 0) return 0;
                    if (frameIndex < currentFrame) return 1;
                    if (frameTick > 0 && frameIndex == currentFrame) {
                        var duration = frame.getDuration();
                        return (frameTick + (isPause ? 0 : Minecraft.getInstance().getDeltaFrameTime())) / Math.max(duration, 1);
                    }
                    return 0;
                }, progressWidth * i + 1 + 1, height - 6 + 1, progressWidth - 4, 4, new ProgressTexture(IGuiTexture.EMPTY, ColorPattern.WHITE.rectTexture())));
                addWidget(new ButtonWidget(progressWidth * i + 1, height - 6, progressWidth - 2, 6, ColorPattern.WHITE.borderTexture(1), cd -> jumpFrame(frame))
                        .setHoverTooltips(frame.tooltips())
                        .setHoverTexture(ColorPattern.GREEN.borderTexture(1)));
            }
        }
    }

    private void zoom(int value) {
        if (!useScene) return;
        var zoom = (float) Mth.m_14045_(sceneWidget.getZoom() + value, 0.1, 999);
        sceneWidget.setZoom(zoom);
    }

    private void renderBeforeWorld(SceneWidget sceneWidget) {
        var graphics = new GuiGraphics(Minecraft.m_91087_(), MultiBufferSource.m_109898_(Tesselator.m_85913_().m_85915_()));
        graphics.m_280168_().m_85836_();
        RenderUtils.moveToFace(graphics.m_280168_(), (minX + maxX) / 2f, minY, (minZ + maxZ) / 2f, Direction.DOWN);
        RenderUtils.rotateToFace(graphics.m_280168_(), Direction.UP, null);
        int w = (maxX - minX) + 3;
        int h = (maxZ - minZ) + 3;
        new ResourceTexture("ldlib:textures/gui/darkened_slot.png").draw(graphics, 0, 0, w / -2f, h / -2f, w, h);
        graphics.m_280168_().m_85849_();
    }

    private void renderAfterWorld(SceneWidget sceneWidget) {
        PoseStack matrixStack = new PoseStack();
        var tick = Math.abs((Minecraft.m_91087_().m_91297_() + gui.getTickCount() % 40) - 20) / 20;
        for (Map.Entry<BlockPosFace, Integer> entry : highlightBlocks.entrySet()) {
            if (entry.getValue() <= 0) continue;
            if (entry.getKey().facing == null) {
                RenderUtils.renderBlockOverLay(matrixStack, entry.getKey().pos, 0.6f * tick, 0, 0, 1.01f);
            } else {
                sceneWidget.drawFacingBorder(matrixStack, entry.getKey(), ColorUtils.color(1 * tick, 0, 0.6f, 0));
            }
        }
        var window = Minecraft.m_91087_().m_91268_();
        for (var pos : tooltipBlocks.keySet()) {
            var result = sceneWidget.getRenderer().project(new Vector3f((float) pos.f_82479_, (float) pos.f_82480_, (float) pos.f_82481_));
            //translate gui coordinates to window's ones (y is inverted)
            var x = result.x() * window.m_85445_() / window.m_85441_();
            var y = (window.m_85442_() - result.y()) * window.m_85446_() / window.m_85442_();
            tooltipPos.put(pos, new Vec2(x, y));
        }
    }

    @Override
    public void drawInForeground(@NotNull GuiGraphics graphics, int mouseX, int mouseY, float partialTicks) {
        super.drawInForeground(graphics, mouseX, mouseY, partialTicks);
        var position = getPosition();
        var size = getSize();
        for (var entry : tooltipBlocks.entrySet()) {
            var screenPos = tooltipPos.get(entry.getKey());
            if (screenPos == null) continue;
            var tuple = entry.getValue().getLeft();
            var ingredient = tuple.m_14418_();
            var items = Arrays.stream(ingredient.ingredient().m_43908_()).map(i -> {
                var copied = i.m_41777_();
                copied.m_41764_(ingredient.count());
                return copied;
            }).toArray(ItemStack[]::new);
            var tooltips = tuple.m_14419_();
            mouseX = (int)(position.x + size.width * entry.getValue().getMiddle().f_82470_);
            mouseY = (int)(position.y + size.height * entry.getValue().getMiddle().f_82471_);
            DrawerHelper.drawLines(graphics, List.of(screenPos, new Vec2(mouseX, mouseY)), -1, -1, 0.75f);
            var componentPanelWidget = new ComponentPanelWidget(0, 0, tooltips).clickHandler(CompassManager::onComponentClick);
            componentPanelWidget.setBackground(TooltipBGTexture.INSTANCE);
            var maxWidth = 200;
            while (maxWidth < size.width && (componentPanelWidget.getSize().height == 0 || componentPanelWidget.getSize().height + mouseY < size.height * (1 - entry.getValue().getMiddle().f_82471_))) {
                componentPanelWidget.setMaxWidthLimit(maxWidth);
                maxWidth += 50;
            }
            componentPanelWidget.addSelfPosition(mouseX, mouseY);
            // check width
            maxWidth = componentPanelWidget.getSize().width;
            var maxHeight = componentPanelWidget.getSize().height;
            int rightSpace = getGui().getScreenWidth() - mouseX;
            int bottomSpace = getGui().getScreenHeight() - mouseY;
            if (rightSpace < maxWidth) { // move to Left
                componentPanelWidget.addSelfPosition(rightSpace - maxWidth, 0);
            }
            // check height
            if (bottomSpace < maxHeight) {
                componentPanelWidget.addSelfPosition(0, bottomSpace - maxHeight);
            }
            graphics.m_280168_().m_85836_();
            graphics.m_280168_().m_252880_(0, 0, 200);
            componentPanelWidget.drawInBackground(graphics, mouseX, mouseY, partialTicks);
            if (items.length > 0) {
                new ItemStackTexture(items).draw(graphics, mouseX, mouseY,
                        componentPanelWidget.getPosition().x + 2,
                        componentPanelWidget.getPosition().y - 20,
                        16, 16);
            }
            graphics.m_280168_().m_85849_();
        }
    }

    private void playPause(ClickData clickData, boolean isPressed) {
        isPause = isPressed;
    }

    private void rotation(ClickData clickData) {
        if (!useScene) return;
        float current = sceneWidget.getRotationPitch();
        sceneWidget.setCameraYawAndPitchAnima(sceneWidget.getRotationYaw(), current + 90, 20);
    }

    private void resetScene() {
        world.clear();
        sceneWidget.setCameraYawAndPitch(sceneWidget.getRotationYaw(), -135);
        sceneWidget.getCore().clear();
        sceneWidget.getCore().add(BlockPos.f_121853_);
        headerGroup.clearAllWidgets();
        addedBlocks.clear();
        removedBlocks.clear();
        highlightBlocks.clear();
        tooltipBlocks.clear();
        currentFrame = -1;
    }

    private void jumpFrame(AnimationFrame frame) {
        resetScene();
        currentFrame = 0;
        for (AnimationFrame animationFrame : frames) {
            if (animationFrame != frame) {
                animationFrame.performFrameResult(this);
                currentFrame++;
            } else {
                frameTick = 0;
                break;
            }
        }
    }

    private void replay(ClickData clickData) {
        resetScene();
        isPause = false;
    }

    @Override
    public void updateScreen() {
        super.updateScreen();
        if (frames.isEmpty()) return;
        // update frames
        if (!isPause) {
            addedBlocks.forEach((p, v) -> v.m_145025_(v.m_14419_() - 1));
            removedBlocks.forEach((p, v) -> v.m_145025_(v.m_14419_() - 1));
            highlightBlocks.replaceAll((p, v) -> v - 1);
            tooltipBlocks.forEach((p, v) -> v.setRight(v.getRight() - 1));

            addedBlocks.entrySet().removeIf(e -> e.getValue().m_14419_() <= 0);
            highlightBlocks.entrySet().removeIf(e -> e.getValue() <= 0);
            tooltipBlocks.entrySet().removeIf(e -> e.getValue().getRight() <= 0);
            final var iterator = removedBlocks.entrySet().iterator();
            while (iterator.hasNext()) {
                final var entry = iterator.next();
                if (entry.getValue().m_14419_() <= 0) {
                    world.removeBlock(entry.getKey());
                    sceneWidget.getCore().remove(entry.getKey());
                    iterator.remove();
                }
            }
            if (currentFrame < 0) {
                nextFrame();
            }
            if (currentFrame >= 0 && currentFrame < frames.size()) {
                if (frames.get(currentFrame).onFrameTick(this, frameTick++)) {
                    nextFrame();
                }
            }

            if (tickScene && useScene) {
                world.tickWorld();
            }
        }
    }

    protected void nextFrame() {
        if (currentFrame < 0) {
            currentFrame = 0;
        } else {
            currentFrame++;
            if (currentFrame >= frames.size()) {
                currentFrame = -1;
                frameTick = 0;
                isPause = true;
                return;
            }
        }
        frameTick = -frames.get(currentFrame).delay();
    }

    @Override
    public void applyEntity(Level world, Entity entity, PoseStack poseStack, float partialTicks) {
        ISceneEntityRenderHook.super.applyEntity(world, entity, poseStack, partialTicks);
    }

    @Override
    public void applyBESR(Level world, BlockPos pos, BlockEntity blockEntity, PoseStack poseStack, float partialTicks) {
        if (isPause) partialTicks = 0;
        if (removedBlocks.containsKey(pos)) {
            var tuple = removedBlocks.get(pos);
            var anima = tuple.m_14418_();
            var tick = 1 - ((tuple.m_14419_() - partialTicks) / anima.duration());
            if (tick > 0) {
                poseStack.m_85837_(tick * anima.offset().f_82479_, tick * anima.offset().f_82480_, tick * anima.offset().f_82481_);
            }
        } else if (addedBlocks.containsKey(pos)) {
            var tuple = addedBlocks.get(pos);
            var anima = tuple.m_14418_();
            var tick = (tuple.m_14419_() - partialTicks) / anima.duration();
            if (tick > 0) {
                tick = Eases.EaseQuadIn.getInterpolation(tick);
                poseStack.m_85837_(tick * anima.offset().f_82479_, tick * anima.offset().f_82480_, tick * anima.offset().f_82481_);
            }
        }
    }

    @Override
    public void applyVertexConsumerWrapper(Level world, BlockPos pos, BlockState state, WorldSceneRenderer.VertexConsumerWrapper wrapperBuffer, RenderType layer, float partialTicks) {
        if (isPause) partialTicks = 0;
        if (removedBlocks.containsKey(pos)) {
            var tuple = removedBlocks.get(pos);
            var anima = tuple.m_14418_();
            var tick = 1 - ((tuple.m_14419_() - partialTicks) / anima.duration());
            if (tick > 0) {
                wrapperBuffer.addOffset(tick * anima.offset().f_82479_, tick * anima.offset().f_82480_, tick * anima.offset().f_82481_);
            }
        } else if (addedBlocks.containsKey(pos)) {
            var tuple = addedBlocks.get(pos);
            var anima = tuple.m_14418_();
            var tick = (tuple.m_14419_() - partialTicks) / anima.duration();
            if (tick > 0) {
                tick = Eases.EaseQuadIn.getInterpolation(tick);
                wrapperBuffer.addOffset(tick * anima.offset().f_82479_, tick * anima.offset().f_82480_, tick * anima.offset().f_82481_);
            }
        }
    }

    public void addBlock(BlockPos pos, BlockInfo blockInfo, @Nullable BlockAnima anima) {
        if (!useScene) return;
        if (blockInfo.getBlockState().m_60734_() == Blocks.f_50016_) {
            removeBlock(pos, null);
            return;
        }
        world.addBlock(pos, blockInfo);
        sceneWidget.getCore().add(pos);
        if (anima != null) {
            addedBlocks.put(pos, new Tuple<>(anima, anima.duration()));
        }
    }

    public void removeBlock(BlockPos pos, @Nullable BlockAnima anima) {
        if (!useScene) return;
        if (anima != null) {
            removedBlocks.put(pos, new Tuple<>(anima, anima.duration()));
        } else {
            world.removeBlock(pos);
            sceneWidget.getCore().remove(pos);
        }
    }

    public void highlightBlock(BlockPosFace key, int duration) {
        if (!useScene) return;
        highlightBlocks.put(key, duration);
    }

    public void addEntity(EntityInfo entityInfo, @Nullable Vec3 pos, boolean update) {
        Entity entity = null;
        if (update) {
            entity = world.entities.get(entityInfo.getId());
        }
        entity = (entity == null && entityInfo.getEntityType() != null) ? entityInfo.getEntityType().create(world) : entity;
        if (entity == null && entityInfo.getEntityType() == EntityType.f_20532_) {
            entity = new RemotePlayer(world.getAsClientWorld().get(), Minecraft.m_91087_().f_91074_.m_36316_());
        }
        if (entity != null) {
            entity.m_20234_(entityInfo.getId());
            if (pos != null) {
                entity.m_6034_(pos.f_82479_, pos.f_82480_, pos.f_82481_);
            }
            if (entityInfo.getTag() != null) {
                entity.m_20258_(entity.m_20240_(new CompoundTag()).m_6426_().m_128391_(entityInfo.getTag()));
            }
            world.m_7967_(entity);
        }
    }

    public void removeEntity(EntityInfo entityInfo, boolean force) {
        if (force) {
            world.entities.remove(entityInfo.getId());
        } else {
            Optional.ofNullable(world.entities.get(entityInfo.getId())).ifPresent(Entity::m_146870_);
        }
    }


    public void addTooltip(Vec3 pos, Tuple<XmlUtils.SizedIngredient, List<Component>> tuple, Vec2 middle, Integer time) {
        tooltipBlocks.put(pos, MutableTriple.of(tuple, middle, time));
    }

    public void rotate(float rotation, boolean anima) {
        if (!useScene) return;
        var current = sceneWidget.getRotationPitch();
        current += (rotation - 135 - current) % 360;
        if (anima) {
            sceneWidget.setCameraYawAndPitchAnima(sceneWidget.getRotationYaw(), current, 20);
        } else {
            sceneWidget.setCameraYawAndPitch(sceneWidget.getRotationYaw(), current);
        }
    }

    public void addInformation(Widget widget, boolean anima) {
        if (anima) {
            for (Widget child : headerGroup.widgets) {
                headerGroup.removeWidgetAnima(child, new Transform().duration(500).offset(-child.getSelfPosition().x - child.getSize().width, 0));
            }
        } else {
            headerGroup.clearAllWidgets();
        }

        var size = headerGroup.getSize();
        widget.setSelfPosition(new Position((size.width - widget.getSize().width) / 2, (size.height - widget.getSize().height) / 2));
        if (anima) {
            headerGroup.addWidgetAnima(widget, new Transform().ease(Eases.EaseQuadOut).duration(500).offset(size.width - widget.getSelfPosition().x, 0));
        } else {
            headerGroup.addWidget(widget);
        }
    }
}
