package com.lowdragmc.lowdraglib.gui.util;

import F;
import com.lowdragmc.lowdraglib.LDLib;
import com.lowdragmc.lowdraglib.Platform;
import com.lowdragmc.lowdraglib.client.shader.Shaders;
import com.lowdragmc.lowdraglib.client.shader.management.ShaderProgram;
import com.lowdragmc.lowdraglib.client.shader.uniform.UniformCache;
import com.lowdragmc.lowdraglib.client.utils.RenderBufferUtils;
import com.lowdragmc.lowdraglib.side.fluid.FluidHelper;
import com.lowdragmc.lowdraglib.side.fluid.FluidStack;
import com.lowdragmc.lowdraglib.utils.ColorUtils;
import com.lowdragmc.lowdraglib.utils.Position;
import com.lowdragmc.lowdraglib.utils.Rect;
import com.mojang.blaze3d.platform.GlStateManager;
import com.mojang.blaze3d.platform.Window;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.*;
import dev.architectury.injectables.annotations.ExpectPlatform;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.Util;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.Font;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.screens.Screen;
import net.minecraft.client.gui.screens.inventory.tooltip.ClientTooltipComponent;
import net.minecraft.client.renderer.GameRenderer;
import net.minecraft.client.renderer.texture.MissingTextureAtlasSprite;
import net.minecraft.client.renderer.texture.TextureAtlas;
import net.minecraft.client.renderer.texture.TextureAtlasSprite;
import net.minecraft.network.chat.Component;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.inventory.InventoryMenu;
import net.minecraft.world.inventory.tooltip.TooltipComponent;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.phys.Vec2;
import org.joml.Matrix4f;
import org.joml.Vector4f;
import org.lwjgl.opengl.GL11;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;

@Environment(EnvType.CLIENT)
public class DrawerHelper {

    public static ShaderProgram ROUND;
    public static ShaderProgram PANEL_BG;
    public static ShaderProgram ROUND_BOX;
    public static ShaderProgram PROGRESS_ROUND_BOX;
    public static ShaderProgram FRAME_ROUND_BOX;
    public static ShaderProgram ROUND_LINE;

    public static void init() {
        ROUND = Util.m_137469_(new ShaderProgram(), program
                -> program.attach(Shaders.ROUND_F).attach(Shaders.SCREEN_V));
        PANEL_BG = Util.m_137469_(new ShaderProgram(), program
                -> program.attach(Shaders.PANEL_BG_F).attach(Shaders.SCREEN_V));
        ROUND_BOX = Util.m_137469_(new ShaderProgram(), program
                -> program.attach(Shaders.ROUND_BOX_F).attach(Shaders.SCREEN_V));
        PROGRESS_ROUND_BOX = Util.m_137469_(new ShaderProgram(), program
                -> program.attach(Shaders.PROGRESS_ROUND_BOX_F).attach(Shaders.SCREEN_V));
        FRAME_ROUND_BOX = Util.m_137469_(new ShaderProgram(), program
                -> program.attach(Shaders.FRAME_ROUND_BOX_F).attach(Shaders.SCREEN_V));
        ROUND_LINE = Util.m_137469_(new ShaderProgram(), program
                -> program.attach(Shaders.ROUND_LINE_F).attach(Shaders.SCREEN_V));
    }

    public static void drawFluidTexture(@Nonnull GuiGraphics graphics, float xCoord, float yCoord, TextureAtlasSprite textureSprite, float maskTop, float maskRight, float zLevel, int fluidColor) {
        float uMin = textureSprite.m_118409_();
        float uMax = textureSprite.m_118410_();
        float vMin = textureSprite.m_118411_();
        float vMax = textureSprite.m_118412_();
        uMax = uMax - maskRight / 16f * (uMax - uMin);
        vMax = vMax - maskTop / 16f * (vMax - vMin);

        BufferBuilder buffer = Tesselator.m_85913_().m_85915_();
        RenderSystem.setShader(GameRenderer::m_172820_);
        buffer.m_166779_(VertexFormat.Mode.QUADS, DefaultVertexFormat.f_85819_);
        var mat = graphics.m_280168_().m_85850_().m_252922_();
        buffer.m_252986_(mat, xCoord, yCoord + 16, zLevel).m_7421_(uMin, vMax).m_193479_(fluidColor).m_5752_();
        buffer.m_252986_(mat, xCoord + 16 - maskRight, yCoord + 16, zLevel).m_7421_(uMax, vMax).m_193479_(fluidColor).m_5752_();
        buffer.m_252986_(mat, xCoord + 16 - maskRight, yCoord + maskTop, zLevel).m_7421_(uMax, vMin).m_193479_(fluidColor).m_5752_();
        buffer.m_252986_(mat, xCoord, yCoord + maskTop, zLevel).m_7421_(uMin, vMin).m_193479_(fluidColor).m_5752_();

        BufferUploader.m_231202_(buffer.m_231175_());
    }

    @Deprecated(since = "1.21")
    public static void drawFluidForGui(@Nonnull GuiGraphics graphics, FluidStack contents, long tankCapacity, int startX, int startY, int widthT, int heightT) {
        drawFluidForGui(graphics, contents, startX, startY, widthT, heightT);
    }

    public static void drawFluidForGui(@Nonnull GuiGraphics graphics, FluidStack contents, float startX, float startY, float widthT, float heightT) {
        ResourceLocation LOCATION_BLOCKS_TEXTURE = InventoryMenu.f_39692_;
        TextureAtlasSprite fluidStillSprite = FluidHelper.getStillTexture(contents);
        if (fluidStillSprite == null) {
            fluidStillSprite = Minecraft.m_91087_().m_91258_(TextureAtlas.f_118259_).apply(MissingTextureAtlasSprite.m_118071_());
            if (Platform.isDevEnv()) {
                LDLib.LOGGER.error("Missing fluid texture for fluid: " + contents.getDisplayName().getString());
            }
        }
        int fluidColor = FluidHelper.getColor(contents) | 0xff000000;
        RenderSystem.enableBlend();
        RenderSystem.setShaderTexture(0, LOCATION_BLOCKS_TEXTURE);

        final int xTileCount = (int) (widthT / 16);
        final float xRemainder = widthT - xTileCount * 16;
        final int yTileCount = (int) (heightT / 16);
        final float yRemainder = heightT - yTileCount * 16;

        final float yStart = startY + heightT;

        for (int xTile = 0; xTile <= xTileCount; xTile++) {
            for (int yTile = 0; yTile <= yTileCount; yTile++) {
                float width = xTile == xTileCount ? xRemainder : 16;
                float height = yTile == yTileCount ? yRemainder : 16;
                float x = startX + xTile * 16;
                float y = yStart - (yTile + 1) * 16;
                if (width > 0 && height > 0) {
                    float maskTop = 16 - height;
                    float maskRight = 16 - width;
                    drawFluidTexture(graphics, x, y, fluidStillSprite, maskTop, maskRight, 0, fluidColor);
                }
            }
        }
        RenderSystem.enableBlend();
    }

    public static void drawBorder(@Nonnull GuiGraphics graphics, int x, int y, int width, int height, int color, int border) {
        graphics.m_286007_(() -> {
            drawSolidRect(graphics,x - border, y - border, width + 2 * border, border, color);
            drawSolidRect(graphics,x - border, y + height, width + 2 * border, border, color);
            drawSolidRect(graphics,x - border, y, border, height, color);
            drawSolidRect(graphics,x + width, y, border, height, color);
        });
    }

    public static void drawStringSized(@Nonnull GuiGraphics graphics, String text, float x, float y, int color, boolean dropShadow, float scale, boolean center) {
        graphics.m_280168_().m_85836_();
        Font fontRenderer = Minecraft.m_91087_().f_91062_;
        double scaledTextWidth = center ? fontRenderer.m_92895_(text) * scale : 0.0;
        graphics.m_280168_().m_85837_(x - scaledTextWidth / 2.0, y, 0.0f);
        graphics.m_280168_().m_85841_(scale, scale, scale);
        graphics.m_280056_(fontRenderer, text, 0, 0, color, dropShadow);
        graphics.m_280168_().m_85849_();
    }

    public static void drawStringFixedCorner(@Nonnull GuiGraphics graphics, String text, float x, float y, int color, boolean dropShadow, float scale) {
        Font fontRenderer = Minecraft.m_91087_().f_91062_;
        float scaledWidth = fontRenderer.m_92895_(text) * scale;
        float scaledHeight = fontRenderer.f_92710_ * scale;
        drawStringSized(graphics, text, x - scaledWidth, y - scaledHeight, color, dropShadow, scale, false);
    }

    public static void drawText(@Nonnull GuiGraphics graphics, String text, float x, float y, float scale, int color) {
        drawText(graphics, text, x, y, scale, color, false);
    }

    public static void drawText(@Nonnull GuiGraphics graphics, String text, float x, float y, float scale, int color, boolean shadow) {
        Font fontRenderer = Minecraft.m_91087_().f_91062_;
        RenderSystem.disableBlend();
        graphics.m_280168_().m_85836_();
        graphics.m_280168_().m_85841_(scale, scale, 0f);
        float sf = 1 / scale;
        graphics.m_280056_(fontRenderer, text, (int) (x * sf), (int) (y * sf), color, shadow);
        graphics.m_280168_().m_85849_();
        RenderSystem.enableBlend();
    }

    public static void drawItemStack(@Nonnull GuiGraphics graphics, ItemStack itemStack, int x, int y, int color, @Nullable String altTxt) {
        var a = ColorUtils.alpha(color);
        var r = ColorUtils.red(color);
        var g = ColorUtils.green(color);
        var b = ColorUtils.blue(color);
        RenderSystem.setShaderColor(r, g, b, a);

        RenderSystem.enableDepthTest();
        RenderSystem.depthMask(true);

        Minecraft mc = Minecraft.m_91087_();

        graphics.m_280168_().m_85836_();
        graphics.m_280168_().m_252880_(0, 0, 232);
        graphics.m_280480_(itemStack, x, y);
        graphics.m_280302_(mc.f_91062_, itemStack, x, y, altTxt);
        graphics.m_280168_().m_85849_();

        // clear depth buffer,it may cause some rendering issues?
        RenderSystem.clear(GL11.GL_DEPTH_BUFFER_BIT, Minecraft.f_91002_);
        RenderSystem.depthMask(false);
        RenderSystem.setShaderColor(1F, 1F, 1F, 1F);
        RenderSystem.enableBlend();
        RenderSystem.disableDepthTest();
    }

    public static List<Component> getItemToolTip(ItemStack itemStack) {
        Minecraft mc = Minecraft.m_91087_();
        return Screen.m_280152_(mc, itemStack);
    }

    public static void drawSolidRect(@Nonnull GuiGraphics graphics, int x, int y, int width, int height, int color) {
        graphics.m_280509_(x, y, x + width, y + height, color);
        RenderSystem.enableBlend();
    }

    public static void drawSolidRect(@Nonnull GuiGraphics graphics, Rect rect, int color) {
        drawSolidRect(graphics, rect.left, rect.up, rect.right, rect.down, color);
    }

    public static void drawRectShadow(@Nonnull GuiGraphics graphics, int x, int y, int width, int height, int distance) {
        drawGradientRect(graphics, x + distance, y + height, width - distance, distance, 0x4f000000, 0, false);
        drawGradientRect(graphics, x + width, y + distance, distance, height - distance, 0x4f000000, 0, true);

        float startAlpha = (float) (0x4f) / 255.0F;
        RenderSystem.enableBlend();
        RenderSystem.blendFuncSeparate(GlStateManager.SourceFactor.SRC_ALPHA, GlStateManager.DestFactor.ONE_MINUS_SRC_ALPHA, GlStateManager.SourceFactor.ONE, GlStateManager.DestFactor.ZERO);
        Tesselator tesselator = Tesselator.m_85913_();
        BufferBuilder buffer = tesselator.m_85915_();
        RenderSystem.setShader(GameRenderer::m_172811_);
        buffer.m_166779_(VertexFormat.Mode.TRIANGLES, DefaultVertexFormat.f_85815_);
        x += width;
        y += height;
        Matrix4f mat = graphics.m_280168_().m_85850_().m_252922_();
        buffer.m_252986_(mat, x, y, 0).m_85950_(0, 0, 0, startAlpha).m_5752_();
        buffer.m_252986_(mat, x, y + distance, 0).m_6122_(0, 0, 0, 0).m_5752_();
        buffer.m_252986_(mat, x + distance, y + distance, 0).m_6122_(0, 0, 0, 0).m_5752_();

        buffer.m_252986_(mat, x, y, 0).m_85950_(0, 0, 0, startAlpha).m_5752_();
        buffer.m_252986_(mat, x + distance, y + distance, 0).m_6122_(0, 0, 0, 0).m_5752_();
        buffer.m_252986_(mat, x + distance, y, 0).m_6122_(0, 0, 0, 0).m_5752_();
        tesselator.m_85914_();
    }

    public static void drawGradientRect(@Nonnull GuiGraphics graphics, int x, int y, int width, int height, int startColor, int endColor) {
        drawGradientRect(graphics, x, y, width, height, startColor, endColor, false);
    }

    public static void drawGradientRect(@Nonnull GuiGraphics graphics, float x, float y, float width, float height, int startColor, int endColor, boolean horizontal) {
        float startAlpha = (float)(startColor >> 24 & 255) / 255.0F;
        float startRed   = (float)(startColor >> 16 & 255) / 255.0F;
        float startGreen = (float)(startColor >>  8 & 255) / 255.0F;
        float startBlue  = (float)(startColor       & 255) / 255.0F;
        float endAlpha   = (float)(endColor   >> 24 & 255) / 255.0F;
        float endRed     = (float)(endColor   >> 16 & 255) / 255.0F;
        float endGreen   = (float)(endColor   >>  8 & 255) / 255.0F;
        float endBlue    = (float)(endColor         & 255) / 255.0F;
        RenderSystem.enableBlend();
        RenderSystem.blendFuncSeparate(GlStateManager.SourceFactor.SRC_ALPHA, GlStateManager.DestFactor.ONE_MINUS_SRC_ALPHA, GlStateManager.SourceFactor.ONE, GlStateManager.DestFactor.ZERO);
        Matrix4f mat = graphics.m_280168_().m_85850_().m_252922_();
        Tesselator tesselator = Tesselator.m_85913_();
        BufferBuilder buffer = tesselator.m_85915_();
        RenderSystem.setShader(GameRenderer::m_172811_);
        buffer.m_166779_(VertexFormat.Mode.QUADS, DefaultVertexFormat.f_85815_);
        if (horizontal) {
            buffer.m_252986_(mat,x + width, y, 0).m_85950_(endRed, endGreen, endBlue, endAlpha).m_5752_();
            buffer.m_252986_(mat,x, y, 0).m_85950_(startRed, startGreen, startBlue, startAlpha).m_5752_();
            buffer.m_252986_(mat,x, y + height, 0).m_85950_(startRed, startGreen, startBlue, startAlpha).m_5752_();
            buffer.m_252986_(mat,x + width, y + height, 0).m_85950_(endRed, endGreen, endBlue, endAlpha).m_5752_();
            tesselator.m_85914_();
        } else {
            buffer.m_252986_(mat,x + width, y, 0).m_85950_(startRed, startGreen, startBlue, startAlpha).m_5752_();
            buffer.m_252986_(mat,x, y, 0).m_85950_(startRed, startGreen, startBlue, startAlpha).m_5752_();
            buffer.m_252986_(mat,x, y + height, 0).m_85950_(endRed, endGreen, endBlue, endAlpha).m_5752_();
            buffer.m_252986_(mat,x + width, y + height, 0).m_85950_(endRed, endGreen, endBlue, endAlpha).m_5752_();
            tesselator.m_85914_();
        }
    }

    public static void drawLines(@Nonnull GuiGraphics graphics, List<Vec2> Vec2s, int startColor, int endColor, float width) {
        Tesselator tesselator = Tesselator.m_85913_();
        BufferBuilder bufferbuilder = tesselator.m_85915_();
        RenderSystem.enableBlend();
        RenderSystem.defaultBlendFunc();
        RenderSystem.setShader(GameRenderer::m_172811_);
        bufferbuilder.m_166779_(VertexFormat.Mode.TRIANGLE_STRIP, DefaultVertexFormat.f_85815_);

        RenderBufferUtils.drawColorLines(graphics.m_280168_(), bufferbuilder, Vec2s, startColor, endColor, width);

        tesselator.m_85914_();
        RenderSystem.defaultBlendFunc();
    }

    public static void drawTextureRect(@Nonnull GuiGraphics graphics, float x, float y, float width, float height) {
        Tesselator tesselator = Tesselator.m_85913_();
        BufferBuilder buffer = tesselator.m_85915_();
        Matrix4f mat = graphics.m_280168_().m_85850_().m_252922_();
        RenderSystem.setShader(GameRenderer::m_172817_);
        buffer.m_166779_(VertexFormat.Mode.QUADS, DefaultVertexFormat.f_85817_);
        buffer.m_252986_(mat, x, y + height, 0).m_7421_(0, 0).m_5752_();
        buffer.m_252986_(mat, x + width, y + height, 0).m_7421_(1, 0).m_5752_();
        buffer.m_252986_(mat, x + width, y, 0).m_7421_(1, 1).m_5752_();
        buffer.m_252986_(mat, x, y, 0).m_7421_(0, 1).m_5752_();
        tesselator.m_85914_();
    }

    public static void updateScreenVshUniform(@Nonnull GuiGraphics graphics, UniformCache uniform) {
        var window = Minecraft.m_91087_().m_91268_();

        uniform.glUniform1F("GuiScale", (float) window.m_85449_());
        uniform.glUniform2F("ScreenSize", (float) window.m_85441_(), (float) window.m_85442_());
        uniform.glUniformMatrix4F("PoseStack",graphics.m_280168_().m_85850_().m_252922_());
        uniform.glUniformMatrix4F("ProjMat", RenderSystem.getProjectionMatrix());
    }

    public static void drawRound(@Nonnull GuiGraphics graphics, int color, float radius, Position centerPos) {
        DrawerHelper.ROUND.use(uniform -> {

            DrawerHelper.updateScreenVshUniform(graphics, uniform);

            uniform.fillRGBAColor("Color", color);

            uniform.glUniform1F("StepLength", 1f);
            uniform.glUniform1F("Radius", radius);
            uniform.glUniform2F("CenterPos", centerPos.x, centerPos.y);
        });

        RenderSystem.enableBlend();
        uploadScreenPosVertex();
    }

    public static void drawPanelBg(@Nonnull GuiGraphics graphics) {
        DrawerHelper.PANEL_BG.use(uniform -> {

            DrawerHelper.updateScreenVshUniform(graphics, uniform);

            uniform.glUniform1F("Density", 5);
            uniform.glUniform1F("SquareSize", 0.1f);
            var bg = 20f / 255f;
            uniform.glUniform4F("BgColor", bg, bg, bg, 0.95f);
            var square = 40f / 255f;
            uniform.glUniform4F("SquareColor", square, square, square, 0.95f);
        });

        RenderSystem.enableBlend();
        uploadScreenPosVertex();
    }

    public static void drawRoundBox(@Nonnull GuiGraphics graphics, Rect square, Vector4f radius, int color) {
        DrawerHelper.ROUND_BOX.use(uniform -> {
            DrawerHelper.updateScreenVshUniform(graphics, uniform);
            uniform.glUniformMatrix4F("PoseStack", new Matrix4f());
            var point1 = new Vector4f(square.left - 0.25f, square.up - 0.25f, 0, 1);
            var point2 = new Vector4f(square.right - 0.25f, square.down - 0.25f, 0, 1);
            var matrix = graphics.m_280168_().m_85850_().m_252922_();
            point1.mul(matrix);
            point2.mul(matrix);
            var v1 = matrix.transform(new Vector4f(1, 1, 1, 1));
            var v2 = matrix.transform(new Vector4f(0, 0, 0, 1));
            var scale = v1.x - v2.x; // we just use the x scale

            uniform.glUniform4F("SquareVertex", point1.x, point1.y, point2.x, point2.y);
            uniform.glUniform4F("RoundRadius", radius.x() * scale, radius.y() * scale, radius.z() * scale, radius.w() * scale);
            uniform.fillRGBAColor("Color", color);
            uniform.glUniform1F("Blur", 2);
        });

        RenderSystem.enableBlend();
        uploadScreenPosVertex();
    }

    public static void drawProgressRoundBox(@Nonnull GuiGraphics graphics, Rect square, Vector4f radius, int color1, int color2, float progress) {
        DrawerHelper.PROGRESS_ROUND_BOX.use(uniform -> {
            DrawerHelper.updateScreenVshUniform(graphics, uniform);

            uniform.glUniform4F("SquareVertex", square.left, square.up, square.right, square.down);
            uniform.glUniform4F("RoundRadius", radius.x(), radius.y(), radius.z(), radius.w());
            uniform.fillRGBAColor("Color1", color1);
            uniform.fillRGBAColor("Color2", color2);
            uniform.glUniform1F("Blur", 2);
            uniform.glUniform1F("Progress", progress);
        });

        RenderSystem.enableBlend();
        uploadScreenPosVertex();
    }

    public static void drawFrameRoundBox(@Nonnull GuiGraphics graphics, Rect square, float thickness, Vector4f radius1, Vector4f radius2, int color) {
        DrawerHelper.FRAME_ROUND_BOX.use(uniform -> {
            DrawerHelper.updateScreenVshUniform(graphics, uniform);
            uniform.glUniformMatrix4F("PoseStack", new Matrix4f());
            var point1 = new Vector4f(square.left - 0.25f, square.up - 0.25f, 0, 1);
            var point2 = new Vector4f(square.right - 0.25f, square.down - 0.25f, 0, 1);
            var matrix = graphics.m_280168_().m_85850_().m_252922_();
            point1.mul(matrix);
            point2.mul(matrix);
            var v1 = matrix.transform(new Vector4f(1, 1, 1, 1));
            var v2 = matrix.transform(new Vector4f(0, 0, 0, 1));
            var scale = v1.x - v2.x; // we just use the x scale

            uniform.glUniform4F("SquareVertex", point1.x, point1.y, point2.x, point2.y);
            uniform.glUniform4F("RoundRadius1", radius1.x() * scale, radius1.y() * scale, radius1.z() * scale, radius1.w() * scale);
            uniform.glUniform4F("RoundRadius2", radius2.x() * scale, radius2.y() * scale, radius2.z() * scale, radius2.w() * scale);
            uniform.glUniform1F("Thickness", thickness * scale);
            uniform.fillRGBAColor("Color", color);
            uniform.glUniform1F("Blur", 2);
        });

        RenderSystem.enableBlend();
        uploadScreenPosVertex();
    }

    public static void drawRoundLine(@Nonnull GuiGraphics graphics, Position begin, Position end, int width, int color1, int color2) {
        DrawerHelper.ROUND_LINE.use(uniform -> {
            DrawerHelper.updateScreenVshUniform(graphics, uniform);

            uniform.glUniform1F("Width", width);
            uniform.glUniform2F("Point1", begin.x, begin.y);
            uniform.glUniform2F("Point2", end.x, end.y);
            uniform.fillRGBAColor("Color1", color1);
            uniform.fillRGBAColor("Color2", color2);
            uniform.glUniform1F("Blur", 2);
        });

        RenderSystem.enableBlend();
        uploadScreenPosVertex();
    }

    private static void uploadScreenPosVertex() {
        var builder = Tesselator.m_85913_().m_85915_();

        builder.m_166779_(VertexFormat.Mode.QUADS, DefaultVertexFormat.f_85814_);
        builder.m_5483_(-1.0, 1.0, 0.0).m_5752_();
        builder.m_5483_(-1.0, -1.0, 0.0).m_5752_();
        builder.m_5483_(1.0, -1.0, 0.0).m_5752_();
        builder.m_5483_(1.0, 1.0, 0.0).m_5752_();
        BufferUploader.m_231209_(builder.m_231175_());
    }

    @ExpectPlatform
    public static void drawTooltip(GuiGraphics graphics, int mouseX, int mouseY, List<Component> tooltipTexts, ItemStack tooltipStack, @Nullable TooltipComponent tooltipComponent, Font tooltipFont) {
        throw new AssertionError();
    }

    @ExpectPlatform
    public static ClientTooltipComponent getClientTooltipComponent(TooltipComponent component) {
        throw new AssertionError();
    }
}
