package com.lowdragmc.lowdraglib.gui.widget;

import I;
import com.lowdragmc.lowdraglib.gui.util.ClickData;
import com.lowdragmc.lowdraglib.utils.Size;
import lombok.Getter;
import lombok.Setter;
import lombok.experimental.Accessors;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.ChatFormatting;
import net.minecraft.Util;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.Font;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.components.ComponentRenderUtils;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.network.chat.ClickEvent;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.HoverEvent;
import net.minecraft.network.chat.Style;
import net.minecraft.util.FormattedCharSequence;
import net.minecraft.world.item.ItemStack;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

/**
 * @author KilaBash
 * @date 2023/3/16
 * @implNote ComponentTextWidget
 */
@Accessors(fluent = true)
public class ComponentPanelWidget extends Widget {
    protected int maxWidthLimit;
    @Setter @Nullable
    protected Consumer<List<Component>> textSupplier;
    @Setter
    protected BiConsumer<String, ClickData> clickHandler;
    protected List<Component> lastText = new ArrayList<>();
    @Getter
    protected List<FormattedCharSequence> cacheLines = Collections.emptyList();
    protected boolean isCenter = false;
    protected int space = 2;

    public ComponentPanelWidget(int x, int y, @Nonnull Consumer<List<Component>> textSupplier) {
        super(x, y, 0, 0);
        this.textSupplier = textSupplier;
        this.textSupplier.accept(lastText);
    }

    public ComponentPanelWidget(int x, int y, List<Component> text) {
        super(x, y, 0, 0);
        this.lastText.addAll(text);
    }

    public static Component withButton(Component textComponent, String componentData) {
        var style = textComponent.m_7383_();
        style = style.m_131142_(new ClickEvent(ClickEvent.Action.OPEN_URL, "@!" + componentData));
        style = style.m_131140_(ChatFormatting.YELLOW);
        return textComponent.m_6881_().m_130948_(style);
    }

    public static Component withButton(Component textComponent, String componentData, int color) {
        var style = textComponent.m_7383_();
        style = style.m_131142_(new ClickEvent(ClickEvent.Action.OPEN_URL, "@!" + componentData));
        style = style.m_178520_(color);
        return textComponent.m_6881_().m_130948_(style);
    }

    public static Component withHoverTextTranslate(Component textComponent, Component hover) {
        Style style = textComponent.m_7383_();
        style = style.m_131144_(new HoverEvent(HoverEvent.Action.f_130831_, hover));
        return textComponent.m_6881_().m_130948_(style);
    }

    public ComponentPanelWidget setMaxWidthLimit(int maxWidthLimit) {
        this.maxWidthLimit = maxWidthLimit;
        if (isRemote()) {
            formatDisplayText();
            updateComponentTextSize();
        }
        return this;
    }

    public ComponentPanelWidget setCenter(boolean center) {
        isCenter = center;
        if (isRemote()) {
            formatDisplayText();
            updateComponentTextSize();
        }
        return this;
    }

    public ComponentPanelWidget setSpace(int space) {
        this.space = space;
        if (isRemote()) {
            formatDisplayText();
            updateComponentTextSize();
        }
        return this;
    }

    @Override
    public void writeInitialData(FriendlyByteBuf buffer) {
        super.writeInitialData(buffer);
        buffer.m_130130_(lastText.size());
        for (Component textComponent : lastText) {
            buffer.m_130083_(textComponent);
        }
    }

    @Override
    public void readInitialData(FriendlyByteBuf buffer) {
        super.readInitialData(buffer);
        readUpdateInfo(1, buffer);
    }

    @Override
    public void initWidget() {
        super.initWidget();
        if (textSupplier != null) {
            lastText.clear();
            textSupplier.accept(lastText);
        }
        if (isClientSideWidget && isRemote()) {
            formatDisplayText();
            updateComponentTextSize();
        }
    }

    @Override
    public void updateScreen() {
        super.updateScreen();
        if (isClientSideWidget && textSupplier != null) {
            List<Component> textBuffer = new ArrayList<>();
            textSupplier.accept(textBuffer);
            if (!lastText.equals(textBuffer)){
                this.lastText = textBuffer;
                formatDisplayText();
                updateComponentTextSize();
            }
        }
    }

    @Override
    public void detectAndSendChanges() {
        super.detectAndSendChanges();
        if (textSupplier != null) {
            List<Component> textBuffer = new ArrayList<>();
            textSupplier.accept(textBuffer);
            if (!lastText.equals(textBuffer)) {
                this.lastText = textBuffer;
                writeUpdateInfo(1, buffer -> {
                    buffer.m_130130_(lastText.size());
                    for (Component textComponent : lastText) {
                        buffer.m_130083_(textComponent);
                    }
                });
            }
        }
    }

    @Override
    public void readUpdateInfo(int id, FriendlyByteBuf buffer) {
        if (id == 1) {
            this.lastText.clear();
            int count = buffer.m_130242_();
            for (int i = 0; i < count; i++) {
                this.lastText.add(buffer.m_130238_());
            }
            formatDisplayText();
            updateComponentTextSize();
        }
    }

    @Override
    public void handleClientAction(int id, FriendlyByteBuf buffer) {
        if (id == 1) {
            ClickData clickData = ClickData.readFromBuf(buffer);
            String componentData = buffer.m_130277_();
            if (clickHandler != null) {
                clickHandler.accept(componentData, clickData);
            }
        } else {
            super.handleClientAction(id, buffer);
        }
    }

    @Environment(EnvType.CLIENT)
    public void updateComponentTextSize() {
        var fontRenderer = Minecraft.m_91087_().f_91062_;
        int totalHeight = cacheLines.size() * (fontRenderer.f_92710_ + space);
        if (totalHeight > 0) {
            totalHeight -= space;
        }
        if (isCenter) {
            setSize(new Size(maxWidthLimit, totalHeight));
        } else {
            int maxStringWidth = 0;
            for (var line : cacheLines) {
                maxStringWidth = Math.max(fontRenderer.m_92724_(line), maxStringWidth);
            }
            setSize(new Size(maxWidthLimit == 0 ? maxStringWidth : Math.min(maxWidthLimit, maxStringWidth), totalHeight));
        }
    }

    @Environment(EnvType.CLIENT)
    public void formatDisplayText() {
        var fontRenderer = Minecraft.m_91087_().f_91062_;
        int maxTextWidthResult = maxWidthLimit == 0 ? Integer.MAX_VALUE : maxWidthLimit;
        this.cacheLines = lastText.stream().flatMap(component ->
                ComponentRenderUtils.m_94005_(component, maxTextWidthResult, fontRenderer).stream())
                .toList();
    }

    @Environment(EnvType.CLIENT)
    @Nullable
    protected Style getStyleUnderMouse(double mouseX, double mouseY) {
        var fontRenderer = Minecraft.m_91087_().f_91062_;
        var position = getPosition();
        var size = getSize();

        var selectedLine = (mouseY - position.y) / (fontRenderer.f_92710_ + space);
        if (isCenter) {
            if (selectedLine >= 0 && selectedLine < cacheLines.size()) {
                var cacheLine = cacheLines.get((int) selectedLine);
                var lineWidth = fontRenderer.m_92724_(cacheLine);
                var offsetX = position.x + (size.width - lineWidth) / 2f;
                if (mouseX >= offsetX) {
                    var mouseOffset = (int)(mouseX - position.x);
                    return fontRenderer.m_92865_().m_92338_(cacheLine, mouseOffset);
                }
            }
        } else {
            if (mouseX >= position.x && selectedLine >= 0 && selectedLine < cacheLines.size()) {
                var cacheLine = cacheLines.get((int) selectedLine);
                var mouseOffset = (int)(mouseX - position.x);
                return fontRenderer.m_92865_().m_92338_(cacheLine, mouseOffset);
            }
        }
        return null;
    }
    
    @Override
    @Environment(EnvType.CLIENT)
    public boolean mouseClicked(double mouseX, double mouseY, int button) {
        var style = getStyleUnderMouse(mouseX, mouseY);
        if (style != null) {
            if (style.m_131182_() != null) {
                ClickEvent clickEvent = style.m_131182_();
                String componentText = clickEvent.m_130623_();
                if (clickEvent.m_130622_() == ClickEvent.Action.OPEN_URL) {
                    if (componentText.startsWith("@!")) {
                        String rawText = componentText.substring(2);
                        ClickData clickData = new ClickData();
                        if (clickHandler != null) {
                            clickHandler.accept(rawText, clickData);
                        }
                        writeClientAction(1, buf -> {
                            clickData.writeToBuf(buf);
                            buf.m_130070_(rawText);
                        });
                    } else if (componentText.startsWith("@#")) {
                        String rawText = componentText.substring(2);
                        Util.m_137581_().m_137646_(rawText);
                    }
                    playButtonClickSound();
                    return true;
                }
            }
        }
        return super.mouseClicked(mouseX, mouseY, button);
    }

    @Override
    @Environment(EnvType.CLIENT)
    public void drawInForeground(@NotNull @Nonnull GuiGraphics graphics, int mouseX, int mouseY, float partialTicks) {
        var style = getStyleUnderMouse(mouseX, mouseY);
        if (style != null) {
            if (style.m_131186_() != null) {
                var hoverEvent = style.m_131186_();
                var hoverTips = hoverEvent.m_130823_(HoverEvent.Action.f_130831_);
                if (hoverTips != null) {
                    gui.getModularUIGui().setHoverTooltip(List.of(hoverTips), ItemStack.f_41583_, null, null);
                    return;
                }
            }
        }
        super.drawInForeground(graphics, mouseX, mouseY, partialTicks);
    }

    @Override
    @Environment(EnvType.CLIENT)
    public void drawInBackground(@Nonnull GuiGraphics graphics, int mouseX, int mouseY, float partialTicks) {
        super.drawInBackground(graphics, mouseX, mouseY, partialTicks);
        var fontRenderer = Minecraft.m_91087_().f_91062_;
        var position = getPosition();
        var size = getSize();
        for (int i = 0; i < cacheLines.size(); i++) {
            var cacheLine = cacheLines.get(i);
            if (isCenter) {
                var lineWidth = fontRenderer.m_92724_(cacheLine);
                graphics.m_280648_(fontRenderer, cacheLine, position.x + (size.width - lineWidth) / 2, position.y + i * (fontRenderer.f_92710_ + space), -1);
            } else {
                graphics.m_280648_(fontRenderer, cacheLines.get(i), position.x, position.y + i * (fontRenderer.f_92710_ + 2), -1);
            }
        }
    }
}
