package com.lowdragmc.lowdraglib.gui.widget;

import com.lowdragmc.lowdraglib.gui.editor.annotation.ConfigSetter;
import com.lowdragmc.lowdraglib.gui.editor.annotation.Configurable;
import com.lowdragmc.lowdraglib.gui.editor.annotation.LDLRegister;
import com.lowdragmc.lowdraglib.gui.editor.annotation.NumberColor;
import com.lowdragmc.lowdraglib.gui.editor.configurator.IConfigurableWidget;
import com.lowdragmc.lowdraglib.utils.LocalizationUtils;
import com.lowdragmc.lowdraglib.utils.Position;
import com.lowdragmc.lowdraglib.utils.Size;
import lombok.Setter;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.Font;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.network.chat.Component;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.function.Supplier;

@Configurable(name = "ldlib.gui.editor.register.widget.label", collapse = false)
@LDLRegister(name = "label", group = "widget.basic")
public class LabelWidget extends Widget implements IConfigurableWidget {

    @Setter
    @Nonnull
    protected Supplier<String> textSupplier;

    @Nullable
    protected Component component;

    @Configurable(name = "ldlib.gui.editor.name.text")
    private String lastTextValue = "";

    @Configurable(name = "ldlib.gui.editor.name.color")
    @NumberColor
    private int color;

    @Configurable(name = "ldlib.gui.editor.name.isShadow")
    private boolean dropShadow;

    public LabelWidget() {
        this(0, 0, "label");
    }

    public LabelWidget(int xPosition, int yPosition, String text) {
        this(xPosition, yPosition, ()->text);
    }

    public LabelWidget(int xPosition, int yPosition, Component component) {
        super(new Position(xPosition, yPosition), new Size(10, 10));
        setDropShadow(true);
        setTextColor(-1);
        setComponent(component);
    }

    public LabelWidget(int xPosition, int yPosition, Supplier<String> text) {
        super(new Position(xPosition, yPosition), new Size(10, 10));
        setDropShadow(true);
        setTextColor(-1);
        setTextProvider(text);
    }

    @ConfigSetter(field = "lastTextValue")
    public void setText(String text) {
        textSupplier = () -> text;
        if (isRemote()) {
            lastTextValue = textSupplier.get();
            updateSize();
        }
    }

    public void setTextProvider(Supplier<String> textProvider) {
        textSupplier = textProvider;
        if (isRemote()) {
            lastTextValue = textSupplier.get();
            updateSize();
        }
    }

    public void setComponent(Component component) {
        this.component = component;
        if (isRemote()) {
            lastTextValue = component.getString();
            updateSize();
        }
    }

    @Deprecated
    public LabelWidget setTextColor(int color) {
        this.color = color;
        if (this.component != null) this.component = this.component.m_6881_().m_130948_(this.component.m_7383_().m_178520_(color));
        return this;
    }

    @Deprecated
    public LabelWidget setDropShadow(boolean dropShadow) {
        this.dropShadow = dropShadow;
        return this;
    }

    public void setColor(int color) {
        setTextColor(color);
    }

    @Override
    public void writeInitialData(FriendlyByteBuf buffer) {
        super.writeInitialData(buffer);
        if (!isClientSideWidget) {
            if (this.component != null) {
                buffer.writeBoolean(true);
                buffer.m_130083_(this.component);
            } else {
                buffer.writeBoolean(false);
                this.lastTextValue = textSupplier.get();
                buffer.m_130070_(lastTextValue);
            }
        } else {
            buffer.writeBoolean(false);
            buffer.m_130070_(lastTextValue);
        }
    }

    @Override
    public void readInitialData(FriendlyByteBuf buffer) {
        super.readInitialData(buffer);
        if (buffer.readBoolean()) {
            this.component = buffer.m_130238_();
            this.lastTextValue = component.getString();
        } else {
            this.lastTextValue = buffer.m_130277_();
        }
    }

    @Override
    public void detectAndSendChanges() {
        super.detectAndSendChanges();
        if (!isClientSideWidget) {
            if (this.component != null) {
                String latest = component.getString();
                if (!latest.equals(lastTextValue)) {
                    this.lastTextValue = latest;
                    writeUpdateInfo(-2, buffer -> buffer.m_130083_(this.component));
                }
            }
            String latest = textSupplier.get();
            if (!latest.equals(lastTextValue)) {
                this.lastTextValue = latest;
                writeUpdateInfo(-1, buffer -> buffer.m_130070_(this.lastTextValue));
            }
        }
    }

    @Override
    @Environment(EnvType.CLIENT)
    public void readUpdateInfo(int id, FriendlyByteBuf buffer) {
        if (id == -1) {
            this.lastTextValue = buffer.m_130277_();
            updateSize();
        } else if(id == -2) {
            this.component = buffer.m_130238_();
            this.lastTextValue = component.getString();
            updateSize();
        } else {
            super.readUpdateInfo(id, buffer);
        }
    }

    @Override
    @Environment(EnvType.CLIENT)
    public void updateScreen() {
        super.updateScreen();
        if (isClientSideWidget) {
            String latest = component == null ? textSupplier.get() : component.getString();
            if (!latest.equals(lastTextValue)) {
                this.lastTextValue = latest;
                updateSize();
            }
        }
    }

    @Environment(EnvType.CLIENT)
    private void updateSize() {
        Font fontRenderer = Minecraft.m_91087_().f_91062_;
        setSize(new Size(this.component == null ? fontRenderer.m_92895_(LocalizationUtils.format(lastTextValue)) : fontRenderer.m_92852_(this.component), fontRenderer.f_92710_));
    }

    @Override
    @Environment(EnvType.CLIENT)
    public void drawInBackground(@Nonnull GuiGraphics graphics, int mouseX, int mouseY, float partialTicks) {
        super.drawInBackground(graphics, mouseX, mouseY, partialTicks);
        Font fontRenderer = Minecraft.m_91087_().f_91062_;
        Position position = getPosition();
        if (component == null) {
            String suppliedText = LocalizationUtils.format(lastTextValue);
            String[] split = suppliedText.split("\n");
            for (int i = 0; i < split.length; i++) {
                int y = position.y + (i * (fontRenderer.f_92710_ + 2));
                graphics.m_280056_(fontRenderer, split[i], position.x, y, color, dropShadow);
            }
        } else {
            graphics.m_280614_(fontRenderer, component, position.x, position.y, color, dropShadow);
        }
    }

    @Override
    public boolean handleDragging(Object dragging) {
        if (dragging instanceof String string) {
            setText(string);
            return true;
        } else return IConfigurableWidget.super.handleDragging(dragging);
    }

}
