package com.lowdragmc.lowdraglib.gui.widget;

import com.lowdragmc.lowdraglib.LDLib;
import com.lowdragmc.lowdraglib.core.mixins.accessor.SlotAccessor;
import com.lowdragmc.lowdraglib.gui.editor.annotation.Configurable;
import com.lowdragmc.lowdraglib.gui.editor.annotation.LDLRegister;
import com.lowdragmc.lowdraglib.gui.editor.configurator.ConfiguratorGroup;
import com.lowdragmc.lowdraglib.gui.editor.configurator.IConfigurableWidget;
import com.lowdragmc.lowdraglib.gui.editor.configurator.WrapperConfigurator;
import com.lowdragmc.lowdraglib.gui.ingredient.IRecipeIngredientSlot;
import com.lowdragmc.lowdraglib.gui.modular.ModularUI;
import com.lowdragmc.lowdraglib.gui.modular.ModularUIGuiContainer;
import com.lowdragmc.lowdraglib.gui.texture.IGuiTexture;
import com.lowdragmc.lowdraglib.gui.texture.ResourceBorderTexture;
import com.lowdragmc.lowdraglib.gui.util.DrawerHelper;
import com.lowdragmc.lowdraglib.jei.IngredientIO;
import com.lowdragmc.lowdraglib.jei.JEIPlugin;
import com.lowdragmc.lowdraglib.misc.ItemStackTransfer;
import com.lowdragmc.lowdraglib.side.item.IItemTransfer;
import com.lowdragmc.lowdraglib.utils.CycleItemStackHandler;
import com.lowdragmc.lowdraglib.utils.Position;
import com.lowdragmc.lowdraglib.utils.Size;
import com.lowdragmc.lowdraglib.utils.TagOrCycleItemStackTransfer;
import com.mojang.blaze3d.platform.InputConstants;
import com.mojang.blaze3d.platform.Window;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.datafixers.util.Either;
import com.mojang.datafixers.util.Pair;
import dev.emi.emi.api.stack.EmiIngredient;
import dev.emi.emi.api.stack.EmiStack;
import dev.emi.emi.api.stack.EmiStackInteraction;
import dev.emi.emi.screen.EmiScreenManager;
import lombok.Getter;
import lombok.Setter;
import lombok.experimental.Accessors;
import me.shedaniel.rei.api.common.entry.EntryIngredient;
import me.shedaniel.rei.api.common.util.EntryIngredients;
import me.shedaniel.rei.api.common.util.EntryStacks;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.core.HolderSet;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.network.chat.Component;
import net.minecraft.tags.TagKey;
import net.minecraft.world.Container;
import net.minecraft.world.SimpleContainer;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.inventory.AbstractContainerMenu;
import net.minecraft.world.inventory.ClickType;
import net.minecraft.world.inventory.Slot;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.block.Blocks;
import org.jetbrains.annotations.NotNull;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@SuppressWarnings("UnusedReturnValue")
@LDLRegister(name = "item_slot", group = "widget.container")
@Accessors(chain = true)
public class SlotWidget extends Widget implements IRecipeIngredientSlot, IConfigurableWidget {
    public final static ResourceBorderTexture ITEM_SLOT_TEXTURE = new ResourceBorderTexture("ldlib:textures/gui/slot.png", 18, 18, 1, 1);

    @Nullable
    protected static Slot HOVER_SLOT = null;
    @Nullable
    protected Slot slotReference;
    @Configurable(name = "ldlib.gui.editor.name.canTakeItems")
    @Setter
    protected boolean canTakeItems;
    @Configurable(name = "ldlib.gui.editor.name.canPutItems")
    @Setter
    protected boolean canPutItems;
    public boolean isPlayerContainer;
    public boolean isPlayerHotBar;
    @Configurable(name = "ldlib.gui.editor.name.drawHoverOverlay")
    @Setter
    public boolean drawHoverOverlay = true;
    @Configurable(name = "ldlib.gui.editor.name.drawHoverTips")
    @Setter
    public boolean drawHoverTips = true;

    @Setter
    protected Runnable changeListener;
    @Setter
    protected BiConsumer<SlotWidget, List<Component>> onAddedTooltips;
    @Setter
    protected Function<ItemStack, ItemStack> itemHook;
    @Setter
    @Getter
    protected IngredientIO ingredientIO = IngredientIO.RENDER_ONLY;
    @Setter
    @Getter
    protected float XEIChance = 1f;
    @Getter
    private ItemStack lastItem = ItemStack.f_41583_;

    public SlotWidget() {
        super(new Position(0, 0), new Size(18, 18));
    }

    @Override
    public void initTemplate() {
        setBackgroundTexture(ITEM_SLOT_TEXTURE);
        this.canTakeItems = true;
        this.canPutItems = true;
    }

    public SlotWidget(Container inventory, int slotIndex, int xPosition, int yPosition, boolean canTakeItems, boolean canPutItems) {
        super(new Position(xPosition, yPosition), new Size(18, 18));
        setBackgroundTexture(SlotWidget.ITEM_SLOT_TEXTURE);
        this.canTakeItems = canTakeItems;
        this.canPutItems = canPutItems;
        setContainerSlot(inventory, slotIndex);
    }

    public SlotWidget(IItemTransfer itemHandler, int slotIndex, int xPosition, int yPosition, boolean canTakeItems, boolean canPutItems) {
        super(new Position(xPosition, yPosition), new Size(18, 18));
        setBackgroundTexture(SlotWidget.ITEM_SLOT_TEXTURE);
        this.canTakeItems = canTakeItems;
        this.canPutItems = canPutItems;
        setHandlerSlot(itemHandler, slotIndex);
    }

    protected Slot createSlot(Container inventory, int index) {
        return new WidgetSlot(inventory, index, 0, 0);
    }

    protected Slot createSlot(IItemTransfer itemHandler, int index) {
        return new WidgetSlotItemTransfer(itemHandler, index, 0, 0);
    }

    public SlotWidget setContainerSlot(Container inventory, int slotIndex) {
        updateSlot(createSlot(inventory, slotIndex));
        return this;
    }

    public SlotWidget setHandlerSlot(IItemTransfer itemHandler, int slotIndex) {
        updateSlot(createSlot(itemHandler, slotIndex));
        return this;
    }

    protected void updateSlot(Slot slot) {
        if (this.slotReference != null && this.gui != null && !isClientSideWidget) {
            getGui().removeNativeSlot(this.slotReference);
        }
        this.slotReference = slot;
        if (this.gui != null && !isClientSideWidget) {
            getGui().addNativeSlot(this.slotReference, this);
        }
    }

    public ItemStack getItem() {
        return slotReference == null ? ItemStack.f_41583_ : slotReference.m_7993_();
    }

    public void setItem(ItemStack stack) {
        if (slotReference != null) {
            slotReference.m_5852_(stack);
        }
    }

    public void setItem(ItemStack stack, boolean notify) {
        if (slotReference != null) {
            var lastListener = changeListener;
            if (!notify) {
                changeListener = null;
            }
            slotReference.m_5852_(stack);
            changeListener = lastListener;
        }
    }

    @Override
    public final void setSize(Size size) {
        // you cant modify size.
    }

    @Override
    public void setGui(ModularUI gui) {
        if (!isClientSideWidget && this.gui != gui) {
            if (this.gui != null && slotReference != null) {
                this.gui.removeNativeSlot(slotReference);
            }
            if (gui != null && slotReference != null) {
                gui.addNativeSlot(slotReference, this);
            }
        }
        super.setGui(gui);
    }

    @Override
    public void detectAndSendChanges() {
        super.detectAndSendChanges();
        this.lastItem = getItem();
    }

    @Override
    @Environment(EnvType.CLIENT)
    public void updateScreen() {
        super.updateScreen();
        this.lastItem = getItem();
    }

    @Override
    @Environment(EnvType.CLIENT)
    public void drawInForeground(@Nonnull GuiGraphics graphics, int mouseX, int mouseY, float partialTicks) {
        if (slotReference != null && drawHoverTips && isMouseOverElement(mouseX, mouseY) && getHoverElement(mouseX, mouseY) == this) {
            ItemStack stack = slotReference.m_7993_();
            if (gui != null) {
                gui.getModularUIGui().setHoveredSlot(slotReference);
            }
            if (!stack.m_41619_() && gui != null) {
                gui.getModularUIGui().setHoverTooltip(getFullTooltipTexts(), stack, null, stack.m_150921_().orElse(null));
            } else {
                super.drawInForeground(graphics, mouseX, mouseY, partialTicks);
            }
        } else {
            super.drawInForeground(graphics, mouseX, mouseY, partialTicks);
        }
    }

    @Override
    @Environment(EnvType.CLIENT)
    public void drawInBackground(@Nonnull GuiGraphics graphics, int mouseX, int mouseY, float partialTicks) {
        super.drawInBackground(graphics, mouseX, mouseY, partialTicks);
        Position pos = getPosition();
        if (slotReference != null) {
            ItemStack itemStack = getRealStack(slotReference.m_7993_());
            ModularUIGuiContainer modularUIGui = gui == null ? null : gui.getModularUIGui();
            if (itemStack.m_41619_() && modularUIGui != null && modularUIGui.getQuickCrafting() && modularUIGui.getQuickCraftSlots().contains(slotReference)) { // draw split
                int splitSize = modularUIGui.getQuickCraftSlots().size();
                itemStack = gui.getModularUIContainer().m_142621_();
                if (!itemStack.m_41619_() && splitSize > 1 && AbstractContainerMenu.m_38899_(slotReference, itemStack, true)) {
                    itemStack = itemStack.m_41777_();
                    itemStack.m_41769_(AbstractContainerMenu.m_278794_(modularUIGui.getQuickCraftSlots(), modularUIGui.dragSplittingLimit, itemStack));
                    int k = Math.min(itemStack.m_41741_(), slotReference.m_5866_(itemStack));
                    if (itemStack.m_41613_() > k) {
                        itemStack.m_41764_(k);
                    }
                }
            }
            if (!itemStack.m_41619_()) {
                DrawerHelper.drawItemStack(graphics, itemStack, pos.x + 1, pos.y + 1, -1, null);
            }
        }
        drawOverlay(graphics, mouseX, mouseY, partialTicks);
        if (drawHoverOverlay && isMouseOverElement(mouseX, mouseY) && getHoverElement(mouseX, mouseY) == this) {
            RenderSystem.colorMask(true, true, true, false);
            DrawerHelper.drawSolidRect(graphics, getPosition().x + 1, getPosition().y + 1, 16, 16, 0x80FFFFFF);
            RenderSystem.colorMask(true, true, true, true);
        }
    }

    @Override
    @Environment(EnvType.CLIENT)
    public boolean mouseClicked(double mouseX, double mouseY, int button) {
        if (slotReference != null && isMouseOverElement(mouseX, mouseY) && gui != null) {
            var stack = slotReference.m_7993_();
            if (canPutItems && stack.m_41619_() || canTakeItems && !stack.m_41619_()) {
                ModularUIGuiContainer modularUIGui = gui.getModularUIGui();
                boolean last = modularUIGui.getQuickCrafting();
                InputConstants.Key mouseKey = InputConstants.Type.MOUSE.m_84895_(button);
                HOVER_SLOT = slotReference;
                gui.getModularUIGui().superMouseClicked(mouseX, mouseY, button);
                HOVER_SLOT = null;
                if (last != modularUIGui.getQuickCrafting()) {
                    modularUIGui.dragSplittingButton = button;
                    if (button == 0) {
                        modularUIGui.dragSplittingLimit = 0;
                    } else if (button == 1) {
                        modularUIGui.dragSplittingLimit = 1;
                    } else if (Minecraft.m_91087_().f_91066_.f_92097_.m_90830_(mouseKey.m_84873_())) {
                        modularUIGui.dragSplittingLimit = 2;
                    }
                }
                return true;
            } else if (LDLib.isEmiLoaded()) {
                return EMICallWrapper.mouseClicked(getXEICurrentIngredient(), button);
            } else {
                return false;
            }
        }
        return false;
    }

    @Override
    @Environment(EnvType.CLIENT)
    public boolean keyPressed(int keyCode, int scanCode, int modifiers) {
        Window window = Minecraft.m_91087_().m_91268_();
        double mouseX = Minecraft.m_91087_().f_91067_.m_91589_() * window.m_85445_() / window.m_85443_();
        double mouseY = Minecraft.m_91087_().f_91067_.m_91594_() * window.m_85446_() / window.m_85444_();
        if (isMouseOverElement(mouseX, mouseY)) {
            if (LDLib.isEmiLoaded()) {
                if (EMICallWrapper.keyPressed(getXEICurrentIngredient(), keyCode, scanCode, modifiers)) {
                    return true;
                }
            }
        }
        return super.keyPressed(keyCode, scanCode, modifiers);
    }

    @Override
    @Environment(EnvType.CLIENT)
    public boolean mouseReleased(double mouseX, double mouseY, int button) {
        if (isMouseOverElement(mouseX, mouseY) && gui != null) {
            HOVER_SLOT = slotReference;
            gui.getModularUIGui().superMouseReleased(mouseX, mouseY, button);
            HOVER_SLOT = null;
            return getIngredientIO() == IngredientIO.RENDER_ONLY && (canPutItems || canTakeItems);
        }
        return false;
    }

    @Override
    @Environment(EnvType.CLIENT)
    public boolean mouseDragged(double mouseX, double mouseY, int button, double dragX, double dragY) {
        if (isMouseOverElement(mouseX, mouseY) && gui != null) {
            gui.getModularUIGui().superMouseDragged(mouseX, mouseY, button, dragX, dragY);
            return true;
        }
        return false;
    }

    @Override
    protected void onPositionUpdate() {
        if (gui != null) {
            Position position = getPosition();
            if (slotReference != null) {
                ((SlotAccessor) slotReference).setX(position.x + 1 - gui.getGuiLeft());
                ((SlotAccessor) slotReference).setY(position.y + 1 - gui.getGuiTop());
            }
        }
    }

    public SlotWidget(IItemTransfer itemHandler, int slotIndex, int xPosition, int yPosition) {
        this(itemHandler, slotIndex, xPosition, yPosition, true, true);
    }

    public SlotWidget(Container inventory, int slotIndex, int xPosition, int yPosition) {
        this(inventory, slotIndex, xPosition, yPosition, true, true);
    }

    public SlotWidget setBackgroundTexture(IGuiTexture backgroundTexture) {
        this.backgroundTexture = backgroundTexture;
        return this;
    }

    public boolean canPutStack(ItemStack stack) {
        return isEnabled() && canPutItems;
    }

    public boolean canTakeStack(Player player) {
        return isEnabled() && canTakeItems;
    }

    public boolean isEnabled() {
        return this.isActive() && isVisible();
    }

    public boolean canMergeSlot(ItemStack stack) {
        return isEnabled();
    }

    public void onSlotChanged() {
        if (gui == null) return;
        gui.holder.markAsDirty();
    }

    public ItemStack slotClick(int dragType, ClickType clickTypeIn, Player player) {
        return null;
    }

    @Nullable
    public final Slot getHandler() {
        return slotReference;
    }

    public SlotWidget setLocationInfo(boolean isPlayerContainer, boolean isPlayerHotBar) {
        this.isPlayerHotBar = isPlayerHotBar;
        this.isPlayerContainer = isPlayerContainer;
        return this;
    }

    @Override
    public List<Component> getTooltipTexts() {
        List<Component> tooltips = getAdditionalToolTips(new ArrayList<>());
        tooltips.addAll(tooltipTexts);
        return tooltips;
    }

    public List<Component> getAdditionalToolTips(List<Component> list) {
        if (this.onAddedTooltips != null) {
            this.onAddedTooltips.accept(this, list);
        }
        return list;
    }

    @Override
    public List<Component> getFullTooltipTexts() {
        if (slotReference != null) {
            var stack = slotReference.m_7993_();
            if (!stack.m_41619_()) {
                var tips = new ArrayList<>(DrawerHelper.getItemToolTip(stack));
                tips.addAll(getTooltipTexts());
                return tips;
            }
        }
        return Collections.emptyList();
    }


    @Nullable
    @Override
    public Object getXEIIngredientOverMouse(double mouseX, double mouseY) {
        if (self().isMouseOverElement(mouseX, mouseY)) {
            var handler = getHandler();
            if (handler == null) return null;
            ItemStack realStack = getRealStack(handler.m_7993_());

            if (handler instanceof WidgetSlotItemTransfer widgetSlotItemTransfer) {
                if (widgetSlotItemTransfer.itemHandler instanceof CycleItemStackHandler cycleItemStackHandler) {
                    return getXEIIngredientsFromCycleTransferClickable(cycleItemStackHandler, widgetSlotItemTransfer.index);
                } else if (widgetSlotItemTransfer.itemHandler instanceof TagOrCycleItemStackTransfer transfer) {
                    return getXEIIngredientsFromTagOrCycleTransferClickable(transfer, widgetSlotItemTransfer.index);
                }
            }

            if (LDLib.isJeiLoaded() && !realStack.m_41619_()) {
                return JEIPlugin.getItemIngredient(realStack, getPosition().x, getPosition().y, getSize().width, getSize().height);
            } else if (LDLib.isReiLoaded()) {
                return REICallWrapper.getReiIngredients(realStack);
            } else if (LDLib.isEmiLoaded()) {
                return EMICallWrapper.getEmiIngredients(realStack, getXEIChance());
            }
            return realStack;
        }
        return null;
    }

    @Override
    public List<Object> getXEIIngredients() {
        if (slotReference == null || slotReference.m_7993_().m_41619_()) return Collections.emptyList();
        var handler = getHandler();
        if (handler == null) return Collections.emptyList();
        ItemStack realStack = getRealStack(handler.m_7993_());

        if (handler instanceof WidgetSlotItemTransfer widgetSlotItemTransfer) {
            if (widgetSlotItemTransfer.itemHandler instanceof CycleItemStackHandler cycleItemStackHandler) {
                return getXEIIngredientsFromCycleTransferClickable(cycleItemStackHandler, widgetSlotItemTransfer.index);
            } else if (widgetSlotItemTransfer.itemHandler instanceof TagOrCycleItemStackTransfer transfer) {
                return getXEIIngredientsFromTagOrCycleTransferClickable(transfer, widgetSlotItemTransfer.index);
            }
        }

        if (LDLib.isJeiLoaded()) {
            var ingredient = JEIPlugin.getItemIngredient(realStack, getPosition().x, getPosition().y, getSize().width, getSize().height);
            return ingredient == null ? Collections.emptyList() : List.of(ingredient);
        } else if (LDLib.isReiLoaded()) {
            return REICallWrapper.getReiIngredients(realStack);
        } else if (LDLib.isEmiLoaded()) {
            return EMICallWrapper.getEmiIngredients(realStack, getXEIChance());
        }
        return List.of(realStack);
    }

    public ItemStack getRealStack(ItemStack itemStack) {
        if (itemHook != null) return itemHook.apply(itemStack);
        return itemStack;
    }

    @Override
    public Object getXEICurrentIngredient() {
        if (slotReference == null || slotReference.m_7993_().m_41619_()) return null;
        var handler = getHandler();
        if (handler == null) return null;
        ItemStack realStack = getRealStack(handler.m_7993_());

        if (LDLib.isJeiLoaded()) {
            return JEIPlugin.getItemIngredient(realStack, getPosition().x, getPosition().y, getSize().width, getSize().height);
        } else if (LDLib.isEmiLoaded()) {
            return EMICallWrapper.getEmiIngredient(realStack, getXEIChance());
        }
        return null;
    }

    private List<Object> getXEIIngredientsFromCycleTransfer(CycleItemStackHandler transfer, int index) {
        var stream = transfer.getStackList(index).stream().map(this::getRealStack);
        if (LDLib.isJeiLoaded()) {
            return stream.filter(stack -> !stack.m_41619_()).collect(Collectors.toList());
        } else if (LDLib.isReiLoaded()) {
            return REICallWrapper.getReiIngredients(stream);
        } else if (LDLib.isEmiLoaded()) {
            return EMICallWrapper.getEmiIngredients(stream, getXEIChance());
        }
        return Collections.emptyList();
    }

    private List<Object> getXEIIngredientsFromCycleTransferClickable(CycleItemStackHandler transfer, int index) {
        var stream = transfer.getStackList(index).stream().map(this::getRealStack);
        if (LDLib.isJeiLoaded()) {
            return stream
                    .filter(stack -> !stack.m_41619_())
                    .map(item -> JEIPlugin.getItemIngredient(item, getPosition().x, getPosition().y, getSize().width, getSize().height))
                    .toList();
        } else if (LDLib.isReiLoaded()) {
            return REICallWrapper.getReiIngredients(stream);
        } else if (LDLib.isEmiLoaded()) {
            return EMICallWrapper.getEmiIngredients(stream, getXEIChance());
        }
        return Collections.emptyList();
    }

    private List<Object> getXEIIngredientsFromTagOrCycleTransfer(TagOrCycleItemStackTransfer transfer, int index) {
        Either<List<Pair<TagKey<Item>, Integer>>, List<ItemStack>> either = transfer
                .getStacks()
                .get(index);
        var ref = new Object() {
            List<Object> returnValue = Collections.emptyList();
        };
        either.ifLeft(list -> {
            if (LDLib.isJeiLoaded()) {
                ref.returnValue = list.stream()
                        .flatMap(pair -> BuiltInRegistries.f_257033_
                                .m_203431_(pair.getFirst())
                                .stream()
                                .flatMap(HolderSet.ListBacked::m_203614_)
                                .map(item -> getRealStack(new ItemStack(item.m_203334_(), pair.getSecond()))))
                        .collect(Collectors.toList());
            } else if (LDLib.isReiLoaded()) {
                ref.returnValue = REICallWrapper.getReiIngredients(this::getRealStack, list);
            } else if (LDLib.isEmiLoaded()) {
                ref.returnValue = EMICallWrapper.getEmiIngredients(list, getXEIChance());
            }
        }).ifRight(items -> {
            var stream = items.stream().map(this::getRealStack);
            if (LDLib.isJeiLoaded()) {
                ref.returnValue = stream
                        .filter(stack -> !stack.m_41619_())
                        .collect(Collectors.toList());
            } else if (LDLib.isReiLoaded()) {
                ref.returnValue = REICallWrapper.getReiIngredients(stream);
            } else if (LDLib.isEmiLoaded()) {
                ref.returnValue = EMICallWrapper.getEmiIngredients(stream, getXEIChance());
            }
        });
        return ref.returnValue;
    }

    private List<Object> getXEIIngredientsFromTagOrCycleTransferClickable(TagOrCycleItemStackTransfer transfer, int index) {
        Either<List<Pair<TagKey<Item>, Integer>>, List<ItemStack>> either = transfer
                .getStacks()
                .get(index);
        var ref = new Object() {
            List<Object> returnValue = Collections.emptyList();
        };
        either.ifLeft(list -> {
            if (LDLib.isJeiLoaded()) {
                ref.returnValue = list.stream()
                        .flatMap(pair -> BuiltInRegistries.f_257033_
                                .m_203431_(pair.getFirst())
                                .stream()
                                .flatMap(HolderSet.ListBacked::m_203614_)
                                .map(item -> JEIPlugin.getItemIngredient(getRealStack(new ItemStack(item.m_203334_(), pair.getSecond())), getPosition().x, getPosition().y, getSize().width, getSize().height)))
                        .collect(Collectors.toList());
            } else if (LDLib.isReiLoaded()) {
                ref.returnValue = REICallWrapper.getReiIngredients(this::getRealStack, list);
            } else if (LDLib.isEmiLoaded()) {
                ref.returnValue = EMICallWrapper.getEmiIngredients(list, getXEIChance());
            }
        }).ifRight(items -> {
            var stream = items.stream().map(this::getRealStack);
            if (LDLib.isJeiLoaded()) {
                ref.returnValue = stream
                        .filter(stack -> !stack.m_41619_())
                        .map(item -> JEIPlugin.getItemIngredient(item, getPosition().x, getPosition().y, getSize().width, getSize().height))
                        .toList();
            } else if (LDLib.isReiLoaded()) {
                ref.returnValue = REICallWrapper.getReiIngredients(stream);
            } else if (LDLib.isEmiLoaded()) {
                ref.returnValue = EMICallWrapper.getEmiIngredients(stream, getXEIChance());
            }
        });
        return ref.returnValue;
    }


    protected class WidgetSlot extends Slot {

        public WidgetSlot(Container inventory, int index, int xPosition, int yPosition) {
            super(inventory, index, xPosition, yPosition);
        }

        @Override
        public boolean m_5857_(@Nonnull ItemStack stack) {
            return SlotWidget.this.canPutStack(stack) && super.m_5857_(stack);
        }

        @Override
        public boolean m_8010_(@Nonnull Player playerIn) {
            return SlotWidget.this.canTakeStack(playerIn) && super.m_8010_(playerIn);
        }

        @Override
        public void m_5852_(@Nonnull ItemStack stack) {
//            if(!SlotWidget.this.canPutStack(stack)) return;
            super.m_5852_(stack);
        }

        @Override
        public void m_6654_() {
            if (changeListener != null) {
                changeListener.run();
            }
            SlotWidget.this.onSlotChanged();
        }

        @Override
        public boolean m_6659_() {
            return SlotWidget.this.isEnabled() && (HOVER_SLOT == null || HOVER_SLOT == this);
        }

    }

    public class WidgetSlotItemTransfer extends Slot {
        private static final Container emptyInventory = new SimpleContainer(0);
        @Getter
        private final IItemTransfer itemHandler;
        private final int index;


        public WidgetSlotItemTransfer(IItemTransfer itemHandler, int index, int xPosition, int yPosition) {
            super(emptyInventory, index, xPosition, yPosition);
            this.itemHandler = itemHandler;
            this.f_40219_ = index;
        }

        @Override
        public boolean m_5857_(@Nonnull ItemStack stack) {
            return SlotWidget.this.canPutStack(stack) && (!stack.m_41619_() && this.itemHandler.isItemValid(this.f_40219_, stack));
        }

        @Override
        public boolean m_8010_(@Nullable Player playerIn) {
            return SlotWidget.this.canTakeStack(playerIn) && !this.itemHandler.extractItem(f_40219_, 1, true).m_41619_();
        }

        @Override
        @Nonnull
        public ItemStack m_7993_() {
            return this.itemHandler.getStackInSlot(f_40219_);
        }

        @Override
        public void m_5852_(@Nonnull ItemStack stack) {
            this.itemHandler.setStackInSlot(f_40219_, stack);
            this.m_6654_();
        }

        @Override
        public void m_40234_(@Nonnull ItemStack oldStackIn, @Nonnull ItemStack newStackIn) {

        }

        @Override
        public int m_6641_() {
            return this.itemHandler.getSlotLimit(this.f_40219_);
        }

        @Override
        public int m_5866_(@Nonnull ItemStack stack) {
            ItemStack maxAdd = stack.m_41777_();
            int maxInput = stack.m_41741_();
            maxAdd.m_41764_(maxInput);
            ItemStack currentStack = this.itemHandler.getStackInSlot(f_40219_);
            this.itemHandler.setStackInSlot(f_40219_, ItemStack.f_41583_);
            ItemStack remainder = this.itemHandler.insertItem(f_40219_, maxAdd, true);
            this.itemHandler.setStackInSlot(f_40219_, currentStack);
            return maxInput - remainder.m_41613_();
        }

        @NotNull
        @Override
        public ItemStack m_6201_(int amount) {
            var result = this.itemHandler.extractItem(f_40219_, amount, false);
            if (changeListener != null && !m_7993_().m_41619_()) {
                changeListener.run();
            }
            return result;
        }

        @Override
        public void m_6654_() {
            this.itemHandler.onContentsChanged();
            if (changeListener != null) {
                changeListener.run();
            }
            SlotWidget.this.onSlotChanged();
        }

        @Override
        public boolean m_6659_() {
            return SlotWidget.this.isEnabled() && (HOVER_SLOT == null || HOVER_SLOT == this);
        }

    }

    @Override
    public void buildConfigurator(ConfiguratorGroup father) {
        var handler = new ItemStackTransfer();
        handler.setStackInSlot(0, Blocks.f_50069_.m_5456_().m_7968_());
        father.addConfigurators(new WrapperConfigurator("ldlib.gui.editor.group.preview", new SlotWidget() {
            @Override
            public void updateScreen() {
                super.updateScreen();
                setHoverTooltips(SlotWidget.this.tooltipTexts);
                this.backgroundTexture = SlotWidget.this.backgroundTexture;
                this.hoverTexture = SlotWidget.this.hoverTexture;
                this.drawHoverOverlay = SlotWidget.this.drawHoverOverlay;
                this.drawHoverTips = SlotWidget.this.drawHoverTips;
                this.overlay = SlotWidget.this.overlay;
            }
        }.setCanPutItems(false).setCanTakeItems(false).setHandlerSlot(handler, 0)));

        IConfigurableWidget.super.buildConfigurator(father);
    }

    public static final class REICallWrapper {
        public static List<Object> getReiIngredients(Stream<ItemStack> stream) {
            return List.of(EntryIngredient.of(stream.map(EntryStacks::of).toList()));
        }

        public static List<Object> getReiIngredients(UnaryOperator<ItemStack> realStack, List<Pair<TagKey<Item>, Integer>> list) {
            return list.stream()
                    .map(pair -> EntryIngredients.ofTag(pair.getFirst(), holder -> EntryStacks.of(realStack.apply(new ItemStack(holder.m_203334_(), pair.getSecond())))))
                    .collect(Collectors.toList());
        }

        public static List<Object> getReiIngredients(ItemStack stack) {
            return List.of(EntryStacks.of(stack));
        }
    }

    public static final class EMICallWrapper {
        public static List<Object> getEmiIngredients(Stream<ItemStack> stream, float xeiChance) {
            return List.of(EmiIngredient.of(stream.map(EmiStack::of).toList()).setChance(xeiChance));
        }
        public static List<Object> getEmiIngredients(List<Pair<TagKey<Item>, Integer>> list, float xeiChance) {
            return list.stream()
                    .map(pair -> EmiIngredient.of(pair.getFirst()).setAmount(pair.getSecond()).setChance(xeiChance))
                    .collect(Collectors.toList());
        }
        public static List<Object> getEmiIngredients(ItemStack stack, float xeiChance) {
            return List.of(EmiStack.of(stack).setChance(xeiChance));
        }

        public static Object getEmiIngredient(ItemStack stack, float xeiChance) {
            return EmiStack.of(stack).setChance(xeiChance);
        }

        public static boolean mouseClicked(Object ingredient, int button) {
            if (ingredient instanceof EmiStack emiStack) {
                EmiScreenManager.stackInteraction(new EmiStackInteraction(emiStack), (bind) -> bind.matchesMouse(button));
                return true;
            }
            return false;
        }

        public static boolean keyPressed(Object ingredient, int keyCode, int scanCode, int modifiers) {
            if (ingredient instanceof EmiStack emiStack) {
                EmiScreenManager.stackInteraction(new EmiStackInteraction(emiStack), (bind) -> bind.matchesKey(keyCode, scanCode));
                return true;
            }
            return false;
        }
    }
}
