package com.lowdragmc.lowdraglib.misc;

import com.lowdragmc.lowdraglib.LDLib;
import com.lowdragmc.lowdraglib.side.fluid.FluidStack;
import com.lowdragmc.lowdraglib.side.fluid.IFluidTransfer;
import com.lowdragmc.lowdraglib.syncdata.ITagSerializable;
import lombok.Setter;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.ListTag;
import org.jetbrains.annotations.NotNull;

import java.util.Arrays;
import java.util.List;
import java.util.function.Predicate;

/**
 * @author KilaBash
 * @date 2023/2/25
 * @implNote FluidTransferList
 */
public class FluidTransferList implements IFluidTransfer, ITagSerializable<CompoundTag> {
    public final IFluidTransfer[] transfers;
    @Setter
    protected Predicate<FluidStack> filter = fluid -> true;

    public FluidTransferList(IFluidTransfer... transfers) {
        this.transfers = transfers;
    }

    public FluidTransferList(List<IFluidTransfer> transfers) {
        this.transfers = transfers.toArray(IFluidTransfer[]::new);
    }

    @Override
    public int getTanks() {
        return Arrays.stream(transfers).mapToInt(IFluidTransfer::getTanks).sum();
    }

    @NotNull
    @Override
    public FluidStack getFluidInTank(int tank) {
        int index = 0;
        for (IFluidTransfer transfer : transfers) {
            if (tank - index < transfer.getTanks()) {
                return transfer.getFluidInTank(tank - index);
            }
            index += transfer.getTanks();
        }
        return FluidStack.empty();
    }

    @Override
    public void setFluidInTank(int tank, @NotNull FluidStack fluidStack) {
        int index = 0;
        for (IFluidTransfer transfer : transfers) {
            if (tank - index < transfer.getTanks()) {
                transfer.setFluidInTank(tank - index, fluidStack);
                return;
            }
            index += transfer.getTanks();
        }
    }

    @Override
    public long getTankCapacity(int tank) {
        int index = 0;
        for (IFluidTransfer transfer : transfers) {
            if (tank - index < transfer.getTanks()) {
                return transfer.getTankCapacity(tank - index);
            }
            index += transfer.getTanks();
        }
        return 0;
    }

    @Override
    public boolean isFluidValid(int tank, @NotNull FluidStack stack) {
        if (!filter.test(stack)) {
            return false;
        }
        int index = 0;
        for (IFluidTransfer transfer : transfers) {
            if (tank - index < transfer.getTanks()) {
                return transfer.isFluidValid(tank - index, stack);
            }
            index += transfer.getTanks();
        }
        return false;
    }

    @Override
    public long fill(int tank, FluidStack resource, boolean simulate, boolean notifyChanges) {
        int index = 0;
        for (IFluidTransfer transfer : transfers) {
            if (tank - index < transfer.getTanks()) {
                return transfer.fill(tank - index, resource, simulate, notifyChanges);
            }
            index += transfer.getTanks();
        }
        return 0;
    }

    @Override
    public long fill(FluidStack resource, boolean simulate, boolean notifyChange) {
        if (resource.isEmpty() || !filter.test(resource)) return 0;
        var copied = resource.copy();
        for (var transfer : transfers) {
            var candidate = copied.copy();
            copied.shrink(transfer.fill(candidate, simulate, notifyChange));
            if (copied.isEmpty()) break;
        }
        return resource.getAmount() - copied.getAmount();
    }

    @NotNull
    @Override
    public FluidStack drain(int tank, FluidStack resource, boolean simulate, boolean notifyChanges) {
        int index = 0;
        for (IFluidTransfer transfer : transfers) {
            if (tank - index < transfer.getTanks()) {
                return transfer.drain(tank - index, resource, simulate, notifyChanges);
            }
            index += transfer.getTanks();
        }
        return FluidStack.empty();
    }

    @NotNull
    @Override
    public FluidStack drain(FluidStack resource, boolean simulate, boolean notifyChange) {
        if (resource.isEmpty() || !filter.test(resource)) return FluidStack.empty();
        var copied = resource.copy();
        for (var transfer : transfers) {
            var candidate = copied.copy();
            copied.shrink(transfer.drain(candidate, simulate, notifyChange).getAmount());
            if (copied.isEmpty()) break;
        }
        copied.setAmount(resource.getAmount() - copied.getAmount());
        return copied;
    }

    @NotNull
    @Override
    public FluidStack drain(long maxDrain, boolean simulate, boolean notifyChange) {
        if (maxDrain == 0) {
            return FluidStack.empty();
        }
        FluidStack totalDrained = null;
        for (var storage : transfers) {
            if (totalDrained == null || totalDrained.isEmpty()) {
                totalDrained = storage.drain(maxDrain, simulate, notifyChange);
                if (totalDrained.isEmpty()) {
                    totalDrained = null;
                } else {
                    maxDrain -= totalDrained.getAmount();
                }
            } else {
                FluidStack copy = totalDrained.copy();
                copy.setAmount(maxDrain);
                FluidStack drain = storage.drain(copy, simulate, notifyChange);
                totalDrained.grow(drain.getAmount());
                maxDrain -= drain.getAmount();
            }
            if (maxDrain <= 0) break;
        }
        return totalDrained == null ? FluidStack.empty() : totalDrained;
    }

    @Override
    public final void onContentsChanged() {
        for (IFluidTransfer transfer : transfers) {
            transfer.onContentsChanged();
        }
    }

    @NotNull
    @Override
    public Object createSnapshot() {
        return Arrays.stream(transfers).map(IFluidTransfer::createSnapshot).toArray(Object[]::new);
    }

    @Override
    public void restoreFromSnapshot(Object snapshot) {
        if (snapshot instanceof Object[] array && array.length == transfers.length) {
            for (int i = 0; i < array.length; i++) {
                transfers[i].restoreFromSnapshot(array[i]);
            }
        }
    }

    @Override
    public CompoundTag serializeNBT() {
        var tag = new CompoundTag();
        var list = new ListTag();
        for (IFluidTransfer transfer : transfers) {
            if (transfer instanceof ITagSerializable<?> serializable) {
                list.add(serializable.serializeNBT());
            } else {
                LDLib.LOGGER.warn("[FluidTransferList] internal tank doesn't support serialization");
            }
        }
        tag.m_128365_("tanks", list);
        tag.m_128344_("type", list.m_7264_());
        return tag;
    }

    @Override
    public void deserializeNBT(CompoundTag nbt) {
        var list = nbt.m_128437_("tanks", nbt.m_128445_("type"));
        for (int i = 0; i < list.size(); i++) {
            if (transfers[i] instanceof ITagSerializable serializable) {
                serializable.deserializeNBT(list.get(i));
            } else {
                LDLib.LOGGER.warn("[FluidTransferList] internal tank doesn't support serialization");
            }
        }
    }

    @Override
    public boolean supportsFill(int tank) {
        for (IFluidTransfer transfer : transfers) {
            if (tank >= transfer.getTanks()) {
                tank -= transfer.getTanks();
                continue;
            }

            return transfer.supportsFill(tank);
        }

        return false;
    }

    @Override
    public boolean supportsDrain(int tank) {
        for (IFluidTransfer transfer : transfers) {
            if (tank >= transfer.getTanks()) {
                tank -= transfer.getTanks();
                continue;
            }

            return transfer.supportsDrain(tank);
        }

        return false;
    }
}
