package com.lowdragmc.lowdraglib.misc;

import com.lowdragmc.lowdraglib.side.item.IItemTransfer;
import com.lowdragmc.lowdraglib.side.item.ItemTransferHelper;
import com.lowdragmc.lowdraglib.syncdata.IContentChangeAware;
import com.lowdragmc.lowdraglib.syncdata.ITagSerializable;
import lombok.Getter;
import lombok.Setter;
import net.minecraft.core.NonNullList;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.ListTag;
import net.minecraft.nbt.Tag;
import net.minecraft.world.item.ItemStack;
import org.jetbrains.annotations.NotNull;

import javax.annotation.Nonnull;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author KilaBash
 * @date 2023/2/10
 * @implNote ItemStackTransfer
 */
public class ItemStackTransfer implements IItemTransfer, ITagSerializable<CompoundTag>, IContentChangeAware {
    protected NonNullList<ItemStack> stacks;

    @Getter
    @Setter
    private Runnable onContentsChanged = () -> {};

    @Setter
    private Function<ItemStack, Boolean> filter;

    public ItemStackTransfer() {
        this(1);
    }

    public ItemStackTransfer(int size) {
        stacks = NonNullList.m_122780_(size, ItemStack.f_41583_);
    }

    public ItemStackTransfer(NonNullList<ItemStack> stacks) {
        this.stacks = stacks;
    }

    public ItemStackTransfer(ItemStack stack) {
        this(NonNullList.m_122783_(ItemStack.f_41583_, stack));
    }

    public void setSize(int size) {
        stacks = NonNullList.m_122780_(size, ItemStack.f_41583_);
    }

    @Override
    public void setStackInSlot(int slot, @Nonnull ItemStack stack) {
        validateSlotIndex(slot);
        this.stacks.set(slot, stack);
    }

    @Override
    public int getSlots() {
        return stacks.size();
    }

    @Override
    @Nonnull
    public ItemStack getStackInSlot(int slot) {
        validateSlotIndex(slot);
        return this.stacks.get(slot);
    }

    @Override
    @Nonnull
    public ItemStack insertItem(int slot, @Nonnull ItemStack stack, boolean simulate, boolean notifyChanges) {
        if (stack.m_41619_())
            return ItemStack.f_41583_;

        if (!isItemValid(slot, stack))
            return stack;

        validateSlotIndex(slot);

        ItemStack existing = this.stacks.get(slot);

        int limit = getStackLimit(slot, stack);

        if (!existing.m_41619_()) {
            if (!ItemTransferHelper.canItemStacksStack(stack, existing))
                return stack;

            limit -= existing.m_41613_();
        }

        if (limit <= 0)
            return stack;

        boolean reachedLimit = stack.m_41613_() > limit;

        if (!simulate) {
            if (existing.m_41619_()) {
                this.stacks.set(slot, reachedLimit ? ItemTransferHelper.copyStackWithSize(stack, limit) : stack);
            } else {
                existing.m_41769_(reachedLimit ? limit : stack.m_41613_());
            }
            if (notifyChanges) {
                onContentsChanged(slot);
            }
        }

        return reachedLimit ? ItemTransferHelper.copyStackWithSize(stack, stack.m_41613_() - limit) : ItemStack.f_41583_;
    }

    @Override
    @Nonnull
    public ItemStack extractItem(int slot, int amount, boolean simulate, boolean notifyChanges) {
        if (amount == 0)
            return ItemStack.f_41583_;

        validateSlotIndex(slot);

        ItemStack existing = this.stacks.get(slot);

        if (existing.m_41619_())
            return ItemStack.f_41583_;

        int toExtract = Math.min(amount, existing.m_41741_());

        if (existing.m_41613_() <= toExtract) {
            if (!simulate) {
                this.stacks.set(slot, ItemStack.f_41583_);
                if (notifyChanges) {
                    onContentsChanged(slot);
                }
                return existing;
            } else {
                return existing.m_41777_();
            }
        } else {
            if (!simulate) {
                this.stacks.set(slot, ItemTransferHelper.copyStackWithSize(existing, existing.m_41613_() - toExtract));
                if (notifyChanges) {
                    onContentsChanged(slot);
                }
            }

            return ItemTransferHelper.copyStackWithSize(existing, toExtract);
        }
    }

    @Override
    public int getSlotLimit(int slot) {
        return 64;
    }

    protected int getStackLimit(int slot, @Nonnull ItemStack stack) {
        return Math.min(getSlotLimit(slot), stack.m_41741_());
    }

    @Override
    public boolean isItemValid(int slot, @Nonnull ItemStack stack) {
        return filter == null || filter.apply(stack);
    }

    @Override
    public CompoundTag serializeNBT() {
        ListTag nbtTagList = new ListTag();
        for (int i = 0; i < stacks.size(); i++) {
            if (!stacks.get(i).m_41619_()) {
                CompoundTag itemTag = new CompoundTag();
                itemTag.m_128405_("Slot", i);
                stacks.get(i).m_41739_(itemTag);
                nbtTagList.add(itemTag);
            }
        }
        CompoundTag nbt = new CompoundTag();
        nbt.m_128365_("Items", nbtTagList);
        nbt.m_128405_("Size", stacks.size());
        return nbt;
    }

    @Override
    public void deserializeNBT(CompoundTag nbt) {
        setSize(nbt.m_128425_("Size", Tag.f_178196_) ? nbt.m_128451_("Size") : stacks.size());
        ListTag tagList = nbt.m_128437_("Items", Tag.f_178203_);
        for (int i = 0; i < tagList.size(); i++) {
            CompoundTag itemTags = tagList.m_128728_(i);
            int slot = itemTags.m_128451_("Slot");

            if (slot >= 0 && slot < stacks.size()) {
                stacks.set(slot, ItemStack.m_41712_(itemTags));
            }
        }
        onLoad();
    }

    protected void validateSlotIndex(int slot) {
        if (slot < 0 || slot >= stacks.size())
            throw new RuntimeException("Slot " + slot + " not in valid range - [0," + stacks.size() + ")");
    }

    protected void onLoad() {

    }

    public void onContentsChanged() {
        onContentsChanged.run();
    }

    public void onContentsChanged(int slot) {
        onContentsChanged();
    }

    @NotNull
    @Override
    public Object createSnapshot() {
        return stacks.stream().map(ItemStack::m_41777_).toArray(ItemStack[]::new);
    }

    @Override
    public void restoreFromSnapshot(Object snapshot) {
        if (snapshot instanceof ItemStack[] copied && copied.length == stacks.size()) {
            for (int i = 0; i < stacks.size(); i++) {
                stacks.set(i, copied[i].m_41777_());
            }
        }
    }


    public ItemStackTransfer copy() {
        var copiedStack = NonNullList.m_122780_(stacks.size(), ItemStack.f_41583_);
        for (int i = 0; i < stacks.size(); i++) {
            copiedStack.set(i, stacks.get(i).m_41777_());
        }
        var copied = new ItemStackTransfer(copiedStack);
        copied.setFilter(filter);
        return copied;
    }

}
