package com.lowdragmc.lowdraglib.networking.s2c;

import ;
import com.lowdragmc.lowdraglib.LDLib;
import com.lowdragmc.lowdraglib.networking.IHandlerContext;
import com.lowdragmc.lowdraglib.networking.IPacket;
import com.lowdragmc.lowdraglib.networking.PacketIntLocation;
import com.lowdragmc.lowdraglib.syncdata.IManagedStorage;
import com.lowdragmc.lowdraglib.syncdata.TypedPayloadRegistries;
import com.lowdragmc.lowdraglib.syncdata.accessor.IManagedAccessor;
import com.lowdragmc.lowdraglib.syncdata.blockentity.IAutoSyncBlockEntity;
import com.lowdragmc.lowdraglib.syncdata.field.ManagedKey;
import com.lowdragmc.lowdraglib.syncdata.managed.IRef;
import com.lowdragmc.lowdraglib.syncdata.payload.ITypedPayload;
import lombok.NoArgsConstructor;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.minecraft.client.Minecraft;
import net.minecraft.client.multiplayer.ClientLevel;
import net.minecraft.core.BlockPos;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.ListTag;
import net.minecraft.nbt.Tag;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.level.block.entity.BlockEntityType;
import org.jetbrains.annotations.NotNull;

import java.util.*;

/**
 * a packet that contains payload for managed fields
 */
@NoArgsConstructor
public class SPacketManagedPayload extends PacketIntLocation implements IPacket {
    private CompoundTag extra;
    private BlockEntityType<?> blockEntityType;
    private BitSet changed;

    private ITypedPayload<?>[] payloads;

    public SPacketManagedPayload(BlockEntityType<?> type, BlockPos pos, BitSet changed, ITypedPayload<?>[] payloads, CompoundTag extra) {
        super(pos);
        blockEntityType = type;
        this.changed = changed;
        this.payloads = payloads;
        this.extra = extra;
    }

    public SPacketManagedPayload(CompoundTag tag) {
        super(BlockPos.m_122022_(tag.m_128454_("p")));
        blockEntityType = BuiltInRegistries.f_257049_.m_7745_(new ResourceLocation(tag.m_128461_("t")));
        changed = BitSet.valueOf(tag.m_128463_("c"));
        ListTag list = tag.m_128437_("l", 10);
        payloads = new ITypedPayload<?>[list.size()];
        for (int i = 0; i < payloads.length; i++) {
            CompoundTag payloadTag = list.m_128728_(i);
            byte id = payloadTag.m_128445_("t");
            var payload = TypedPayloadRegistries.create(id);
            payload.deserializeNBT(payloadTag.m_128423_("d"));
            payloads[i] = payload;
        }
        extra = tag.m_128469_("e");
    }


    public CompoundTag serializeNBT() {
        CompoundTag tag = new CompoundTag();
        tag.m_128356_("p", pos.m_121878_());
        tag.m_128359_("t", Objects.requireNonNull(BuiltInRegistries.f_257049_.m_7981_(blockEntityType)).toString());
        tag.m_128382_("c", changed.toByteArray());
        ListTag list = new ListTag();
        for (ITypedPayload<?> payload : payloads) {
            CompoundTag payloadTag = new CompoundTag();
            payloadTag.m_128344_("t", payload.getType());
            var data = payload.serializeNBT();
            if (data != null) {
                payloadTag.m_128365_("d", data);
            }
            list.add(payloadTag);
        }
        tag.m_128365_("l", list);
        tag.m_128365_("e", extra);

        return tag;
    }

    public static SPacketManagedPayload of(IAutoSyncBlockEntity tile, boolean force) {
        BitSet changed = new BitSet();

        List<ITypedPayload<?>> payloads = new ArrayList<>();
        var syncedFields = tile.getRootStorage().getSyncFields();
        for (int i = 0; i < syncedFields.length; i++) {
            var field = syncedFields[i];
            if (force || field.isSyncDirty()) {
                changed.set(i);
                var key = field.getKey();
                payloads.add(key.readSyncedField(field, force));
                field.clearSyncDirty();
            }
        }
        var extra = new CompoundTag();
        tile.writeCustomSyncData(extra);

        return new SPacketManagedPayload(tile.getBlockEntityType(), tile.getCurrentPos(), changed, payloads.toArray(new ITypedPayload<?>[0]), extra);
    }

    public void processPacket(@NotNull IAutoSyncBlockEntity blockEntity) {
        if (blockEntity.getSelf().m_58903_() != blockEntityType) {
            LDLib.LOGGER.warn("Block entity type mismatch in managed payload packet!");
            return;
        }
        var storage = blockEntity.getRootStorage();
        var syncedFields = storage.getSyncFields();

        IManagedAccessor.writeSyncedFields(storage, syncedFields, changed, payloads);
        blockEntity.readCustomSyncData(extra);
    }

    @Override
    public void encode(FriendlyByteBuf buf) {
        super.encode(buf);
        buf.m_130085_(Objects.requireNonNull(BuiltInRegistries.f_257049_.m_7981_(blockEntityType)));
        buf.m_130087_(changed.toByteArray());
        for (ITypedPayload<?> payload : payloads) {
            buf.writeByte(payload.getType());
            payload.writePayload(buf);
        }
        buf.m_130079_(extra);
    }

    @Override
    public void decode(FriendlyByteBuf buffer) {
        super.decode(buffer);
        blockEntityType = BuiltInRegistries.f_257049_.m_7745_(buffer.m_130281_());
        changed = BitSet.valueOf(buffer.m_130052_());
        payloads = new ITypedPayload<?>[changed.cardinality()];
        for (int i = 0; i < payloads.length; i++) {
            byte id = buffer.readByte();
            var payload = TypedPayloadRegistries.create(id);
            payload.readPayload(buffer);
            payloads[i] = payload;
        }
        extra = buffer.m_130260_();
    }

    @Override
    @Environment(EnvType.CLIENT)
    public void execute(IHandlerContext handler) {
        if (handler.isClient()) {
            var level = Minecraft.m_91087_().f_91073_;
            if (level != null) {
                if (level.m_7702_(pos) instanceof IAutoSyncBlockEntity autoSyncBlockEntity) {
                    processPacket(autoSyncBlockEntity);
                }
            }
        }
    }
}
