package com.lowdragmc.lowdraglib.side.fluid;

import lombok.Setter;
import net.minecraft.core.Registry;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.Tag;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.network.chat.Component;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.level.material.Fluid;
import net.minecraft.world.level.material.Fluids;

import javax.annotation.Nonnull;
import java.util.Objects;

/**
 * @author KilaBash
 * @date 2023/2/10
 * @implNote FluidStack
 */
public class FluidStack {
    public static FluidStack empty() {
        return EMPTY;
    }

    public static FluidStack create(Fluid fluid, long amount, CompoundTag nbt) {
        return new FluidStack(fluid, amount, nbt);
    }

    public static FluidStack create(Fluid fluid, long amount) {
        return create(fluid, amount, null);
    }

    public static FluidStack create(FluidStack stack, long amount) {
        return create(stack.getFluid(), amount, stack.getTag());
    }

    private static final FluidStack EMPTY = new FluidStack(Fluids.f_76191_, 0, null);
    private boolean isEmpty;
    private long amount;
    private CompoundTag tag;
    @Setter
    private Fluid fluid;

    private FluidStack(Fluid fluid, long amount, CompoundTag nbt) {
        this.fluid = fluid;
        this.amount = amount;
        if (nbt != null) {
            tag = nbt.m_6426_();
        }
        updateEmpty();

    }

    public static FluidStack loadFromTag(CompoundTag nbt) {
        if (nbt == null) {
            return EMPTY;
        }
        if (!nbt.m_128425_("FluidName", Tag.f_178201_)) {
            return EMPTY;
        }

        ResourceLocation fluidName = new ResourceLocation(nbt.m_128461_("FluidName"));
        Fluid fluid = BuiltInRegistries.f_257020_.m_7745_(fluidName);
        if (fluid == Fluids.f_76191_) {
            return EMPTY;
        }
        var stack = FluidStack.create(fluid, nbt.m_128454_("Amount"));

        if (nbt.m_128425_("Tag", Tag.f_178203_)) {
            stack.setTag(nbt.m_128469_("Tag"));
        }
        return stack;
    }

    public static FluidStack readFromBuf(FriendlyByteBuf buf) {
        Fluid fluid = BuiltInRegistries.f_257020_.m_7745_(new ResourceLocation(buf.m_130277_()));
        int amount = buf.m_130242_();
        CompoundTag tag = buf.m_130260_();
        if (fluid == Fluids.f_76191_) return EMPTY;
        return FluidStack.create(fluid, amount, tag);
    }

    protected void updateEmpty() {
        isEmpty = getRawFluid() == Fluids.f_76191_ || amount <= 0;
    }

    public CompoundTag saveToTag(CompoundTag nbt) {
        nbt.m_128359_("FluidName", BuiltInRegistries.f_257020_.m_7981_(fluid).toString());
        nbt.m_128356_("Amount", amount);

        if (tag != null) {
            nbt.m_128365_("Tag", tag);
        }
        return nbt;
    }

    public void writeToBuf(FriendlyByteBuf buf) {
        buf.m_130070_(BuiltInRegistries.f_257020_.m_7981_(fluid).toString());
        buf.m_130103_(getAmount());
        buf.m_130079_(tag);
    }

    public Fluid getFluid() {
        return isEmpty ? Fluids.f_76191_ : fluid;
    }

    public final Fluid getRawFluid() {
        return fluid;
    }

    public long getAmount() {
        return isEmpty ? 0 : amount;
    }

    public void setAmount(long amount) {
        if (fluid == Fluids.f_76191_) throw new IllegalStateException("Can't modify the empty stack.");
        this.amount = amount;
        updateEmpty();
    }

    public boolean hasTag() {
        return tag != null;
    }

    public CompoundTag getTag() {
        return tag;
    }

    public void setTag(CompoundTag tag) {
        if (getRawFluid() == Fluids.f_76191_) throw new IllegalStateException("Can't modify the empty stack.");
        this.tag = tag;
    }

    public boolean isEmpty() {
        return getAmount() == 0 || this == empty();
    }

    public Component getDisplayName() {
        return FluidHelper.getDisplayName(this);
    }

    public FluidStack copy() {
        return create(getFluid(), getAmount(), getTag() == null ? null : getTag().m_6426_());
    }

    public FluidStack copy(long amount) {
        return create(getFluid(), amount, getTag());
    }

    public boolean isFluidEqual(@Nonnull FluidStack other) {
        return getFluid() == other.getFluid() && Objects.equals(getTag(), other.getTag());
    }

    public boolean isFluidStackEqual(@Nonnull FluidStack other) {
        return isFluidEqual(other) && getAmount() == other.getAmount();
    }

    public void grow(long amount) {
        setAmount(this.getAmount() + amount);
    }

    public void shrink(long amount) {
        setAmount(this.getAmount() - amount);
    }
}
