package com.lowdragmc.lowdraglib.syncdata.accessor;

import ;
import com.lowdragmc.lowdraglib.syncdata.AccessorOp;
import com.lowdragmc.lowdraglib.syncdata.IManaged;
import com.lowdragmc.lowdraglib.syncdata.IManagedStorage;
import com.lowdragmc.lowdraglib.syncdata.TypedPayloadRegistries;
import com.lowdragmc.lowdraglib.syncdata.field.ManagedKey;
import com.lowdragmc.lowdraglib.syncdata.managed.IRef;
import com.lowdragmc.lowdraglib.syncdata.payload.ITypedPayload;
import com.lowdragmc.lowdraglib.syncdata.payload.NbtTagPayload;
import com.lowdragmc.lowdraglib.utils.TagUtils;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.ListTag;
import net.minecraft.nbt.Tag;
import java.util.BitSet;

public class IManagedAccessor extends ReadonlyAccessor {

    @Override
    public boolean hasPredicate() {
        return true;
    }

    @Override
    public boolean test(Class<?> type) {
        return IManaged.class.isAssignableFrom(type);
    }

    @Override
    public ITypedPayload<?> readFromReadonlyField(AccessorOp op, Object obj) {
        if(!(obj instanceof IManaged managed)) {
            throw new IllegalArgumentException("Field %s is not ITagSerializable".formatted(obj));
        }

        CompoundTag tag;
        if (op == AccessorOp.SYNCED || op == AccessorOp.FORCE_SYNCED) {
            tag = readSyncedFields(managed, new CompoundTag(), op == AccessorOp.FORCE_SYNCED);
        } else {
            tag = readManagedFields(managed, new CompoundTag());
        }
        return new NbtTagPayload().setPayload(tag);
    }

    public static CompoundTag readSyncedFields(IManaged managed, CompoundTag tag, boolean force) {
        BitSet changed = new BitSet();
        var syncedFields = managed.getSyncStorage().getSyncFields();
        var list = new ListTag();
        for (int i = 0; i < syncedFields.length; i++) {
            var field = syncedFields[i];
            if (force || field.isSyncDirty()) {
                changed.set(i);
                var key = field.getKey();

                var payload = key.readSyncedField(field, force);
                CompoundTag payloadTag = new CompoundTag();
                payloadTag.m_128344_("t", payload.getType());
                var data = payload.serializeNBT();
                if (data != null) {
                    payloadTag.m_128365_("d", data);
                }
                list.add(payloadTag);

                field.clearSyncDirty();
            }
        }
        tag.m_128382_("c", changed.toByteArray());
        tag.m_128365_("l", list);
        return tag;
    }

    public static CompoundTag readManagedFields(IManaged managed, CompoundTag tag) {
        var persistedFields = managed.getSyncStorage().getPersistedFields();
        for (var persistedField : persistedFields) {
            var fieldKey = persistedField.getKey();
            var key = persistedField.getPersistedKey();

            var nbt = fieldKey.readPersistedField(persistedField);

            if (nbt != null) {
                TagUtils.setTagExtended(tag, key, nbt);
            }
        }
        return tag;
    }

    @Override
    public void writeToReadonlyField(AccessorOp op, Object obj, ITypedPayload<?> payload) {
        if(!(obj instanceof IManaged managed)) {
            throw new IllegalArgumentException("Field %s is not ITagSerializable".formatted(obj));
        }


        if(!(payload instanceof NbtTagPayload nbtPayload && nbtPayload.getPayload() instanceof CompoundTag tag)) {
            throw new IllegalArgumentException("Payload %s is not NbtTagPayload".formatted(payload));
        }

        if(op == AccessorOp.SYNCED || op == AccessorOp.FORCE_SYNCED) {
            var storage = managed.getSyncStorage();
            var syncedFields = storage.getSyncFields();

            var changed = BitSet.valueOf(tag.m_128463_("c"));
            var list = tag.m_128437_("l", 10);
            var payloads = new ITypedPayload<?>[list.size()];

            for (int i = 0; i < payloads.length; i++) {
                CompoundTag payloadTag = list.m_128728_(i);
                byte id = payloadTag.m_128445_("t");
                var p = TypedPayloadRegistries.create(id);
                p.deserializeNBT(payloadTag.m_128423_("d"));
                payloads[i] = p;
            }

            writeSyncedFields(storage, syncedFields, changed, payloads);
        } else if (op == AccessorOp.PERSISTED) {
            var refs = managed.getSyncStorage().getPersistedFields();
            writePersistedFields(tag, refs);
        } else {
            throw new IllegalArgumentException("Payload %s does not match op %s".formatted(payload, op));
        }
    }

    public static void writePersistedFields(CompoundTag tag, IRef[] refs) {
        for (var ref : refs) {
            var fieldKey = ref.getKey();
            var key = ref.getPersistedKey();

            var nbt = TagUtils.getTagExtended(tag, key);
            if (nbt != null) {
                fieldKey.writePersistedField(ref, nbt);
            }
        }
    }

    public static void writeSyncedFields(IManagedStorage storage, IRef[] syncedFields, BitSet changed, ITypedPayload<?>[] payloads) {
        int j = 0;
        for (int i = 0; i < changed.length(); i++) {
            if (changed.get(i)) {
                var field = syncedFields[i];
                var key = field.getKey();

                boolean hasListener = storage.hasSyncListener(key);
                Object oldValue = null;
                if (hasListener) {
                    oldValue = field.readRaw();
                }
                key.writeSyncedField(field, payloads[j]);
                if(hasListener) {
                    storage.notifyFieldUpdate(key, field.readRaw(), oldValue);
                }

                j++;
            }
        }
    }

}
