package com.lowdragmc.lowdraglib.utils;

import com.lowdragmc.lowdraglib.gui.editor.annotation.Configurable;
import com.lowdragmc.lowdraglib.gui.editor.configurator.IConfigurable;
import com.lowdragmc.lowdraglib.syncdata.IPersistedSerializable;
import lombok.NoArgsConstructor;
import lombok.Setter;
import net.minecraft.core.BlockPos;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.BlockAndTintGetter;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.EntityBlock;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.state.BlockState;

import java.util.function.Consumer;

/**
 * Author: KilaBash
 * Date: 2022/04/26
 * Description:
 */
@NoArgsConstructor
public class BlockInfo implements IPersistedSerializable, IConfigurable {
    public static final BlockInfo EMPTY = new BlockInfo(Blocks.f_50016_);

    @Setter
    @Configurable
    private BlockState blockState;
    @Setter
    private boolean hasBlockEntity;
    @Setter
    @Configurable
    private CompoundTag tag;
    @Setter
    @Configurable
    private ItemStack itemStack;
    @Setter
    private Consumer<BlockEntity> postCreate;
    // runtime
    private BlockEntity lastEntity;

    public BlockInfo(Block block) {
        this(block.m_49966_());
    }

    public BlockInfo(BlockState blockState) {
        this(blockState, false);
    }

    public BlockInfo(BlockState blockState, boolean hasBlockEntity) {
        this(blockState, hasBlockEntity, null, null);
    }
    public BlockInfo(BlockState blockState, Consumer<BlockEntity> postCreate) {
        this(blockState, true, null, postCreate);
    }

    public BlockInfo(BlockState blockState, boolean hasBlockEntity, ItemStack itemStack, Consumer<BlockEntity> postCreate) {
        this.blockState = blockState;
        this.hasBlockEntity = hasBlockEntity;
        this.itemStack = itemStack;
        this.postCreate = postCreate;
    }

    public static BlockInfo fromBlockState(BlockState state) {
        try {
            if (state.m_60734_() instanceof EntityBlock) {
                BlockEntity blockEntity = ((EntityBlock) state.m_60734_()).m_142194_(BlockPos.f_121853_, state);
                if (blockEntity != null) {
                    return new BlockInfo(state, true);
                }
            }
        } catch (Exception ignored){ }
        return new BlockInfo(state);
    }

    public static BlockInfo fromBlock(Block block) {
        return BlockInfo.fromBlockState(block.m_49966_());
    }

    public BlockState getBlockState() {
        return blockState;
    }

    public boolean hasBlockEntity() {
        return hasBlockEntity;
    }

    public BlockEntity getBlockEntity(BlockPos pos) {
        if (hasBlockEntity && blockState.m_60734_() instanceof EntityBlock entityBlock) {
            if (lastEntity != null && lastEntity.m_58899_().equals(pos)) {
                return lastEntity;
            }
            lastEntity = entityBlock.m_142194_(pos, blockState);
            if (tag != null && lastEntity != null) {
                var compoundTag2 = lastEntity.m_187482_();
                var compoundTag3 = compoundTag2.m_6426_();
                compoundTag2.m_128391_(tag);
                if (!compoundTag2.equals(compoundTag3)) {
                    lastEntity.m_142466_(compoundTag2);
                }
            }
            if (postCreate != null) {
                postCreate.accept(lastEntity);
            }
            return lastEntity;
        }
        return null;
    }

    public BlockEntity getBlockEntity(Level level, BlockPos pos) {
        BlockEntity entity = getBlockEntity(pos);
        if (entity != null) {
            entity.m_142339_(level);
        }
        return entity;
    }

    public ItemStack getItemStackForm() {
        return itemStack == null ? new ItemStack(blockState.m_60734_()) : itemStack;
    }

    public ItemStack getItemStackForm(BlockAndTintGetter level, BlockPos pos) {
        if (itemStack != null) return itemStack;
        return blockState.m_60734_().m_7397_(new FacadeBlockAndTintGetter(level, pos, blockState, null), pos, blockState);
    }

    public void apply(Level world, BlockPos pos) {
        world.m_46597_(pos, blockState);
        BlockEntity blockEntity = getBlockEntity(pos);
        if (blockEntity != null) {
            world.m_151523_(blockEntity);
        }
    }

    public void clearBlockEntityCache() {
        lastEntity = null;
    }
}
