package com.lowdragmc.lowdraglib.utils;

import com.lowdragmc.lowdraglib.LDLib;
import com.lowdragmc.lowdraglib.client.scene.ParticleManager;
import com.lowdragmc.lowdraglib.core.mixins.accessor.EntityAccessor;
import it.unimi.dsi.fastutil.ints.Int2ObjectArrayMap;
import lombok.Getter;
import lombok.Setter;
import net.minecraft.MethodsReturnNonnullByDefault;
import net.minecraft.client.Minecraft;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Holder;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EquipmentSlot;
import net.minecraft.world.entity.decoration.ArmorStand;
import net.minecraft.world.entity.decoration.ItemFrame;
import net.minecraft.world.item.EnchantedBookItem;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.ColorResolver;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.biome.Biome;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.entity.BlockEntityTicker;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.chunk.ChunkSource;
import net.minecraft.world.level.entity.LevelEntityGetter;
import net.minecraft.world.level.material.FluidState;
import org.joml.Vector3f;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.Predicate;

/**
 * Author: KilaBash
 * Date: 2021/08/25
 * Description: TrackedDummyWorld. Used to build a Fake World.
 */
@ParametersAreNonnullByDefault
@MethodsReturnNonnullByDefault
public class TrackedDummyWorld extends DummyWorld {

    @Setter
    private Predicate<BlockPos> renderFilter;
    public final WeakReference<Level> proxyWorld;
    @Getter
    public final Map<BlockPos, BlockInfo> renderedBlocks = new HashMap<>();
    public final Map<BlockPos, BlockEntity> blockEntities = new HashMap<>();
    public final Map<Integer, Entity> entities = new Int2ObjectArrayMap<>();

    @Getter
    public final Vector3f minPos = new Vector3f(Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE);
    @Getter
    public final Vector3f maxPos = new Vector3f(Integer.MIN_VALUE, Integer.MIN_VALUE, Integer.MIN_VALUE);

    public TrackedDummyWorld() {
        super(Minecraft.m_91087_().f_91073_);
        proxyWorld = new WeakReference<>(null);
    }

    public TrackedDummyWorld(Level world) {
        super(world);
        proxyWorld = new WeakReference<>(world);
    }

    public void clear() {
        renderedBlocks.clear();
        blockEntities.clear();
        entities.clear();
    }

    public void addBlocks(Map<BlockPos, BlockInfo> renderedBlocks) {
        renderedBlocks.forEach(this::addBlock);
    }

    public void addBlock(BlockPos pos, BlockInfo blockInfo) {
        if (blockInfo.getBlockState().m_60734_() == Blocks.f_50016_)
            return;
        this.renderedBlocks.put(pos, blockInfo);
        this.blockEntities.remove(pos);
        minPos.x = (Math.min(minPos.x, pos.m_123341_()));
        minPos.y = (Math.min(minPos.y, pos.m_123342_()));
        minPos.z = (Math.min(minPos.z, pos.m_123343_()));
        maxPos.x = (Math.max(maxPos.x, pos.m_123341_()));
        maxPos.y = (Math.max(maxPos.y, pos.m_123342_()));
        maxPos.z = (Math.max(maxPos.z, pos.m_123343_()));
    }

    public BlockInfo removeBlock(BlockPos pos) {
        this.blockEntities.remove(pos);
        return this.renderedBlocks.remove(pos);
    }

    // wth? mcp issue
    public void setInnerBlockEntity(@Nonnull BlockEntity pBlockEntity) {
        blockEntities.put(pBlockEntity.m_58899_(), pBlockEntity);
    }

    @Override
    public void m_151523_(@Nonnull BlockEntity pBlockEntity) {
        blockEntities.put(pBlockEntity.m_58899_(), pBlockEntity);
    }

    @Override
    public boolean m_6933_(@Nonnull BlockPos pos, @Nonnull BlockState state, int a, int b) {
        this.renderedBlocks.put(pos, BlockInfo.fromBlockState(state));
        this.blockEntities.remove(pos);
        return true;
    }

    @Override
    public BlockEntity m_7702_(@Nonnull BlockPos pos) {
        if (renderFilter != null && !renderFilter.test(pos))
            return null;
        Level proxy = proxyWorld.get();
        return proxy != null ? proxy.m_7702_(pos) : blockEntities.computeIfAbsent(pos, p -> renderedBlocks.getOrDefault(p, BlockInfo.EMPTY).getBlockEntity(this, p));
    }

    @Override
    public BlockState m_8055_(@Nonnull BlockPos pos) {
        if (renderFilter != null && !renderFilter.test(pos))
            return Blocks.f_50016_.m_49966_(); //return air if not rendering this
        Level proxy = proxyWorld.get();
        return proxy != null ? proxy.m_8055_(pos) : renderedBlocks.getOrDefault(pos, BlockInfo.EMPTY).getBlockState();
    }

    @Override
    public boolean m_7967_(Entity entity) {
        ((EntityAccessor) entity).invokeSetLevel(this);
        if (entity instanceof ItemFrame itemFrame)
            itemFrame.m_31805_(withUnsafeNBTDiscarded(itemFrame.m_31822_()));
        if (entity instanceof ArmorStand armorStand)
            for (EquipmentSlot equipmentSlot : EquipmentSlot.values())
                armorStand.m_8061_(equipmentSlot,
                        withUnsafeNBTDiscarded(armorStand.m_6844_(equipmentSlot)));
        entities.put(entity.m_19879_(), entity);
        return true;
    }

    public static ItemStack withUnsafeNBTDiscarded(ItemStack stack) {
        if (stack.m_41783_() == null)
            return stack;
        ItemStack copy = stack.m_41777_();
        stack.m_41783_()
                .m_128431_()
                .stream()
                .filter(TrackedDummyWorld::isUnsafeItemNBTKey)
                .forEach(copy::m_41749_);
        if (copy.m_41783_().m_128456_())
            copy.m_41751_(null);
        return copy;
    }

    public static boolean isUnsafeItemNBTKey(String name) {
        if (name.equals(EnchantedBookItem.f_150830_))
            return false;
        if (name.equals("Enchantments"))
            return false;
        if (name.contains("Potion"))
            return false;
        if (name.contains("Damage"))
            return false;
        if (name.equals("display"))
            return false;
        return true;
    }

    @Override
    protected LevelEntityGetter<Entity> m_142646_() {
        return super.m_142646_();
    }

    @Override
    public Entity m_6815_(int id) {
        for (Entity entity : entities.values()) {
            if (entity.m_19879_() == id && entity.m_6084_())
                return entity;
        }
        return super.m_6815_(id);
    }

    public Vector3f getSize() {
        return new Vector3f(maxPos.x - minPos.x + 1, maxPos.y - minPos.y + 1, maxPos.z - minPos.z + 1);
    }

    @Override
    public ChunkSource m_7726_() {
        Level proxy = proxyWorld.get();
        return proxy == null ? super.m_7726_() : proxy.m_7726_();
    }

    @Override
    public FluidState m_6425_(BlockPos pPos) {
        Level proxy = proxyWorld.get();
        return proxy == null ? super.m_6425_(pPos) : proxy.m_6425_(pPos);
    }

    @Override
    public int m_6171_(@Nonnull BlockPos blockPos, @Nonnull ColorResolver colorResolver) {
        Level proxy = proxyWorld.get();
        return proxy == null ? super.m_6171_(blockPos, colorResolver) : proxy.m_6171_(blockPos, colorResolver);
    }

    @Nonnull
    @Override
    public Holder<Biome> m_204166_(@Nonnull BlockPos pos) {
        Level proxy = proxyWorld.get();
        return proxy == null ? super.m_204166_(pos) : proxy.m_204166_(pos);
    }

    @Override
    public void setParticleManager(ParticleManager particleManager) {
        super.setParticleManager(particleManager);
        if (proxyWorld.get() instanceof DummyWorld dummyWorld) {
            dummyWorld.setParticleManager(particleManager);
        }
    }

    @Nullable
    @Override
    public ParticleManager getParticleManager() {
        ParticleManager particleManager = super.getParticleManager();
        if (particleManager == null && proxyWorld.get() instanceof DummyWorld dummyWorld) {
            return dummyWorld.getParticleManager();
        }
        return particleManager;
    }

    public void tickWorld() {
        var iter = entities.values().iterator();
        while (iter.hasNext()) {
            var entity = iter.next();
            entity.f_19797_++;
            entity.m_146867_();
            entity.m_8119_();

            if (entity.m_20186_() <= -.5f)
                entity.m_146870_();

            if (!entity.m_6084_())
                iter.remove();
        }

        for (var entry : renderedBlocks.entrySet()) {
            var blockState = entry.getValue().getBlockState();
            var blockEntity = m_7702_(entry.getKey());
            if (blockEntity != null && blockEntity.m_58903_().m_155262_(blockState)) {
                try {
                    BlockEntityTicker ticker = blockState.m_155944_(this, blockEntity.m_58903_());
                    if (ticker != null) {
                        ticker.m_155252_(this, entry.getKey(), blockState, blockEntity);
                    }
                } catch (Exception e) {
                    LDLib.LOGGER.error("error while update DummyWorld tick, pos {} type {}", entry.getKey(), blockEntity.m_58903_(), e);
                }
            }
        }
    }

    public List<Entity> getAllEntities() {
        var entities = new ArrayList<>(this.entities.values());
        if (proxyWorld.get() instanceof TrackedDummyWorld trackedDummyWorld)
            entities.addAll(trackedDummyWorld.getAllEntities());
        return entities;
    }
}
