package dev.anvilcraft.lite.integration.jei.category.anvil;

import dev.anvilcraft.lib.recipe.component.BlockStatePredicate;
import dev.anvilcraft.lib.recipe.component.ChanceBlockState;
import dev.anvilcraft.lib.recipe.component.ChanceItemStack;
import dev.anvilcraft.lite.init.reicpe.ModRecipeTypes;
import dev.anvilcraft.lite.integration.jei.AnvilCraftJeiPlugin;
import dev.anvilcraft.lite.integration.jei.util.JeiRecipeUtil;
import dev.anvilcraft.lite.integration.jei.util.JeiRenderHelper;
import dev.anvilcraft.lite.recipe.anvil.predicate.block.HasCauldron;
import dev.anvilcraft.lite.recipe.anvil.wrap.SqueezingRecipe;
import dev.anvilcraft.lite.util.CauldronUtil;
import dev.anvilcraft.lite.util.render.RenderHelper;
import mezz.jei.api.gui.ITickTimer;
import mezz.jei.api.gui.builder.IRecipeLayoutBuilder;
import mezz.jei.api.gui.builder.ITooltipBuilder;
import mezz.jei.api.gui.drawable.IDrawable;
import mezz.jei.api.gui.ingredient.IRecipeSlotsView;
import mezz.jei.api.helpers.IGuiHelper;
import mezz.jei.api.recipe.IFocusGroup;
import mezz.jei.api.recipe.RecipeIngredientRole;
import mezz.jei.api.recipe.category.IRecipeCategory;
import mezz.jei.api.recipe.types.IRecipeType;
import mezz.jei.api.registration.IRecipeCatalystRegistration;
import mezz.jei.api.registration.IRecipeRegistration;
import net.minecraft.MethodsReturnNonnullByDefault;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.renderer.Rect2i;
import net.minecraft.core.Holder;
import net.minecraft.network.chat.Component;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.Items;
import net.minecraft.world.item.crafting.Ingredient;
import net.minecraft.world.item.crafting.RecipeHolder;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.state.BlockState;
import org.jetbrains.annotations.Nullable;

import javax.annotation.ParametersAreNonnullByDefault;
import java.util.ArrayList;
import java.util.List;

@MethodsReturnNonnullByDefault
@ParametersAreNonnullByDefault
public class SqueezingCategory implements IRecipeCategory<RecipeHolder<SqueezingRecipe>> {
    public static final int WIDTH = 162;
    public static final int HEIGHT = 64;

    private final IDrawable arrowDefault;
    private final IDrawable icon;
    private final ITickTimer timer;
    private final Component title;

    public SqueezingCategory(IGuiHelper helper) {
        arrowDefault = JeiRenderHelper.getArrowDefault(helper);
        icon = helper.createDrawableItemStack(new ItemStack(Items.ANVIL));
        title = Component.translatable("gui.anvilcraft.category.squeezing");
        timer = helper.createTickTimer(30, 60, true);
    }

    @Override
    public IRecipeType<RecipeHolder<SqueezingRecipe>> getRecipeType() {
        return AnvilCraftJeiPlugin.SQUEEZING;
    }

    @Override
    public Component getTitle() {
        return title;
    }

    @Override
    public int getWidth() {
        return WIDTH;
    }

    @Override
    public int getHeight() {
        return HEIGHT;
    }

    @Override
    public @Nullable IDrawable getIcon() {
        return icon;
    }

    @Override
    public void setRecipe(
        IRecipeLayoutBuilder builder, RecipeHolder<SqueezingRecipe> recipeHolder, IFocusGroup focuses) {
        SqueezingRecipe recipe = recipeHolder.value();
        for (BlockStatePredicate input : recipe.getInputBlocks()) {
            builder.addInvisibleIngredients(RecipeIngredientRole.INPUT)
                .add(Ingredient.of(input.getBlocks().stream().map(Holder::value)));
        }
        for (ChanceItemStack output : recipe.getResultItems()) {
            builder.addInvisibleIngredients(RecipeIngredientRole.OUTPUT)
                .add(output.stack().copyWithCount(output.getMaxCount()));
        }
    }

    @Override
    public void getTooltip(
        ITooltipBuilder tooltip,
        RecipeHolder<SqueezingRecipe> recipeHolder,
        IRecipeSlotsView recipeSlotsView,
        double mouseX,
        double mouseY
    ) {
        SqueezingRecipe recipe = recipeHolder.value();
        if (mouseX >= 40 && mouseX <= 58) {
            if (mouseY >= 24 && mouseY <= 42) {
                tooltip.add(recipe.getInputBlocks().getFirst().constructStatesForRender().getFirst().getBlock().getName());
            }
            if (mouseY >= 42 && mouseY <= 52) {
                tooltip.add(Blocks.CAULDRON.getName());
            }
        }
        if (mouseX >= 100 && mouseX <= 120) {
            if (mouseY >= 24 && mouseY <= 42) {
                List<ChanceBlockState> result = recipe.getResultBlocks();
                if (result.isEmpty()) return;
                tooltip.add(result.get((int) ((System.currentTimeMillis() / 1000) % result.size())).state().getBlock().getName());
            }
            if (mouseY >= 42 && mouseY <= 52) {
                tooltip.add(recipe.getHasCauldron().getTransformCauldron().getName());
            }
        }
    }

    @Override
    public void draw(
        RecipeHolder<SqueezingRecipe> recipeHolder,
        IRecipeSlotsView recipeSlotsView,
        GuiGraphics guiGraphics,
        double mouseX,
        double mouseY
    ) {
        Rect2i area = AnvilCraftJeiPlugin.AREA_WHEN_DRAW.get();
        int left = area.getX() - 9;
        int top = area.getY();
        SqueezingRecipe recipe = recipeHolder.value();
        float anvilYOffset = JeiRenderHelper.getAnvilAnimationOffset(timer);

        List<BlockState> input = new ArrayList<>();
        for (BlockStatePredicate predicate : recipe.getInputBlocks()) {
            input.addAll(predicate.constructStatesForRender());
        }
        if (input.isEmpty()) return;
        BlockState renderedState = input.get((int) ((System.currentTimeMillis() / 1000) % input.size()));
        if (renderedState == null) return;
        RenderHelper.renderMultipleBlocks(
            guiGraphics,
            List.of(renderedState, Blocks.CAULDRON.defaultBlockState()),
            left + 50,
            top + 30,
            12
        );
        RenderHelper.renderSingleBlock(
            guiGraphics,
            Blocks.ANVIL.defaultBlockState(),
            left + 50,
            top + 12 + anvilYOffset,
            12
        );

        RenderHelper.renderSingleBlock(
            guiGraphics,
            SqueezingCategory.getCauldron(recipe),
            left + 110,
            top + 40,
            12
        );
        arrowDefault.draw(guiGraphics, 73, 35);
        List<ChanceBlockState> result = recipe.getResultBlocks();
        if (!result.isEmpty()) {
            renderedState = result.get((int) ((System.currentTimeMillis() / 1000) % result.size())).state();
            RenderHelper.renderSingleBlock(guiGraphics, renderedState, left + 110, top + 30, 12);
        }
        RenderHelper.renderSingleBlock(
            guiGraphics,
            Blocks.ANVIL.defaultBlockState(),
            left + 110,
            top + 20,
            12
        );
    }

    static BlockState getCauldron(SqueezingRecipe recipe) {
        if (recipe.isProduceFluid()) {
            return Blocks.CAULDRON.defaultBlockState();
        } else {
            return CauldronUtil.fullState(HasCauldron.getDefaultCauldron(recipe.getHasCauldron().transform()));
        }
    }

    public static void registerRecipes(IRecipeRegistration registration) {
        registration.addRecipes(
            AnvilCraftJeiPlugin.SQUEEZING,
            JeiRecipeUtil.getRecipeHoldersFromType(ModRecipeTypes.SQUEEZING_TYPE.get())
        );
    }

    public static void registerCraftingStations(IRecipeCatalystRegistration registration) {
        registration.addCraftingStation(AnvilCraftJeiPlugin.SQUEEZING, new ItemStack(Items.ANVIL));
        registration.addCraftingStation(AnvilCraftJeiPlugin.SQUEEZING, new ItemStack(Items.CAULDRON));
    }
}
