package dev.anvilcraft.lite.recipe.anvil.wrap;

import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.anvilcraft.lib.recipe.component.BlockStatePredicate;
import dev.anvilcraft.lib.recipe.component.ChanceBlockState;
import dev.anvilcraft.lib.recipe.component.ChanceItemStack;
import dev.anvilcraft.lib.recipe.component.ItemIngredientPredicate;
import dev.anvilcraft.lite.init.reicpe.ModRecipeTypes;
import dev.anvilcraft.lite.recipe.anvil.util.WrapUtils;
import net.minecraft.core.HolderGetter;
import net.minecraft.core.Vec3i;
import net.minecraft.network.RegistryFriendlyByteBuf;
import net.minecraft.network.codec.ByteBufCodecs;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.crafting.RecipeSerializer;
import net.minecraft.world.item.crafting.RecipeType;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.phys.Vec3;
import java.util.List;
import java.util.function.Supplier;

/**
 * 物品注入配方类
 * <p>
 * 该配方用于在铁砧下落时将物品注入到方块中，需要在铁砧下方放置特定方块作为注入目标
 * </p>
 */
public class ItemInjectRecipe extends AbstractProcessRecipe<ItemInjectRecipe> {
    /**
     * 构造一个物品注入配方
     *
     * @param itemIngredients 物品原料列表
     * @param results         结果物品列表
     * @param blockIngredient 方块原料谓词
     * @param blockResult     方块结果
     */
    public ItemInjectRecipe(List<ItemIngredientPredicate> itemIngredients, List<ChanceItemStack> results, BlockStatePredicate blockIngredient, ChanceBlockState blockResult) {
        super(new Property().setItemInputOffset(Vec3.ZERO).setInputItems(itemIngredients).setItemOutputOffset(new Vec3(0.0, -1.0, 0.0)).setResultItems(results).setBlockInputOffset(new Vec3i(0, -1, 0)).setInputBlocks(blockIngredient).setBlockOutputOffset(new Vec3i(0, -1, 0)).setResultBlocks(blockResult));
    }

    @Override
    public RecipeSerializer<ItemInjectRecipe> getSerializer() {
        return ModRecipeTypes.ITEM_INJECT_SERIALIZER.get();
    }

    @Override
    public RecipeType<ItemInjectRecipe> getType() {
        return ModRecipeTypes.ITEM_INJECT_TYPE.get();
    }

    /**
     * 创建一个构建器实例
     *
     * @return 构建器实例
     */
    public static Builder builder(HolderGetter<Item> getter, HolderGetter<Block> blockGetter) {
        return new Builder(getter, blockGetter);
    }


    /**
     * 物品注入配方序列化器
     */
    public static class Serializer implements RecipeSerializer<ItemInjectRecipe> {
        /**
         * 编解码器
         */
        private static final MapCodec<ItemInjectRecipe> CODEC = RecordCodecBuilder.mapCodec(instance -> instance.group(ItemIngredientPredicate.CODEC.codec().listOf().optionalFieldOf("ingredients", List.of()).forGetter(ItemInjectRecipe::getInputItems), ChanceItemStack.CODEC.listOf().optionalFieldOf("results", List.of()).forGetter(ItemInjectRecipe::getResultItems), BlockStatePredicate.CODEC.fieldOf("block_ingredient").forGetter(ItemInjectRecipe::getFirstInputBlock), ChanceBlockState.CODEC.codec().fieldOf("block_result").forGetter(ItemInjectRecipe::getFirstResultBlock)).apply(instance, ItemInjectRecipe::new));
        /**
         * 流编解码器
         */
        private static final StreamCodec<RegistryFriendlyByteBuf, ItemInjectRecipe> STREAM_CODEC = StreamCodec.composite(ItemIngredientPredicate.STREAM_CODEC.apply(ByteBufCodecs.list()), ItemInjectRecipe::getInputItems, ChanceItemStack.STREAM_CODEC.apply(ByteBufCodecs.list()), ItemInjectRecipe::getResultItems, BlockStatePredicate.STREAM_CODEC, ItemInjectRecipe::getFirstInputBlock, ChanceBlockState.STREAM_CODEC, ItemInjectRecipe::getFirstResultBlock, ItemInjectRecipe::new);

        @Override
        public MapCodec<ItemInjectRecipe> codec() {
            return Serializer.CODEC;
        }

        @Override
        public StreamCodec<RegistryFriendlyByteBuf, ItemInjectRecipe> streamCodec() {
            return Serializer.STREAM_CODEC;
        }
    }


    /**
     * 物品注入配方构建器
     */
    public static class Builder extends SimpleAbstractBuilder<ItemInjectRecipe, Builder> {
        /**
         * 方块原料谓词构建器
         */
        BlockStatePredicate.Builder blockIngredient;
        /**
         * 方块结果
         */
        ChanceBlockState blockResult = null;

        protected Builder(HolderGetter<Item> getter, HolderGetter<Block> blockGetter) {
            super(getter);
            this.blockIngredient = BlockStatePredicate.builder(blockGetter);
        }

        /**
         * 设置输入方块
         *
         * @param block 输入方块
         * @return 构建器实例
         */
        public Builder inputBlock(Block block) {
            this.blockIngredient.of(block);
            return this;
        }

        /**
         * 设置输入方块（供应器形式）
         *
         * @param block 输入方块供应器
         * @return 构建器实例
         */
        public Builder inputBlock(Supplier<? extends Block> block) {
            return this.inputBlock(block.get());
        }

        /**
         * 设置结果方块
         *
         * @param block 结果方块
         * @return 构建器实例
         */
        public Builder resultBlock(Block block) {
            this.blockResult = new ChanceBlockState(block.defaultBlockState(), 1.0F);
            return this;
        }

        /**
         * 设置结果方块（供应器形式）
         *
         * @param block 结果方块供应器
         * @return 构建器实例
         */
        public Builder resultBlock(Supplier<? extends Block> block) {
            return this.resultBlock(block.get());
        }

        @Override
        public String getType() {
            return "item_inject";
        }

        @Override
        public void validate(ResourceLocation pId) {
            if (this.itemIngredients.isEmpty()) {
                throw new IllegalArgumentException("Recipe ingredients must not be empty, RecipeId: " + pId);
            }
        }

        @Override
        protected ItemInjectRecipe of(List<ItemIngredientPredicate> itemIngredients, List<ChanceItemStack> results) {
            return new ItemInjectRecipe(itemIngredients, results, this.blockIngredient.build(), this.blockResult);
        }

        @Override
        public Item getResult() {
            return WrapUtils.getItem(blockResult);
        }

        @Override
        protected Builder getThis() {
            return this;
        }
    }
}
