package dev.anvilcraft.lite.recipe.anvil.wrap;

import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.anvilcraft.lib.recipe.component.BlockStatePredicate;
import dev.anvilcraft.lib.recipe.component.ChanceBlockState;
import dev.anvilcraft.lite.init.reicpe.ModRecipeTypes;
import dev.anvilcraft.lite.recipe.anvil.builder.AbstractRecipeBuilder;
import dev.anvilcraft.lite.recipe.anvil.predicate.block.HasCauldron;
import dev.anvilcraft.lite.recipe.anvil.util.WrapUtils;
import dev.anvilcraft.lite.recipe.component.HasCauldronSimple;
import net.minecraft.core.HolderGetter;
import net.minecraft.core.Vec3i;
import net.minecraft.network.RegistryFriendlyByteBuf;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.tags.TagKey;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.crafting.RecipeSerializer;
import net.minecraft.world.item.crafting.RecipeType;
import net.minecraft.world.level.block.Block;

/**
 * 压榨配方类
 * <p>
 * 该配方用于在铁砧下落时压榨方块，需要在铁砧下方放置炼药锅作为收集容器
 * </p>
 */
public class SqueezingRecipe extends AbstractProcessRecipe<SqueezingRecipe> {
    /**
     * 构造一个压榨配方
     *
     * @param ingredient  原料方块列表
     * @param result      结果方块列表
     * @param hasCauldron 炼药锅条件
     */
    public SqueezingRecipe(BlockStatePredicate ingredient, ChanceBlockState result, HasCauldronSimple hasCauldron) {
        super(new Property().setBlockInputOffset(new Vec3i(0, -1, 0)).setConsumeInputBlocks(true).setInputBlocks(ingredient).setCauldronOffset(new Vec3i(0, -2, 0)).setHasCauldron(hasCauldron).setBlockOutputOffset(new Vec3i(0, -1, 0)).setResultBlocks(result));
    }

    @Override
    public RecipeSerializer<SqueezingRecipe> getSerializer() {
        return ModRecipeTypes.SQUEEZING_SERIALIZER.get();
    }

    @Override
    public RecipeType<SqueezingRecipe> getType() {
        return ModRecipeTypes.SQUEEZING_TYPE.get();
    }

    /**
     * 创建一个构建器实例
     *
     * @return 构建器实例
     */
    public static Builder builder(HolderGetter<Block> getter) {
        return new Builder(getter);
    }

    /**
     * 是否产生流体
     *
     * @return 如果产生流体返回true，否则返回false
     */
    public boolean isProduceFluid() {
        HasCauldronSimple hasCauldron = this.getHasCauldron();
        return HasCauldron.isNotEmpty(hasCauldron.transform()) && this.getHasCauldron().consume() < 0;
    }


    /**
     * 压榨配方序列化器
     */
    public static class Serializer implements RecipeSerializer<SqueezingRecipe> {
        /**
         * 编解码器
         */
        public static final MapCodec<SqueezingRecipe> CODEC = RecordCodecBuilder.mapCodec(instance -> instance.group(BlockStatePredicate.CODEC.fieldOf("ingredient").forGetter(SqueezingRecipe::getFirstInputBlock), ChanceBlockState.CODEC.codec().fieldOf("result").forGetter(SqueezingRecipe::getFirstResultBlock), HasCauldronSimple.CODEC.forGetter(SqueezingRecipe::getHasCauldron)).apply(instance, SqueezingRecipe::new));
        /**
         * 流编解码器
         */
        public static final StreamCodec<RegistryFriendlyByteBuf, SqueezingRecipe> STREAM_CODEC = StreamCodec.composite(BlockStatePredicate.STREAM_CODEC, SqueezingRecipe::getFirstInputBlock, ChanceBlockState.STREAM_CODEC, SqueezingRecipe::getFirstResultBlock, HasCauldronSimple.STREAM_CODEC, SqueezingRecipe::getHasCauldron, SqueezingRecipe::new);

        @Override
        public MapCodec<SqueezingRecipe> codec() {
            return Serializer.CODEC;
        }

        @Override
        public StreamCodec<RegistryFriendlyByteBuf, SqueezingRecipe> streamCodec() {
            return Serializer.STREAM_CODEC;
        }
    }


    /**
     * 压榨配方构建器
     */
    public static class Builder extends AbstractRecipeBuilder<SqueezingRecipe> {
        private final HolderGetter<Block> getter;
        /**
         * 原料列表
         */
        private BlockStatePredicate ingredient = null;
        /**
         * 结果列表
         */
        private ChanceBlockState result = null;
        /**
         * 炼药锅条件构建器
         */
        private final HasCauldronSimple.Builder hasCauldron = HasCauldronSimple.empty();

        public Builder(HolderGetter<Block> getter) {
            this.getter = getter;
        }

        /**
         * 添加原料方块
         *
         * @param ingredient 原料方块谓词
         * @return 构建器实例
         */
        public Builder requires(BlockStatePredicate ingredient) {
            this.ingredient = ingredient;
            return this;
        }

        /**
         * 添加原料方块
         *
         * @param ingredient 原料方块
         * @return 构建器实例
         */
        public Builder requires(Block ingredient) {
            return this.requires(BlockStatePredicate.builder(this.getter).of(ingredient).build());
        }

        /**
         * 添加原料方块（标签形式）
         *
         * @param ingredient 原料方块标签
         * @return 构建器实例
         */
        public Builder requires(TagKey<Block> ingredient) {
            return this.requires(BlockStatePredicate.builder(this.getter).of(ingredient).build());
        }

        /**
         * 添加结果方块
         *
         * @param result 结果方块
         * @return 构建器实例
         */
        public Builder result(ChanceBlockState result) {
            this.result = result;
            return this;
        }

        /**
         * 添加结果方块（指定概率）
         *
         * @param result 结果方块
         * @param chance 概率
         * @return 构建器实例
         */
        public Builder result(Block result, float chance) {
            return this.result(new ChanceBlockState(result.defaultBlockState(), chance));
        }

        /**
         * 添加结果方块（默认概率为1.0f）
         *
         * @param result 结果方块
         * @return 构建器实例
         */
        public Builder result(Block result) {
            return this.result(result, 1.0F);
        }

        /**
         * 设置炼药锅流体
         *
         * @param fluid 流体ID
         * @return 构建器实例
         */
        public Builder cauldron(ResourceLocation fluid) {
            this.hasCauldron.fluid(fluid);
            return this;
        }

        /**
         * 设置炼药锅方块
         *
         * @param cauldron 炼药锅方块
         * @return 构建器实例
         */
        public Builder cauldron(Block cauldron) {
            this.cauldron(WrapUtils.cauldron2Fluid(cauldron));
            return this;
        }

        /**
         * 设置转换后的流体
         *
         * @param transform 转换后的流体ID
         * @return 构建器实例
         */
        public Builder transform(ResourceLocation transform) {
            this.hasCauldron.transform(transform);
            return this;
        }

        /**
         * 设置转换后的炼药锅方块
         *
         * @param transform 转换后的炼药锅方块
         * @return 构建器实例
         */
        public Builder transform(Block transform) {
            this.hasCauldron.transform(WrapUtils.cauldron2Fluid(transform));
            return this;
        }

        /**
         * 设置是否产生流体
         *
         * @param produce 是否产生流体
         * @return 构建器实例
         */
        public Builder produce(int produce) {
            if (produce <= 0) return this;
            this.hasCauldron.consume(-produce);
            return this;
        }

        /**
         * 设置是否消耗流体
         *
         * @param consume 是否消耗流体
         * @return 构建器实例
         */
        public Builder consume(int consume) {
            if (consume <= 0) return this;
            this.hasCauldron.consume(consume);
            return this;
        }

        @Override
        public SqueezingRecipe buildRecipe() {
            return new SqueezingRecipe(this.ingredient, this.result, hasCauldron.build());
        }

        @Override
        public void validate(ResourceLocation pId) {
        }

        @Override
        public String getType() {
            return "bulging";
        }

        @Override
        public Item getResult() {
            return WrapUtils.getItem(this.result);
        }
    }
}
