package com.jaquadro.minecraft.storagedrawers.block.tile;

import java.util.EnumSet;
import java.util.UUID;

import javax.annotation.Nullable;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.ISidedInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.Packet;
import net.minecraft.network.play.server.S35PacketUpdateTileEntity;
import net.minecraft.util.MathHelper;
import net.minecraftforge.common.util.Constants;

import org.apache.logging.log4j.Level;

import com.jaquadro.minecraft.storagedrawers.StorageDrawers;
import com.jaquadro.minecraft.storagedrawers.api.inventory.IDrawerInventory;
import com.jaquadro.minecraft.storagedrawers.api.security.ISecurityProvider;
import com.jaquadro.minecraft.storagedrawers.api.storage.IDrawer;
import com.jaquadro.minecraft.storagedrawers.api.storage.IDrawerGroupInteractive;
import com.jaquadro.minecraft.storagedrawers.api.storage.attribute.IDowngradable;
import com.jaquadro.minecraft.storagedrawers.api.storage.attribute.ILockable;
import com.jaquadro.minecraft.storagedrawers.api.storage.attribute.IProtectable;
import com.jaquadro.minecraft.storagedrawers.api.storage.attribute.ISealable;
import com.jaquadro.minecraft.storagedrawers.api.storage.attribute.LockAttribute;
import com.jaquadro.minecraft.storagedrawers.block.BlockDrawersCustom;
import com.jaquadro.minecraft.storagedrawers.config.ConfigManager;
import com.jaquadro.minecraft.storagedrawers.core.ModItems;
import com.jaquadro.minecraft.storagedrawers.inventory.ISideManager;
import com.jaquadro.minecraft.storagedrawers.inventory.StorageInventory;
import com.jaquadro.minecraft.storagedrawers.network.CountUpdateMessage;
import com.jaquadro.minecraft.storagedrawers.storage.IUpgradeProvider;

import cpw.mods.fml.common.FMLLog;
import cpw.mods.fml.common.network.NetworkRegistry;
import cpw.mods.fml.common.network.simpleimpl.IMessage;

public abstract class TileEntityDrawers extends BaseTileEntity implements IDrawerGroupInteractive, ISidedInventory,
        IUpgradeProvider, ILockable, ISealable, IProtectable, IDowngradable {

    private IDrawer[] drawers;
    private IDrawerInventory inventory;

    private int[] autoSides = new int[] { 0, 1, 2, 3, 4, 5 };

    private int direction;
    private int drawerCapacity = 1;
    private boolean shrouded = false;
    private boolean quantified = false;
    private boolean taped = false;
    private boolean hideUpgrade = false;
    private boolean downgraded = false;

    private UUID owner;
    private String securityKey;

    private EnumSet<LockAttribute> lockAttributes = null;

    private ItemStack[] upgrades = new ItemStack[5];

    private long lastClickTime;
    private UUID lastClickUUID;

    private String customName;

    private ItemStack materialSide;
    private ItemStack materialFront;
    private ItemStack materialTrim;

    protected TileEntityDrawers(int drawerCount) {
        initWithDrawerCount(drawerCount);
    }

    protected abstract IDrawer createDrawer(int slot);

    protected ISideManager getSideManager() {
        return new DefaultSideManager();
    }

    protected void initWithDrawerCount(int drawerCount) {
        drawers = new IDrawer[drawerCount];
        for (int i = 0; i < drawerCount; i++) drawers[i] = createDrawer(i);

        inventory = new StorageInventory(this, getSideManager(), this);
    }

    public int getDirection() {
        return direction;
    }

    public void setDirection(int direction) {
        this.direction = direction % 6;
    }

    public int getMaxStorageLevel() {
        // Could use iterables and lambdas instead
        int maxLevel = 1;
        for (ItemStack upgrade : upgrades) {
            if (upgrade != null && upgrade.getItem() == ModItems.upgrade)
                maxLevel = Math.max(maxLevel, upgrade.getItemDamage());
        }

        return maxLevel;
    }

    public int getEffectiveStorageLevel() {
        int level = 0;
        for (ItemStack upgrade : upgrades) {
            if (upgrade != null && upgrade.getItem() == ModItems.upgrade) level += upgrade.getItemDamage();
        }

        return Math.max(level, 1);
    }

    public int getEffectiveStorageMultiplier() {
        ConfigManager config = StorageDrawers.config;
        int multiplier = 0;
        for (ItemStack stack : upgrades) {
            if (stack != null && stack.getItem() == ModItems.upgrade)
                multiplier += config.getStorageUpgradeMultiplier(stack.getItemDamage());
        }

        if (multiplier == 0) multiplier = config.getStorageUpgradeMultiplier(1);

        return multiplier;
    }

    public int getEffectiveStatusLevel() {
        int maxLevel = 0;
        for (ItemStack upgrade : upgrades) {
            if (upgrade != null && upgrade.getItem() == ModItems.upgradeStatus) maxLevel = upgrade.getItemDamage();
        }

        return maxLevel;
    }

    public int getUpgradeSlotCount() {
        return 5;
    }

    public ItemStack getUpgrade(int slot) {
        slot = MathHelper.clamp_int(slot, 0, 4);
        return upgrades[slot];
    }

    public boolean addUpgrade(ItemStack upgrade) {
        int slot = getNextUpgradeSlot();
        if (slot == -1) return false;

        setUpgrade(slot, upgrade);
        checkDowngraded();
        return true;
    }

    public void setUpgrade(int slot, ItemStack upgrade) {
        slot = MathHelper.clamp_int(slot, 0, 4);

        if (upgrade != null) {
            upgrade = upgrade.copy();
            upgrade.stackSize = 1;
        }

        upgrades[slot] = upgrade;

        if (worldObj != null) {
            if (!worldObj.isRemote) {
                markDirty();
                worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
            }
            worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord, zCoord, getBlockType());
            worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord - 1, zCoord, getBlockType());
        }
    }

    public int getNextUpgradeSlot() {
        for (int i = 0; i < upgrades.length; i++) {
            if (upgrades[i] == null) return i;
        }

        return -1;
    }

    public int getDrawerCapacity() {
        return isDowngraded() ? 1 : drawerCapacity;
    }

    public void setDrawerCapacity(int stackCount) {
        drawerCapacity = stackCount;
    }

    @Override
    public boolean isLocked(LockAttribute attr) {
        if (!StorageDrawers.config.cache.enableLockUpgrades || lockAttributes == null) return false;

        return lockAttributes.contains(attr);
    }

    @Override
    public boolean canLock(LockAttribute attr) {
        if (!StorageDrawers.config.cache.enableLockUpgrades) return false;

        return true;
    }

    @Override
    public void setLocked(LockAttribute attr, boolean isLocked) {
        if (!StorageDrawers.config.cache.enableLockUpgrades) return;

        if (isLocked && (lockAttributes == null || !lockAttributes.contains(attr))) {
            if (lockAttributes == null) lockAttributes = EnumSet.of(attr);
            else lockAttributes.add(attr);

            if (worldObj != null && !worldObj.isRemote) {
                markDirty();
                worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
            }
        } else if (!isLocked && lockAttributes != null && lockAttributes.contains(attr)) {
            lockAttributes.remove(attr);

            if (worldObj != null && !worldObj.isRemote) {
                markDirty();
                worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
            }
        }
    }

    public boolean isShrouded() {
        if (!StorageDrawers.config.cache.enableShroudUpgrades) return false;

        return shrouded;
    }

    public void setIsShrouded(boolean shrouded) {
        if (this.shrouded != shrouded) {
            this.shrouded = shrouded;

            if (worldObj != null && !worldObj.isRemote) {
                markDirty();
                worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
            }
        }
    }

    public boolean isQuantified() {
        if (!StorageDrawers.config.cache.enableQuantifyUpgrades) return false;

        return quantified;
    }

    public void setIsQuantified(boolean quantified) {
        if (this.quantified != quantified) {
            this.quantified = quantified;

            if (worldObj != null && !worldObj.isRemote) {
                markDirty();
                worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
            }
        }
    }

    @Override
    public UUID getOwner() {
        if (!StorageDrawers.config.cache.enablePersonalUpgrades) return null;

        return owner;
    }

    @Override
    public boolean setOwner(UUID owner) {
        if (!StorageDrawers.config.cache.enablePersonalUpgrades) return false;

        if ((this.owner != null && !this.owner.equals(owner)) || (owner != null && !owner.equals(this.owner))) {
            this.owner = owner;

            if (worldObj != null && !worldObj.isRemote) {
                markDirty();
                worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
            }
        }

        return true;
    }

    @Override
    public ISecurityProvider getSecurityProvider() {
        return StorageDrawers.securityRegistry.getProvider(securityKey);
    }

    @Override
    public boolean setSecurityProvider(ISecurityProvider provider) {
        if (!StorageDrawers.config.cache.enablePersonalUpgrades) return false;

        String newKey = (provider == null) ? null : provider.getProviderID();
        if ((newKey != null && !newKey.equals(securityKey)) || (securityKey != null && !securityKey.equals(newKey))) {
            securityKey = newKey;

            if (worldObj != null && !worldObj.isRemote) {
                markDirty();
                worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
            }
        }

        return true;
    }

    public boolean shouldHideUpgrades() {
        return hideUpgrade;
    }

    public void setShouldHideUpgrades(boolean hide) {
        hideUpgrade = hide;

        if (worldObj != null && !worldObj.isRemote) {
            markDirty();
            worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
        }
    }

    public boolean isSealed() {
        if (!StorageDrawers.config.cache.enableTape) return false;

        return taped;
    }

    public boolean setIsSealed(boolean state) {
        if (!StorageDrawers.config.cache.enableTape) return false;

        if (this.taped != state) {
            this.taped = state;

            if (worldObj != null && !worldObj.isRemote) {
                markDirty();
                worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
            }
        }

        return true;
    }

    public boolean isDowngraded() {
        return downgraded;
    }

    public boolean canDowngrade() {
        // TODO: Config check
        return true;
    }

    public boolean checkDowngraded() {
        for (ItemStack stack : upgrades) {
            if (stack != null && stack.getItem() == ModItems.upgradeDowngrade) {
                return (downgraded = true);
            }
        }
        return (downgraded = false);
    }

    public void setDowngraded(boolean state) {
        downgraded = state;
    }

    public boolean isVoid() {
        if (!StorageDrawers.config.cache.enableVoidUpgrades) return false;

        for (ItemStack stack : upgrades) {
            if (stack != null && stack.getItem() == ModItems.upgradeVoid) return true;
        }

        return false;
    }

    public boolean isUnlimited() {
        if (!StorageDrawers.config.cache.enableCreativeUpgrades) return false;

        for (ItemStack stack : upgrades) {
            if (stack != null && stack.getItem() == ModItems.upgradeCreative) return true;
        }

        return false;
    }

    public boolean isVending() {
        if (!StorageDrawers.config.cache.enableCreativeUpgrades) return false;

        for (ItemStack stack : upgrades) {
            if (stack != null && stack.getItem() == ModItems.upgradeCreative && stack.getItemDamage() == 1) return true;
        }

        return false;
    }

    public boolean isRedstone() {
        if (!StorageDrawers.config.cache.enableRedstoneUpgrades) return false;

        for (ItemStack stack : upgrades) {
            if (stack != null && stack.getItem() == ModItems.upgradeRedstone) return true;
        }

        return false;
    }

    public int getRedstoneLevel() {
        int redstoneType = -1;
        for (ItemStack stack : upgrades) {
            if (stack != null && stack.getItem() == ModItems.upgradeRedstone) {
                redstoneType = stack.getItemDamage();
                break;
            }
        }

        switch (redstoneType) {
            case 0:
                return getCombinedRedstoneLevel();
            case 1:
                return getMaxRedstoneLevel();
            case 2:
                return getMinRedstoneLevel();
            default:
                return 0;
        }
    }

    protected int getCombinedRedstoneLevel() {
        int active = 0;
        float fillRatio = 0;

        for (int i = 0; i < getDrawerCount(); i++) {
            if (!isDrawerEnabled(i)) continue;

            IDrawer drawer = getDrawer(i);
            if (drawer.getMaxCapacity() > 0)
                fillRatio += ((float) drawer.getStoredItemCount() / drawer.getMaxCapacity());

            active++;
        }

        if (active == 0) return 0;

        if (fillRatio == active) return 15;

        return (int) Math.ceil((fillRatio / active) * 14);
    }

    protected int getMinRedstoneLevel() {
        float minRatio = 2;

        for (int i = 0; i < getDrawerCount(); i++) {
            if (!isDrawerEnabled(i)) continue;

            IDrawer drawer = getDrawer(i);
            if (drawer.getMaxCapacity() > 0)
                minRatio = Math.min(minRatio, (float) drawer.getStoredItemCount() / drawer.getMaxCapacity());
            else minRatio = 0;
        }

        if (minRatio > 1) return 0;
        if (minRatio == 1) return 15;

        return (int) Math.ceil(minRatio * 14);
    }

    protected int getMaxRedstoneLevel() {
        float maxRatio = 0;

        for (int i = 0; i < getDrawerCount(); i++) {
            if (!isDrawerEnabled(i)) continue;

            IDrawer drawer = getDrawer(i);
            if (drawer.getMaxCapacity() > 0)
                maxRatio = Math.max(maxRatio, (float) drawer.getStoredItemCount() / drawer.getMaxCapacity());
        }

        if (maxRatio == 1) return 15;

        return (int) Math.ceil(maxRatio * 14);
    }

    public boolean isSorting() {
        return false;
    }

    public ItemStack getMaterialSide() {
        return materialSide;
    }

    public ItemStack getMaterialFront() {
        return materialFront;
    }

    public ItemStack getMaterialTrim() {
        return materialTrim;
    }

    public ItemStack getEffectiveMaterialSide() {
        return materialSide;
    }

    public ItemStack getEffectiveMaterialFront() {
        return materialFront != null ? materialFront : materialSide;
    }

    public ItemStack getEffectiveMaterialTrim() {
        return materialTrim != null ? materialTrim : materialSide;
    }

    public void setMaterialSide(ItemStack material) {
        materialSide = material;
    }

    public void setMaterialFront(ItemStack material) {
        materialFront = material;
    }

    public void setMaterialTrim(ItemStack material) {
        materialTrim = material;
    }

    /**
     * Returns an ItemStack that will have a maximum size of {@link ItemStack#getMaxStackSize()}
     */
    public ItemStack takeItemsFromSlot(int slot, int count) {
        if (slot < 0 || slot >= getDrawerCount()) return null;

        ItemStack stack = getItemsFromSlot(slot, count);
        if (stack == null) return null;

        IDrawer drawer = drawers[slot];
        drawer.setStoredItemCount(drawer.getStoredItemCount() - stack.stackSize);

        if (isRedstone() && worldObj != null) {
            worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord, zCoord, getBlockType());
            worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord - 1, zCoord, getBlockType());
        }

        // TODO: Reset empty drawer in subclasses

        return stack;
    }

    /**
     * Returns an ItemStack that will have a maximum size of {@link ItemStack#getMaxStackSize()}
     */
    protected ItemStack getItemsFromSlot(int slot, int count) {
        if (drawers[slot].isEmpty()) return null;

        ItemStack stack = drawers[slot].getStoredItemCopy();
        stack.stackSize = Math.min(stack.getMaxStackSize(), count);
        stack.stackSize = Math.min(stack.stackSize, drawers[slot].getStoredItemCount());

        return stack;
    }

    public int putItemsIntoSlot(int slot, ItemStack stack, int count) {
        if (slot < 0 || slot >= getDrawerCount()) return 0;

        IDrawer drawer = drawers[slot];
        if (drawer.isEmpty()) drawer.setStoredItem(stack, 0);

        if (!drawer.canItemBeStored(stack)) return 0;

        int countAdded = Math.min(count, stack.stackSize);
        if (!isVoid()) countAdded = Math.min(countAdded, drawer.getRemainingCapacity());

        drawer.setStoredItemCount(drawer.getStoredItemCount() + countAdded);
        stack.stackSize -= countAdded;

        return countAdded;
    }

    public int interactPutCurrentItemIntoSlot(int slot, EntityPlayer player) {
        int count = 0;
        ItemStack playerStack = player.inventory.getCurrentItem();
        if (playerStack != null) count = putItemsIntoSlot(slot, playerStack, playerStack.stackSize);

        if (count > 0) markDirty();

        return count;
    }

    public int interactPutCurrentInventoryIntoSlot(int slot, EntityPlayer player) {
        int count = 0;

        if (!drawers[slot].isEmpty()) {
            for (int i = 0, n = player.inventory.getSizeInventory(); i < n; i++) {
                ItemStack subStack = player.inventory.getStackInSlot(i);
                if (subStack != null) {
                    int subCount = putItemsIntoSlot(slot, subStack, subStack.stackSize);
                    if (subCount > 0 && subStack.stackSize == 0) player.inventory.setInventorySlotContents(i, null);

                    count += subCount;
                }
            }
        }

        if (count > 0) StorageDrawers.proxy.updatePlayerInventory(player);

        markDirty();
        return count;
    }

    public int interactPutItemsIntoSlot(int slot, EntityPlayer player) {
        int count = 0;
        if (worldObj.getTotalWorldTime() - lastClickTime < 10 && player.getPersistentID().equals(lastClickUUID))
            count = interactPutCurrentInventoryIntoSlot(slot, player);
        else count = interactPutCurrentItemIntoSlot(slot, player);

        lastClickTime = worldObj.getTotalWorldTime();
        lastClickUUID = player.getPersistentID();

        return count;
    }

    private void readLegacyUpgradeNBT(NBTTagCompound tag) {
        if (tag.hasKey("Lev") && tag.getByte("Lev") > 1)
            addUpgrade(new ItemStack(ModItems.upgrade, 1, tag.getByte("Lev")));
        if (tag.hasKey("Stat")) addUpgrade(new ItemStack(ModItems.upgradeStatus, 1, tag.getByte("Stat")));
        if (tag.hasKey("Void")) addUpgrade(new ItemStack(ModItems.upgradeVoid));
        if (tag.hasKey("Down")) addUpgrade(new ItemStack(ModItems.upgradeDowngrade));
    }

    @Override
    protected void readFromFixedNBT(NBTTagCompound tag) {
        super.readFromFixedNBT(tag);

        setDirection(tag.getByte("Dir"));

        taped = false;
        if (tag.hasKey("Tape")) taped = tag.getBoolean("Tape");

        customName = null;
        if (tag.hasKey("CustomName", Constants.NBT.TAG_STRING)) customName = tag.getString("CustomName");

        downgraded = tag.hasKey("Down") && tag.getBoolean("Down");
    }

    @Override
    protected void writeToFixedNBT(NBTTagCompound tag) {
        super.writeToFixedNBT(tag);

        tag.setByte("Dir", (byte) direction);

        if (taped) tag.setBoolean("Tape", taped);

        if (hasCustomInventoryName()) tag.setString("CustomName", customName);

        if (checkDowngraded()) tag.setBoolean("Down", downgraded);
    }

    @Override
    public void readFromPortableNBT(NBTTagCompound tag) {
        super.readFromPortableNBT(tag);

        upgrades = new ItemStack[upgrades.length];

        drawerCapacity = tag.getInteger("Cap");

        if (!tag.hasKey("Upgrades")) {
            readLegacyUpgradeNBT(tag);
        } else {
            NBTTagList upgradeList = tag.getTagList("Upgrades", Constants.NBT.TAG_COMPOUND);
            for (int i = 0; i < upgradeList.tagCount(); i++) {
                NBTTagCompound upgradeTag = upgradeList.getCompoundTagAt(i);

                int slot = upgradeTag.getByte("Slot");
                setUpgrade(slot, ItemStack.loadItemStackFromNBT(upgradeTag));
            }
        }

        lockAttributes = null;
        if (tag.hasKey("Lock")) lockAttributes = LockAttribute.getEnumSet(tag.getByte("Lock"));

        shrouded = false;
        if (tag.hasKey("Shr")) shrouded = tag.getBoolean("Shr");

        quantified = false;
        if (tag.hasKey("Qua")) quantified = tag.getBoolean("Qua");

        owner = null;
        if (tag.hasKey("Own")) owner = UUID.fromString(tag.getString("Own"));

        securityKey = null;
        if (tag.hasKey("Sec")) securityKey = tag.getString("Sec");

        hideUpgrade = false;
        if (tag.hasKey("HideUp")) hideUpgrade = tag.getBoolean("HideUp");

        NBTTagList slots = tag.getTagList("Slots", Constants.NBT.TAG_COMPOUND);
        drawers = new IDrawer[slots.tagCount()];

        for (int i = 0, n = drawers.length; i < n; i++) {
            NBTTagCompound slot = slots.getCompoundTagAt(i);
            drawers[i] = createDrawer(i);
            drawers[i].readFromNBT(slot);
        }

        inventory = new StorageInventory(this, getSideManager(), this);

        materialSide = null;
        if (tag.hasKey("MatS")) materialSide = ItemStack.loadItemStackFromNBT(tag.getCompoundTag("MatS"));

        materialFront = null;
        if (tag.hasKey("MatF")) materialFront = ItemStack.loadItemStackFromNBT(tag.getCompoundTag("MatF"));

        materialTrim = null;
        if (tag.hasKey("MatT")) materialTrim = ItemStack.loadItemStackFromNBT(tag.getCompoundTag("MatT"));
    }

    @Override
    public void writeToPortableNBT(NBTTagCompound tag) {
        super.writeToPortableNBT(tag);

        tag.setInteger("Cap", drawerCapacity);

        NBTTagList upgradeList = new NBTTagList();
        for (int i = 0; i < upgrades.length; i++) {
            if (upgrades[i] != null) {
                NBTTagCompound upgradeTag = upgrades[i].writeToNBT(new NBTTagCompound());
                upgradeTag.setByte("Slot", (byte) i);

                upgradeList.appendTag(upgradeTag);
            }
        }

        if (upgradeList.tagCount() > 0) tag.setTag("Upgrades", upgradeList);

        if (lockAttributes != null) tag.setByte("Lock", (byte) LockAttribute.getBitfield(lockAttributes));

        if (shrouded) tag.setBoolean("Shr", shrouded);

        if (quantified) tag.setBoolean("Qua", quantified);

        if (owner != null) tag.setString("Own", owner.toString());

        if (securityKey != null) tag.setString("Sec", securityKey);

        if (hideUpgrade) tag.setBoolean("HideUp", hideUpgrade);

        NBTTagList slots = new NBTTagList();
        for (IDrawer drawer : drawers) {
            NBTTagCompound slot = new NBTTagCompound();
            drawer.writeToNBT(slot);
            slots.appendTag(slot);
        }

        tag.setTag("Slots", slots);

        if (materialSide != null) {
            NBTTagCompound itag = new NBTTagCompound();
            materialSide.writeToNBT(itag);
            tag.setTag("MatS", itag);
        }

        if (materialFront != null) {
            NBTTagCompound itag = new NBTTagCompound();
            materialFront.writeToNBT(itag);
            tag.setTag("MatF", itag);
        }

        if (materialTrim != null) {
            NBTTagCompound itag = new NBTTagCompound();
            materialTrim.writeToNBT(itag);
            tag.setTag("MatT", itag);
        }
    }

    @Override
    public boolean canUpdate() {
        return false;
    }

    @Override
    public void markDirty() {
        inventory.markDirty();
        if (isRedstone() && worldObj != null) {
            worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord, zCoord, getBlockType());
            worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord - 1, zCoord, getBlockType());
        }

        super.markDirty();
    }

    @Override
    public boolean markDirtyIfNeeded() {
        if (inventory.syncInventoryIfNeeded()) {
            super.markDirty();
            return true;
        }

        return false;
    }

    public void clientUpdateCount(int slot, int count) {
        IDrawer drawer = getDrawer(slot);
        if (drawer == null) {
            FMLLog.log(
                    StorageDrawers.MOD_ID,
                    Level.ERROR,
                    "Drawer at (" + this.xCoord
                            + ","
                            + this.yCoord
                            + ","
                            + this.zCoord
                            + ") has no slot "
                            + slot
                            + ". Drawers.length is "
                            + drawers.length);
            return;
        }
        if (drawer.getStoredItemCount() != count) {
            drawer.setStoredItemCount(count);

            switch (getEffectiveStatusLevel()) {
                case 1:
                    if (drawer.getStoredItemCount() == 0 || drawer.getRemainingCapacity() == 0)
                        getWorldObj().func_147479_m(xCoord, yCoord, zCoord); // markBlockForRenderUpdate
                    break;
                case 2:
                    getWorldObj().func_147479_m(xCoord, yCoord, zCoord); // markBlockForRenderUpdate
                    break;
            }
        }
    }

    private void syncClientCount(int slot) {
        IMessage message = new CountUpdateMessage(xCoord, yCoord, zCoord, slot, drawers[slot].getStoredItemCount());
        NetworkRegistry.TargetPoint targetPoint = new NetworkRegistry.TargetPoint(
                worldObj.provider.dimensionId,
                xCoord,
                yCoord,
                zCoord,
                500);

        StorageDrawers.network.sendToAllAround(message, targetPoint);
    }

    // TODO: Eventually eliminate these expensive network updates
    @Override
    public Packet getDescriptionPacket() {
        NBTTagCompound tag = new NBTTagCompound();
        writeToNBT(tag);

        return new S35PacketUpdateTileEntity(xCoord, yCoord, zCoord, 5, tag);
    }

    @Override
    public void onDataPacket(NetworkManager net, S35PacketUpdateTileEntity pkt) {
        readFromNBT(pkt.func_148857_g());
        getWorldObj().func_147479_m(xCoord, yCoord, zCoord); // markBlockForRenderUpdate
    }

    @Override
    public int getDrawerCount() {
        return drawers.length;
    }

    @Override
    @Nullable
    public IDrawer getDrawer(int slot) {
        if (slot < 0 || slot >= drawers.length) return null;

        return drawers[slot];
    }

    @Override
    public IDrawerInventory getDrawerInventory() {
        return inventory;
    }

    @Override
    public boolean isDrawerEnabled(int slot) {
        if (isSealed()) return false;

        if (getBlockType() instanceof BlockDrawersCustom && materialSide == null) return false;

        return getDrawer(slot) != null;
    }

    @Override
    public int[] getAccessibleSlotsFromSide(int side) {
        return inventory.getAccessibleSlotsFromSide(side);
    }

    @Override
    public boolean canInsertItem(int slot, ItemStack stack, int side) {
        if (isSealed()) return false;

        if (isLocked(LockAttribute.LOCK_EMPTY) && inventory instanceof StorageInventory) {
            IDrawer drawer = getDrawer(inventory.getDrawerSlot(slot));
            if (drawer != null && drawer.isEmpty()) return false;
        }

        return inventory.canInsertItem(slot, stack, side);
    }

    @Override
    public boolean canExtractItem(int slot, ItemStack stack, int side) {
        if (isSealed()) return false;

        return inventory.canExtractItem(slot, stack, side);
    }

    @Override
    public int getSizeInventory() {
        return inventory.getSizeInventory();
    }

    @Override
    public ItemStack getStackInSlot(int slot) {
        return inventory.getStackInSlot(slot);
    }

    @Override
    public ItemStack decrStackSize(int slot, int count) {
        return inventory.decrStackSize(slot, count);
    }

    @Override
    public ItemStack getStackInSlotOnClosing(int slot) {
        return inventory.getStackInSlotOnClosing(slot);
    }

    @Override
    public void setInventorySlotContents(int slot, ItemStack stack) {
        inventory.setInventorySlotContents(slot, stack);
    }

    public void setInventoryName(String name) {
        customName = name;
    }

    @Override
    public String getInventoryName() {
        return hasCustomInventoryName() ? customName : "storageDrawers.container.drawers";
    }

    @Override
    public boolean hasCustomInventoryName() {
        return customName != null && customName.length() > 0;
    }

    @Override
    public int getInventoryStackLimit() {
        return inventory.getInventoryStackLimit();
    }

    @Override
    public boolean isUseableByPlayer(EntityPlayer player) {
        if (worldObj.getTileEntity(xCoord, yCoord, zCoord) != this) return false;

        return player.getDistanceSq(xCoord + .5, yCoord + .5, zCoord + .5) <= 64;
    }

    @Override
    public void openInventory() {
        inventory.openInventory();
    }

    @Override
    public void closeInventory() {
        inventory.closeInventory();
    }

    @Override
    public boolean isItemValidForSlot(int slot, ItemStack stack) {
        return inventory.isItemValidForSlot(slot, stack);
    }

    private class DefaultSideManager implements ISideManager {

        @Override
        public int[] getSlotsForSide(int side) {
            return autoSides;
        }
    }
}
