package com.atsuishio.superbwarfare.client.screens;

import com.atsuishio.superbwarfare.Mod;
import com.atsuishio.superbwarfare.block.ContainerBlock;
import com.atsuishio.superbwarfare.client.RenderHelper;
import com.atsuishio.superbwarfare.client.animation.AnimationCurves;
import com.atsuishio.superbwarfare.client.animation.ValueAnimator;
import com.atsuishio.superbwarfare.client.screens.component.*;
import com.atsuishio.superbwarfare.compat.jei.JeiCompatHolder;
import com.atsuishio.superbwarfare.compat.jei.SbwJEIPlugin;
import com.atsuishio.superbwarfare.init.ModItems;
import com.atsuishio.superbwarfare.init.ModRecipes;
import com.atsuishio.superbwarfare.menu.VehicleAssemblingMenu;
import com.atsuishio.superbwarfare.network.message.send.AssembleVehicleMessage;
import com.atsuishio.superbwarfare.recipe.vehicle.VehicleAssemblingRecipe;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.mojang.blaze3d.platform.GlStateManager;
import com.mojang.blaze3d.platform.Lighting;
import com.mojang.blaze3d.platform.Window;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.PoseStack;
import com.mojang.math.Axis;
import it.unimi.dsi.fastutil.ints.Int2IntArrayMap;
import net.minecraft.ChatFormatting;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.screens.inventory.AbstractContainerScreen;
import net.minecraft.client.player.LocalPlayer;
import net.minecraft.client.renderer.MultiBufferSource;
import net.minecraft.client.renderer.entity.EntityRenderDispatcher;
import net.minecraft.client.renderer.texture.OverlayTexture;
import net.minecraft.client.renderer.texture.TextureAtlas;
import net.minecraft.core.component.DataComponents;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.FormattedText;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.util.FormattedCharSequence;
import net.minecraft.util.Mth;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.entity.player.Inventory;
import net.minecraft.world.item.ItemDisplayContext;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.crafting.Ingredient;
import net.minecraft.world.item.crafting.RecipeHolder;
import net.minecraft.world.item.crafting.RecipeManager;
import net.minecraft.world.phys.Vec2;
import net.neoforged.api.distmarker.Dist;
import net.neoforged.api.distmarker.OnlyIn;
import net.neoforged.neoforge.network.PacketDistributor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Code based on TaC-Z
 */
@OnlyIn(Dist.CLIENT)
public class VehicleAssemblingScreen extends AbstractContainerScreen<VehicleAssemblingMenu> {

    public static final ResourceLocation TEXTURE = Mod.loc("textures/gui/vehicle_assembling_table.png");
    public static final int IMAGE_SIZE = 356;
    public static final int PAGE_SIZE = 9;

    public static final float DEFAULT_MODEL_SCALE = 50f;
    public static final float MIN_MODEL_SCALE = 10f;
    public static final float MAX_MODEL_SCALE = 200f;

    public static final int DEFAULT_MODEL_X = 234;
    public static final int DEFAULT_MODEL_Y = 80;

    private final Map<VehicleAssemblingRecipe.Category, List<ResourceLocation>> recipes = Maps.newLinkedHashMap();

    private VehicleAssemblingRecipe.Category currentCategory = VehicleAssemblingRecipe.Category.LAND;
    @Nullable
    private List<ResourceLocation> currentRecipes = new ArrayList<>();
    @Nullable
    private RecipeHolder<VehicleAssemblingRecipe> currentRecipe = null;
    @Nullable
    private Int2IntArrayMap materialCount;
    private int pageIndex = 0;

    private String entityNameCache = "";
    private Entity entityCache = null;

    public VehicleAssemblingScreen(VehicleAssemblingMenu pMenu, Inventory pPlayerInventory, Component pTitle) {
        super(pMenu, pPlayerInventory, pTitle);
        imageWidth = 356;
        imageHeight = 181;
        this.initRecipes();
        this.pageIndex = 0;
        this.currentRecipe = this.getRecipeById(this.currentRecipes == null || this.currentRecipes.isEmpty() ? null : this.currentRecipes.getFirst());
        this.calculateMaterialCount(this.currentRecipe);
    }

    @Override
    protected void init() {
        super.init();
        this.initRecipes();
        this.clearWidgets();

        int posX = (this.width - this.imageWidth) / 2;
        int posY = (this.height - this.imageHeight) / 2;

        this.addCategoryButtons(posX, posY);
        this.addRecipeButtons(posX, posY);
        this.addPageButtons(posX, posY);
        this.addAssembleButton(posX, posY);
        this.addScaleButtons(posX, posY);
    }

    public void initRecipes() {
        this.recipes.clear();

        Minecraft mc = Minecraft.getInstance();
        if (mc.level == null) return;

        RecipeManager recipeManager = Minecraft.getInstance().level.getRecipeManager();
        var recipeList = recipeManager.getAllRecipesFor(ModRecipes.VEHICLE_ASSEMBLING_TYPE.get());

        for (var recipe : recipeList) {
            this.recipes.computeIfAbsent(recipe.value().getCategory(), k -> Lists.newArrayList()).add(recipe.id());
        }
        this.currentRecipes = this.recipes.get(this.currentCategory);
    }

    public void addCategoryButtons(int posX, int posY) {
        int i = 0;
        for (var category : VehicleAssemblingRecipe.Category.values()) {
            CategoryButton button = new CategoryButton(posX, posY + 21 + i * 23, category, b -> {
                this.currentCategory = category;
                this.currentRecipes = this.recipes.get(category);
                this.currentRecipe = this.getRecipeById(this.currentRecipes == null || this.currentRecipes.isEmpty() ? null : this.currentRecipes.getFirst());
                this.pageIndex = 0;
                this.calculateMaterialCount(this.currentRecipe);
                this.init();
            });
            if (this.currentCategory.equals(category)) {
                button.setSelected(true);
            }
            this.addRenderableWidget(button);
            i++;
        }
    }

    @Override
    public void render(@NotNull GuiGraphics guiGraphics, int mouseX, int mouseY, float partialTick) {
        this.renderBackground(guiGraphics, mouseX, mouseY, partialTick);
        super.render(guiGraphics, mouseX, mouseY, partialTick);
        this.renderTooltip(guiGraphics, mouseX, mouseY);

        if (this.currentRecipe != null) {
            this.renderModel(this.currentRecipe, guiGraphics);
            this.renderRecipeInfo(this.currentRecipe, guiGraphics, mouseX, mouseY);
            guiGraphics.drawString(this.font, Component.translatable("container.superbwarfare.vehicle_assembling_table.count", this.currentRecipe.value().getResult().getResult().getCount()), this.leftPos + 214, this.topPos + 164, 5592405, false);
        }

        if (this.currentRecipes != null && !this.currentRecipes.isEmpty()) {
            this.renderIngredients(guiGraphics, mouseX, mouseY);
        }

        this.renderables.stream().filter(w -> w instanceof RecipeButton || w instanceof CategoryButton)
                .forEach(w -> {
                    if (w instanceof RecipeButton recipeButton) {
                        recipeButton.renderTooltips(guiGraphics, mouseX, mouseY);
                    }
                    if (w instanceof CategoryButton categoryButton) {
                        categoryButton.renderTooltips(guiGraphics, mouseX, mouseY);
                    }
                });
    }

    @Override
    protected void renderBg(@NotNull GuiGraphics pGuiGraphics, float pPartialTick, int pMouseX, int pMouseY) {
        int i = (this.width - this.imageWidth) / 2;
        int j = (this.height - this.imageHeight) / 2;
        pGuiGraphics.blit(TEXTURE, i, j, 0, 0, this.imageWidth, this.imageHeight, IMAGE_SIZE, IMAGE_SIZE);
    }

    // 本方法留空
    @Override
    protected void renderLabels(@NotNull GuiGraphics pGuiGraphics, int pMouseX, int pMouseY) {
    }

    @SuppressWarnings("unchecked")
    @Nullable
    private RecipeHolder<VehicleAssemblingRecipe> getRecipeById(ResourceLocation recipeId) {
        if (recipeId == null) return null;
        if (Minecraft.getInstance().level != null) {
            RecipeManager recipeManager = Minecraft.getInstance().level.getRecipeManager();
            var recipe = recipeManager.byKey(recipeId).orElse(null);
            if (recipe == null) return null;
            if (recipe.value() instanceof VehicleAssemblingRecipe) {
                return (RecipeHolder<VehicleAssemblingRecipe>) recipe;
            }
        }
        return null;
    }

    public void calculateMaterialCount(@Nullable RecipeHolder<VehicleAssemblingRecipe> holder) {
        LocalPlayer player = Minecraft.getInstance().player;
        if (player == null || holder == null) return;
        var recipe = holder.value();

        var ingredients = recipe.getInputs();
        int size = ingredients.size();
        this.materialCount = new Int2IntArrayMap(size);

        for (int i = 0; i < size; ++i) {
            var ingredient = ingredients.get(i);
            int count = 0;

            for (ItemStack stack : player.getInventory().items) {
                if (!stack.isEmpty() && ingredient.getIngredient().test(stack)) {
                    count += stack.getCount();
                }
            }

            this.materialCount.put(i, count);
        }
    }

    public void addRecipeButtons(int posX, int posY) {
        if (this.currentRecipes != null && !this.currentRecipes.isEmpty()) {
            for (int i = 0; i < 9; i++) {
                int index = i + this.pageIndex * PAGE_SIZE;
                if (index >= this.currentRecipes.size()) break;

                ResourceLocation id = this.currentRecipes.get(index);
                var recipe = this.getRecipeById(id);
                if (recipe == null) break;

                RecipeButton button = this.addRenderableWidget(new RecipeButton(posX + 26, posY + 21 + i * 17, recipe.value().getResult().getResult(), (b) -> {
                    this.currentRecipe = recipe;
                    this.calculateMaterialCount(recipe);
                    this.init();
                }));
                if (recipe.equals(this.currentRecipe)) {
                    button.setSelected(true);
                }
            }
        }
    }

    private void renderIngredients(GuiGraphics guiGraphics, int mouseX, int mouseY) {
        if (this.currentRecipe == null) return;
        var inputs = this.currentRecipe.value().getInputs();

        int x = (this.width - this.imageWidth) / 2;
        int y = (this.height - this.imageHeight) / 2;

        for (int i = 0; i < 3; i++) {
            for (int j = 0; j < 4; j++) {
                int index = i * 4 + j;
                if (index >= inputs.size()) return;

                int posX = x + 215 + j * 34;
                int posY = y + 118 + i * 14;

                var input = inputs.get(index);
                var ingredient = input.getIngredient();
                var items = ingredient.getItems();
                if (items.length == 0) continue;

                int itemIndex = (int) (System.currentTimeMillis() / 1000L) % items.length;
                var itemStack = items[itemIndex];

                var pose = guiGraphics.pose();

                pose.pushPose();
                pose.scale(0.8F, 0.8F, 1F);
                guiGraphics.renderFakeItem(itemStack, (int) (posX * 1.25f), (int) (posY * 1.25f));
                pose.popPose();

                if (mouseX >= posX && mouseY >= posY && mouseX < posX + 16 * 0.8f && mouseY < posY + 16 * 0.8f) {
                    guiGraphics.renderTooltip(this.font, itemStack, mouseX, mouseY);
                }

                pose.pushPose();
                pose.scale(0.5F, 0.5F, 1F);
                pose.translate(0, 0, 200F);

                int count = input.getCount();
                if (Minecraft.getInstance().player != null && Minecraft.getInstance().player.isCreative()) {
                    Component text = Component.literal(count + "/∞");
                    guiGraphics.drawString(this.font, text, (posX + 14) * 2, (posY + 8) * 2, 0x2C3141, false);
                } else {
                    int hasCount = 0;
                    if (this.materialCount != null && index < this.materialCount.size()) {
                        hasCount = this.materialCount.get(index);
                    }
                    int color = hasCount >= count ? 0x2C3141 : 0xf44d61;
                    Component text = Component.literal(count + "/" + hasCount);
                    guiGraphics.drawString(this.font, text, (posX + 14) * 2, (posY + 8) * 2, color, false);
                }
                pose.popPose();
            }
        }
    }

    @SuppressWarnings("unchecked")
    private final ValueAnimator<Float> scaleAnimator = (ValueAnimator<Float>) new ValueAnimator<>(300, DEFAULT_MODEL_SCALE)
            .animation(AnimationCurves.EASE_OUT_EXPO);

    @SuppressWarnings("unchecked")
    private final ValueAnimator<Vec2> modelPosAnimator = (ValueAnimator<Vec2>) new ValueAnimator<>(300, new Vec2(DEFAULT_MODEL_X, DEFAULT_MODEL_Y))
            .animation(AnimationCurves.EASE_OUT_EXPO);

    @Override
    public boolean mouseDragged(double pMouseX, double pMouseY, int pButton, double pDragX, double pDragY) {
        if (pMouseX >= this.leftPos + 114 && pMouseX <= this.leftPos + 354 && pMouseY >= this.topPos && pMouseY <= this.topPos + 99) {
            var newVec = modelPosAnimator.newValue();
            var posX = Mth.clamp(newVec.x + pDragX, DEFAULT_MODEL_X - 200, DEFAULT_MODEL_X + 200);
            var posY = Mth.clamp(newVec.y + pDragY, DEFAULT_MODEL_Y - 150, DEFAULT_MODEL_Y + 150);
            modelPosAnimator.update(new Vec2((float) posX, (float) posY));
            return true;
        }
        return super.mouseDragged(pMouseX, pMouseY, pButton, pDragX, pDragY);
    }

    @Override
    public boolean mouseScrolled(double pMouseX, double pMouseY, double scrollX, double scrollY) {
        if (pMouseX >= this.leftPos + 26 && pMouseX <= this.leftPos + 106 && pMouseY >= this.topPos + 21 && pMouseY <= this.topPos + 175) {
            if (scrollY > 0) {
                this.pageIndex = Math.max(0, this.pageIndex - 1);
            } else {
                if (this.currentRecipes != null && !this.currentRecipes.isEmpty()) {
                    this.pageIndex = Math.min((this.currentRecipes.size() - 1) / PAGE_SIZE, this.pageIndex + 1);
                }
            }

            this.init();
            return true;
        }
        if (pMouseX >= this.leftPos + 114 && pMouseX <= this.leftPos + 354 && pMouseY >= this.topPos && pMouseY <= this.topPos + 99) {
            float targetScale;
            if (scrollY > 0) {
                targetScale = Math.min(scaleAnimator.lerp(scaleAnimator.oldValue(), scaleAnimator.newValue(), System.currentTimeMillis()) + 20, MAX_MODEL_SCALE);
            } else {
                targetScale = Math.max(scaleAnimator.lerp(scaleAnimator.oldValue(), scaleAnimator.newValue(), System.currentTimeMillis()) - 20, MIN_MODEL_SCALE);
            }

            scaleAnimator.update(targetScale);
            scaleAnimator.beginForward(System.currentTimeMillis());

            return true;
        }
        return super.mouseScrolled(pMouseX, pMouseY, scrollX, scrollY);
    }

    @Override
    public boolean mouseClicked(double pMouseX, double pMouseY, int pButton) {
        var list = this.getIngredientAreas();
        if (!list.isEmpty() && pMouseX >= this.leftPos + 214 && pMouseY >= this.topPos + 117 && pMouseX <= this.leftPos + 350 && pMouseY <= this.topPos + 160) {
            if (JeiCompatHolder.hasJEI()) {
                var ingredientArea = list.stream().filter(area -> area.contains(pMouseX, pMouseY)).findFirst();
                if (ingredientArea.isPresent()) {
                    var items = ingredientArea.get().ingredient.getItems();
                    int itemIndex = (int) (System.currentTimeMillis() / 1000L) % items.length;
                    SbwJEIPlugin.showRecipes(items[itemIndex]);
                    return true;
                }
            }
        }
        return super.mouseClicked(pMouseX, pMouseY, pButton);
    }

    public void addPageButtons(int posX, int posY) {
        PageButton left = this.addRenderableWidget(new PageButton(posX + 95, posY - 1, true, b -> {
            this.pageIndex = Math.max(0, this.pageIndex - 1);
            this.init();
        }));
        PageButton right = this.addRenderableWidget(new PageButton(posX + 103, posY - 1, false, b -> {
            if (this.currentRecipes != null && !this.currentRecipes.isEmpty()) {
                this.pageIndex = Math.min((this.currentRecipes.size() - 1) / PAGE_SIZE, this.pageIndex + 1);
                this.init();
            }
        }));
        if (this.currentRecipes != null && !this.currentRecipes.isEmpty()) {
            left.active = this.pageIndex > 0;
            right.active = this.pageIndex < (this.currentRecipes.size() - 1) / PAGE_SIZE;
        } else {
            left.active = false;
            right.active = false;
        }
    }

    public void addAssembleButton(int posX, int posY) {
        this.addRenderableWidget(new AssembleButton(posX + 295, posY + 163, b -> {
            if (this.currentRecipe == null || this.materialCount == null) return;

            var inputs = this.currentRecipe.value().getInputs();
            int size = inputs.size();

            for (int i = 0; i < size; ++i) {
                if (i >= this.materialCount.size()) {
                    return;
                }

                int hasCount = this.materialCount.get(i);
                int needCount = inputs.get(i).getCount();
                boolean isCreative = Minecraft.getInstance().player != null && Minecraft.getInstance().player.isCreative();
                if (hasCount < needCount && !isCreative) {
                    return;
                }
            }
            PacketDistributor.sendToServer(new AssembleVehicleMessage(this.currentRecipe.id(), this.menu.containerId));
        }));
    }

    public void finishAssembling() {
        if (this.currentRecipe != null) {
            this.calculateMaterialCount(this.currentRecipe);
        }
        this.init();
    }

    public void addScaleButtons(int posX, int posY) {
        this.addRenderableWidget(new ScaleButton(posX + 324, posY + 90, 149, 182,
                b -> {
                    long time = System.currentTimeMillis();
                    scaleAnimator.update(DEFAULT_MODEL_SCALE);
                    scaleAnimator.beginForward(time);
                    modelPosAnimator.update(new Vec2(DEFAULT_MODEL_X, DEFAULT_MODEL_Y));
                    modelPosAnimator.beginForward(time);
                }));
        this.addRenderableWidget(new ScaleButton(posX + 334, posY + 90, 159, 182,
                b -> {
                    scaleAnimator.update(Math.max(scaleAnimator.lerp(scaleAnimator.oldValue(), scaleAnimator.newValue(), System.currentTimeMillis()) - 20, MIN_MODEL_SCALE));
                    scaleAnimator.beginForward(System.currentTimeMillis());
                }));
        this.addRenderableWidget(new ScaleButton(posX + 344, posY + 90, 169, 182,
                b -> {
                    scaleAnimator.update(Math.min(scaleAnimator.lerp(scaleAnimator.oldValue(), scaleAnimator.newValue(), System.currentTimeMillis()) + 20, MAX_MODEL_SCALE));
                    scaleAnimator.beginForward(System.currentTimeMillis());
                }));
    }

    public void renderModel(RecipeHolder<VehicleAssemblingRecipe> holder, GuiGraphics guiGraphics) {
        Minecraft mc = Minecraft.getInstance();
        var level = mc.level;
        if (level == null) return;

        RenderHelper.markGuiRenderTimestamp();
        ItemStack stack = holder.value().getResult().getResult();
        Entity renderEntity = null;

        if (stack.is(ModItems.CONTAINER.get())) {
            var data = stack.get(DataComponents.BLOCK_ENTITY_DATA);
            var tag = data != null ? data.copyTag() : null;
            typeFlag:
            if (tag != null && tag.contains("EntityType")) {
                String key = tag.getString("EntityType");
                if (entityNameCache.equals(key) && entityCache != null) {
                    renderEntity = entityCache;
                } else {
                    renderEntity = EntityType.byString(key)
                            .map(type -> type.create(level))
                            .orElse(null);
                    if (renderEntity == null) break typeFlag;

                    entityNameCache = key;
                    entityCache = renderEntity;

                    scaleAnimator.update(DEFAULT_MODEL_SCALE);
                    modelPosAnimator.update(new Vec2(DEFAULT_MODEL_X, DEFAULT_MODEL_Y));
                }
            }
        }

        if (renderEntity == null) {
            renderDefaultItemModel(stack);
        } else {
            renderEntityModel(guiGraphics, renderEntity);
        }
    }

    @SuppressWarnings("deprecation")
    private void renderDefaultItemModel(ItemStack stack) {
        float rotationPeriod = 8F;
        int width = 240;
        int height = 99;
        float rotPitch = 15F;

        Window window = Minecraft.getInstance().getWindow();
        double windowGuiScale = window.getGuiScale();
        int scissorX = (int) ((this.leftPos + 114) * windowGuiScale);
        int scissorY = (int) (window.getHeight() - (this.topPos + height) * windowGuiScale);
        int scissorW = (int) (width * windowGuiScale);
        int scissorH = (int) (height * windowGuiScale);
        RenderSystem.enableScissor(scissorX, scissorY, scissorW, scissorH);

        Minecraft.getInstance().getTextureManager().getTexture(TextureAtlas.LOCATION_BLOCKS).setFilter(false, false);
        RenderSystem.setShaderTexture(0, TextureAtlas.LOCATION_BLOCKS);
        RenderSystem.enableBlend();
        RenderSystem.blendFunc(GlStateManager.SourceFactor.SRC_ALPHA, GlStateManager.DestFactor.ONE_MINUS_SRC_ALPHA);
        RenderSystem.setShaderColor(1F, 1F, 1F, 1F);

        var posestack = RenderSystem.getModelViewStack();
        posestack.pushMatrix();
        var oldVec = modelPosAnimator.oldValue();
        var newVec = modelPosAnimator.newValue();
        var xOffset = modelPosAnimator.lerp(oldVec.x, newVec.x, System.currentTimeMillis());
        var yOffset = modelPosAnimator.lerp(oldVec.y, newVec.y, System.currentTimeMillis());
        posestack.translate(this.leftPos + xOffset, this.topPos + yOffset - 20, 200F);
        posestack.translate(8.0f, 8.0f, 0.0f);
        posestack.scale(1, -1, 1);
        var currentScale = scaleAnimator.lerp(scaleAnimator.oldValue(), scaleAnimator.newValue(), System.currentTimeMillis());
        posestack.scale(currentScale, currentScale, currentScale);

        float rot = (float) (System.currentTimeMillis() % (long) ((int) (rotationPeriod * 1000F))) * (360F / (rotationPeriod * 1000F));

        posestack.rotate(Axis.XP.rotationDegrees(rotPitch));
        posestack.rotate(Axis.YP.rotationDegrees(rot));
        RenderSystem.applyModelViewMatrix();
        PoseStack tmpPose = new PoseStack();
        MultiBufferSource.BufferSource bufferSource = Minecraft.getInstance().renderBuffers().bufferSource();
        Lighting.setupForFlatItems();

        Minecraft.getInstance().getItemRenderer().renderStatic(stack, ItemDisplayContext.FIXED, 15728880, OverlayTexture.NO_OVERLAY, tmpPose, bufferSource, null, 0);

        bufferSource.endBatch();
        RenderSystem.enableDepthTest();
        Lighting.setupFor3DItems();
        posestack.popMatrix();
        RenderSystem.applyModelViewMatrix();
        RenderSystem.disableScissor();
    }

    private void renderEntityModel(GuiGraphics guiGraphics, Entity renderEntity) {
        if (renderEntity == null) return;

        PoseStack posestack = guiGraphics.pose();

        int width = 240;
        int height = 99;

        Window window = Minecraft.getInstance().getWindow();
        double windowGuiScale = window.getGuiScale();

        int scissorX = (int) ((this.leftPos + 114) * windowGuiScale);
        int scissorY = (int) (window.getHeight() - (this.topPos + height) * windowGuiScale);
        int scissorW = (int) (width * windowGuiScale);
        int scissorH = (int) (height * windowGuiScale);
        RenderSystem.enableScissor(scissorX, scissorY, scissorW, scissorH);

        posestack.pushPose();
        var oldVec = modelPosAnimator.oldValue();
        var newVec = modelPosAnimator.newValue();
        var xOffset = modelPosAnimator.lerp(oldVec.x, newVec.x, System.currentTimeMillis());
        var yOffset = modelPosAnimator.lerp(oldVec.y, newVec.y, System.currentTimeMillis());
        posestack.translate(this.leftPos + xOffset, this.topPos + yOffset, 50);
        var currentScale = scaleAnimator.lerp(scaleAnimator.oldValue(), scaleAnimator.newValue(), System.currentTimeMillis());
        posestack.scale(currentScale, currentScale, -currentScale);

        float size = (float) renderEntity.getBoundingBox().getSize();
        float resizeScale = 1f / Math.max(size, 1.25f);
        posestack.scale(resizeScale, resizeScale, resizeScale);

        Lighting.setupForEntityInInventory();
        EntityRenderDispatcher entityrenderdispatcher = Minecraft.getInstance().getEntityRenderDispatcher();

        float rotationPeriod = 12F;
        float rotPitch = 195F;
        float rot = (float) (System.currentTimeMillis() % (long) ((int) (rotationPeriod * 1000F))) * (360F / (rotationPeriod * 1000F));

        posestack.mulPose(Axis.XP.rotationDegrees(rotPitch));
        posestack.mulPose(Axis.YP.rotationDegrees(rot));

        entityrenderdispatcher.setRenderShadow(false);
        entityrenderdispatcher.render(renderEntity, 0, 0, 0, 0, 1F, posestack, guiGraphics.bufferSource(), 15728880);
        guiGraphics.flush();
        entityrenderdispatcher.setRenderShadow(true);
        posestack.popPose();
        Lighting.setupFor3DItems();
        RenderSystem.disableScissor();
    }

    public void renderRecipeInfo(RecipeHolder<VehicleAssemblingRecipe> holder, GuiGraphics guiGraphics, int mouseX, int mouseY) {
        ItemStack stack = holder.value().getResult().getResult();

        boolean renderItemName = true;
        if (stack.is(ModItems.CONTAINER.get())) {
            var data = stack.get(DataComponents.BLOCK_ENTITY_DATA);
            var tag = data != null ? data.copyTag() : null;
            if (tag != null && tag.contains("EntityType")) {
                String key = tag.getString("EntityType");
                var entityType = EntityType.byString(key).orElse(null);
                if (entityType != null) {
                    this.renderContainerInfo(key, guiGraphics, mouseX, mouseY);
                    renderItemName = false;
                }
            }
        }

        var pose = guiGraphics.pose();
        pose.pushPose();

        pose.scale(0.75f, 0.75f, 1.0f);

        if (renderItemName) {
            RenderHelper.renderScrollingString(guiGraphics, this.font,
                    Component.empty().append(stack.getHoverName()).withStyle(ChatFormatting.UNDERLINE).withStyle(ChatFormatting.YELLOW),
                    0.75f,
                    (int) ((this.leftPos + 122) / 0.75f), (int) ((this.topPos + 119) / 0.75f),
                    (int) ((this.leftPos + 198) / 0.75f), (int) ((this.topPos + 130) / 0.75f),
                    0xFFFFFF);
        }

        var modName = Component.translatableWithFallback("info." + holder.id().getNamespace() + ".mod_id", holder.id().getNamespace());
        var modInfo = Component.translatable("container.superbwarfare.mod_info", modName.withStyle(ChatFormatting.ITALIC).withStyle(ChatFormatting.AQUA));

        RenderHelper.renderScrollingString(guiGraphics, this.font,
                modInfo,
                0.75f,
                (int) ((this.leftPos + 122) / 0.75f), (int) ((this.topPos + 167) / 0.75f),
                (int) ((this.leftPos + 198) / 0.75f), (int) ((this.topPos + 178) / 0.75f),
                0xFFFFFF);

        pose.popPose();
    }

    private void renderContainerInfo(String typeName, GuiGraphics guiGraphics, int mouseX, int mouseY) {
        var pose = guiGraphics.pose();

        String key = ContainerBlock.getEntityTranslationKey(typeName);
        if (key == null) return;
        if (typeName.split(":").length < 2) return;

        var info = Component.translatableWithFallback("info." + typeName.split(":")[0] + "." + typeName.split(":")[1],
                Component.translatable("info.superbwarfare.no_info").getString());
        List<FormattedCharSequence> infoComponents = this.font.split(FormattedText.of(info.getString()), 100);

        pose.pushPose();
        pose.scale(0.75f, 0.75f, 1.0f);

        var hoverName = Component.translatable(key).withStyle(ChatFormatting.UNDERLINE).withStyle(ChatFormatting.YELLOW);
        RenderHelper.renderScrollingString(guiGraphics, this.font,
                hoverName,
                0.75f,
                (int) ((this.leftPos + 122) / 0.75f), (int) ((this.topPos + 119) / 0.75f),
                (int) ((this.leftPos + 198) / 0.75f), (int) ((this.topPos + 130) / 0.75f),
                0xFFFFFF);

        guiGraphics.enableScissor(this.leftPos + 120, this.topPos + 129, this.leftPos + 198, this.topPos + 165);
        for (int j = 0; j < infoComponents.size(); j++) {
            var cachedComponent = j > 3 ? Component.literal("...").getVisualOrderText() : infoComponents.get(j);
            guiGraphics.drawString(this.font, cachedComponent, (int) ((this.leftPos + 122) / 0.75f), (int) ((this.topPos + 129 + j * 7.5f) / 0.75f), 0x2C3141, false);
        }
        guiGraphics.disableScissor();

        pose.popPose();

        if (mouseX >= this.leftPos + 120 && mouseX <= this.leftPos + 200 && mouseY >= this.topPos + 117 && mouseY <= this.topPos + 175) {
            guiGraphics.renderTooltip(this.font, this.font.split(FormattedText.of(info.getString()), 200), mouseX, mouseY);
        }
    }

    @Nullable
    public VehicleAssemblingRecipe getCurrentRecipe() {
        if (this.currentRecipe == null) return null;
        return this.currentRecipe.value();
    }

    public List<IngredientArea> getIngredientAreas() {
        List<IngredientArea> areas = new ArrayList<>();
        if (this.currentRecipe != null) {
            var inputs = this.currentRecipe.value().getInputs();
            for (int i = 0; i < 3; i++) {
                for (int j = 0; j < 4; j++) {
                    int index = i * 4 + j;
                    if (index >= inputs.size()) return areas;
                    var input = inputs.get(index);
                    var ingredient = input.getIngredient();
                    var items = ingredient.getItems();
                    if (items.length == 0) continue;
                    int x = this.leftPos + 215 + j * 34;
                    int y = this.topPos + 118 + i * 14;
                    areas.add(new IngredientArea(ingredient, x, y, 12.8, 12.8));
                }
            }
        }
        return areas;
    }

    public record IngredientArea(Ingredient ingredient, double x, double y, double width, double height) {
        public boolean contains(double mouseX, double mouseY) {
            return mouseX >= x && mouseX <= x + width && mouseY >= y && mouseY <= y + height;
        }
    }
}
