package at.hannibal2.skyhanni.data.mob import at.hannibal2.skyhanni.utils.compat.formattedTextCompatLessResets

import at.hannibal2.skyhanni.SkyHanniMod
import at.hannibal2.skyhanni.api.event.HandleEvent
import at.hannibal2.skyhanni.data.IslandType
import at.hannibal2.skyhanni.data.mob.MobFilter.isDisplayNpc
import at.hannibal2.skyhanni.data.mob.MobFilter.isRealPlayer
import at.hannibal2.skyhanni.data.mob.MobFilter.isSkyBlockMob
import at.hannibal2.skyhanni.events.DebugDataCollectEvent
import at.hannibal2.skyhanni.events.MobEvent
import at.hannibal2.skyhanni.events.entity.EntityHealthUpdateEvent
import at.hannibal2.skyhanni.events.minecraft.ClientDisconnectEvent
import at.hannibal2.skyhanni.events.minecraft.packet.PacketReceivedEvent
import at.hannibal2.skyhanni.skyhannimodule.SkyHanniModule
import at.hannibal2.skyhanni.utils.ChatUtils
import at.hannibal2.skyhanni.utils.EntityUtils
import at.hannibal2.skyhanni.utils.LocationUtils
import at.hannibal2.skyhanni.utils.SkyBlockUtils
import at.hannibal2.skyhanni.utils.collection.CollectionUtils.drainForEach
import at.hannibal2.skyhanni.utils.collection.CollectionUtils.drainTo
import at.hannibal2.skyhanni.utils.collection.CollectionUtils.put
import at.hannibal2.skyhanni.utils.collection.CollectionUtils.refreshReference
import at.hannibal2.skyhanni.utils.compat.MinecraftCompat
import at.hannibal2.skyhanni.utils.getLorenzVec
import net.minecraft.client.network.ClientPlayerEntity
import net.minecraft.entity.LivingEntity
import net.minecraft.entity.decoration.ArmorStandEntity
import net.minecraft.entity.mob.CreeperEntity
import net.minecraft.entity.passive.BatEntity
import net.minecraft.entity.passive.VillagerEntity
import net.minecraft.entity.player.PlayerEntity
import net.minecraft.network.packet.s2c.play.GameJoinS2CPacket
import net.minecraft.network.packet.s2c.play.EntitySpawnS2CPacket
import net.minecraft.entity.damage.DamageSource
import net.minecraft.world.World
import java.util.concurrent.ConcurrentLinkedQueue
import java.util.concurrent.atomic.AtomicBoolean
//#if MC < 1.21
//$$ import net.minecraft.network.packet.s2c.play.PlayerSpawnS2CPacket
//$$ import net.minecraft.network.packet.s2c.play.MobSpawnS2CPacket
//$$
//#endif

@SkyHanniModule
object MobDetection {

    /* Unsupported Entities
        Nicked Players
        Odonata
        Silk Worm
        Fairy (in Dungeon)
        Totem of Corruption
        Worm
        Scatha
        Butterfly
        Exe
        Wai
        Zee
     */

    private const val MAX_RETRIES = 20 * 5

    private val forceReset get() = !SkyHanniMod.feature.dev.mobDebug.enable

    private val shouldClear: AtomicBoolean = AtomicBoolean(false)

    private fun mobDetectionReset() {
        MobData.currentMobs.map {
            it.createDeSpawnEvent()
        }.forEach { it.post() }
        MobData.retries.clear()
    }

    // TODO this is a unused debug function. maybe connect with a debug commmand or remove
    private fun watchdog() {
        val world = MinecraftCompat.localWorldOrNull ?: return
        if (MobData.retries.any { it.value.entity.world != world }) {
            ChatUtils.chat("Watchdog: Retires")
        }
        if (MobData.currentMobs.any { it.watchdogCheck(world) }) {
            ChatUtils.chat("Watchdog: Current Mobs")
        }
        if (MobData.players.any { it.watchdogCheck(world) }) {
            ChatUtils.chat("Watchdog: Players")
        }
        if (MobData.displayNpcs.any { it.watchdogCheck(world) }) {
            ChatUtils.chat("Watchdog: Display NPCs")
        }
        if (MobData.skyblockMobs.any { it.watchdogCheck(world) }) {
            ChatUtils.chat("Watchdog: SkyBlockMobs")
        }
        if (MobData.summoningMobs.any { it.watchdogCheck(world) }) {
            ChatUtils.chat("Watchdog: Summoning")
        }
        if (MobData.special.any { it.watchdogCheck(world) }) {
            ChatUtils.chat("Watchdog: Special")
        }
        if (MobData.notSeenMobs.any { it.watchdogCheck(world) }) {
            ChatUtils.chat("Watchdog: Not Seen Mobs")
        }
    }

    private fun Mob.watchdogCheck(world: World): Boolean =
        this.baseEntity.world != world || (
            this.armorStand?.let { it.world != world } ?: false
            ) || this.extraEntities.any { it.world != world }

    @HandleEvent
    fun onTick() {
        if (shouldClear.get()) { // Needs to work outside skyblock since it needs clearing when leaving skyblock and joining limbo
            mobDetectionReset()
            shouldClear.set(false)
        }
        @Suppress("InSkyBlockEarlyReturn")
        if (!SkyBlockUtils.inSkyBlock) return

        makeEntityReferenceUpdate()

        handleMobsFromPacket()

        handleRetries()

        MobData.previousEntityLiving.clear()
        MobData.previousEntityLiving.addAll(MobData.currentEntityLiving)
        MobData.currentEntityLiving.clear()
        MobData.currentEntityLiving.addAll(
            EntityUtils.getEntities<LivingEntity>()
                .filter { it !is ArmorStandEntity && it !is ClientPlayerEntity },
        )

        if (forceReset) {
            MobData.currentEntityLiving.clear() // Naturally removing the mobs using the despawn
        }

        (MobData.currentEntityLiving - MobData.previousEntityLiving).forEach { addRetry(it) } // Spawn
        (MobData.previousEntityLiving - MobData.currentEntityLiving).forEach { entityDeSpawn(it) } // Despawn

        MobData.notSeenMobs.removeIf(::canBeSeen)

        if (forceReset) {
            mobDetectionReset() // Ensure that all mobs are cleared 100%
        }
    }

    /** Splits the entity into player, displayNPC and other */
    private fun LivingEntity.getRoughType() = when {
        this is PlayerEntity && this.isRealPlayer() -> Mob.Type.PLAYER
        this.isDisplayNpc() -> Mob.Type.DISPLAY_NPC
        this.isSkyBlockMob() && !islandException() -> Mob.Type.BASIC
        else -> null
    }

    private fun addRetry(entity: LivingEntity) = entity.getRoughType()?.let { type ->
        val re = MobData.RetryEntityInstancing(entity, 0, type)
        MobData.retries.put(re.toKeyValuePair())
    }

    private fun removeRetry(entity: LivingEntity) = MobData.retries.remove(entity.id)

    private fun getRetry(entity: LivingEntity) = MobData.retries[entity.id]

    /** @return always true */
    private fun mobDetectionError(string: String) = MobData.logger.log(string).let { true }

    private fun canBeSeen(mob: Mob): Boolean {
        val isVisible = !mob.isInvisible() && mob.canBeSeen()
        if (isVisible) {
            when (mob.mobType) {
                Mob.Type.PLAYER -> MobEvent.FirstSeen.Player(mob)
                Mob.Type.SUMMON -> MobEvent.FirstSeen.Summon(mob)
                Mob.Type.SPECIAL -> MobEvent.FirstSeen.Special(mob)
                Mob.Type.PROJECTILE -> MobEvent.FirstSeen.Projectile(mob)
                Mob.Type.DISPLAY_NPC -> MobEvent.FirstSeen.DisplayNpc(mob)
                Mob.Type.BASIC, Mob.Type.DUNGEON, Mob.Type.BOSS, Mob.Type.SLAYER -> MobEvent.FirstSeen.SkyblockMob(mob)
            }.post()
        }
        return isVisible
    }

    /**@return a false means that it should try again (later)*/
    private fun entitySpawn(entity: LivingEntity, roughType: Mob.Type): Boolean {
        when (roughType) {
            Mob.Type.PLAYER -> MobEvent.Spawn.Player(MobFactories.player(entity)).post()

            Mob.Type.DISPLAY_NPC -> return MobFilter.createDisplayNpc(entity)
            Mob.Type.BASIC -> {
                val (result, mob) = MobFilter.createSkyblockEntity(entity)
                when (result) {
                    MobData.Result.NotYetFound -> return false
                    MobData.Result.Illegal -> return true // Remove entity from the spawning queue
                    MobData.Result.SomethingWentWrong -> return mobDetectionError("Something Went Wrong!")
                    MobData.Result.Found -> {
                        if (mob == null) return mobDetectionError("Mob is null even though result is Found")
                        when (mob.mobType) {
                            Mob.Type.SUMMON -> MobEvent.Spawn.Summon(mob)
                            Mob.Type.BASIC, Mob.Type.DUNGEON, Mob.Type.BOSS, Mob.Type.SLAYER -> MobEvent.Spawn.SkyblockMob(mob)
                            Mob.Type.SPECIAL -> MobEvent.Spawn.Special(mob)
                            Mob.Type.PROJECTILE -> MobEvent.Spawn.Projectile(mob)
                            Mob.Type.DISPLAY_NPC -> MobEvent.Spawn.DisplayNpc(mob) // Needed for some special cases
                            Mob.Type.PLAYER -> return mobDetectionError("An Player Ended Here. How?")
                        }.post()
                    }
                }
            }

            else -> return true
        }
        return true
    }

    private val entityFromPacket = ConcurrentLinkedQueue<Pair<EntityPacketType, Int>>()

    /** For mobs that have default health of the entity */
    private enum class EntityPacketType {
        SPIRIT_BAT,
        VILLAGER,
        CREEPER_VAIL,
    }

    /** Handles some mobs that have default health of the entity, specially using the [EntityHealthUpdateEvent] */
    private fun handleMobsFromPacket() = entityFromPacket.drainForEach { (type, id) ->
        when (type) {
            EntityPacketType.SPIRIT_BAT -> {
                val entity = EntityUtils.getEntityByID(id) as? BatEntity ?: return@drainForEach
                if (MobData.entityToMob[entity] != null) return@drainForEach
                removeRetry(entity)
                MobEvent.Spawn.Projectile(MobFactories.projectile(entity, "Spirit Scepter Bat")).post()
            }

            EntityPacketType.VILLAGER -> {
                val entity = EntityUtils.getEntityByID(id) as? VillagerEntity ?: return@drainForEach
                val mob = MobData.entityToMob[entity]
                if (mob != null && mob.mobType == Mob.Type.DISPLAY_NPC) {
                    MobEvent.DeSpawn.DisplayNpc(mob)
                    addRetry(entity)
                    return@drainForEach
                }
                getRetry(entity)?.let {
                    if (it.roughType == Mob.Type.DISPLAY_NPC) {
                        removeRetry(entity)
                        addRetry(entity)
                    }
                }
            }

            EntityPacketType.CREEPER_VAIL -> {
                val entity = EntityUtils.getEntityByID(id) as? CreeperEntity ?: return@drainForEach
                if (MobData.entityToMob[entity] != null) return@drainForEach
                if (!entity.isCharged) return@drainForEach
                removeRetry(entity)
                MobEvent.Spawn.Special(MobFactories.special(entity, "Creeper Veil")).post()
            }
        }
    }

    @HandleEvent
    fun onEntityHealthUpdateEvent(event: EntityHealthUpdateEvent) {
        when {
            event.entity is BatEntity && event.health == 6 -> {
                entityFromPacket.add(EntityPacketType.SPIRIT_BAT to event.entity.id)
            }

            event.entity is VillagerEntity && event.health != 20 -> {
                entityFromPacket.add(EntityPacketType.VILLAGER to event.entity.id)
            }

            event.entity is CreeperEntity && event.health == 20 -> {
                entityFromPacket.add(EntityPacketType.CREEPER_VAIL to event.entity.id)
            }
        }
    }

    private fun islandException(): Boolean = when (SkyBlockUtils.currentIsland) {
        IslandType.GARDEN_GUEST -> true
        IslandType.PRIVATE_ISLAND_GUEST -> true
        else -> false
    }

    private fun entityDeSpawn(entity: LivingEntity) {
        MobData.entityToMob[entity]?.createDeSpawnEvent()?.post() ?: removeRetry(entity)
        allEntitiesViaPacketId.remove(entity.id)
    }

    private fun Mob.createDeSpawnEvent() = when (this.mobType) {
        Mob.Type.PLAYER -> MobEvent.DeSpawn.Player(this)
        Mob.Type.SUMMON -> MobEvent.DeSpawn.Summon(this)
        Mob.Type.SPECIAL -> MobEvent.DeSpawn.Special(this)
        Mob.Type.PROJECTILE -> MobEvent.DeSpawn.Projectile(this)
        Mob.Type.DISPLAY_NPC -> MobEvent.DeSpawn.DisplayNpc(this)
        Mob.Type.BASIC, Mob.Type.DUNGEON, Mob.Type.BOSS, Mob.Type.SLAYER -> MobEvent.DeSpawn.SkyblockMob(this)
    }

    fun postMobHurtEvent(mob: Mob, source: DamageSource, amount: Float) = when (mob.mobType) {
        Mob.Type.PLAYER -> MobEvent.Hurt.Player(mob, source, amount)
        Mob.Type.SUMMON -> MobEvent.Hurt.Summon(mob, source, amount)
        Mob.Type.SPECIAL -> MobEvent.Hurt.Special(mob, source, amount)
        Mob.Type.PROJECTILE -> MobEvent.Hurt.Projectile(mob, source, amount)
        Mob.Type.DISPLAY_NPC -> MobEvent.Hurt.DisplayNpc(mob, source, amount)
        Mob.Type.BASIC, Mob.Type.DUNGEON, Mob.Type.BOSS, Mob.Type.SLAYER -> MobEvent.Hurt.SkyblockMob(mob, source, amount)
    }.post()

    private fun handleRetries() {
        val iterator = MobData.retries.iterator()
        while (iterator.hasNext()) {
            val (_, retry) = iterator.next()

            if (MobData.externRemoveOfRetryAmount > 0) {
                iterator.remove()
                MobData.externRemoveOfRetryAmount--
                continue
            }

            if (retry.outsideRange()) continue

            val entity = retry.entity
            if (retry.times == MAX_RETRIES) {
                MobData.logger.log(
                    "`${retry.entity.name.formattedTextCompatLessResets()}`${retry.entity.id} missed {\n " +
                        "is already Found: ${MobData.entityToMob[retry.entity] != null})." +
                        "\n Position: ${retry.entity.getLorenzVec()}\n " +
                        "DistanceC: ${
                            entity.getLorenzVec().distanceChebyshevIgnoreY(LocationUtils.playerLocation())
                        }\n" +
                        "Relative Position: ${entity.getLorenzVec() - LocationUtils.playerLocation()}\n " +
                        "}",
                )
                // Uncomment this to make it closed a loop
                // iterator.remove()
                // continue
            }
            if (!entitySpawn(entity, retry.roughType)) {
                retry.times++
                continue
            }
            iterator.remove()
        }
    }

    private val entityUpdatePackets = ConcurrentLinkedQueue<Int>()
    private val entitiesThatRequireUpdate = mutableSetOf<Int>() // needs to be distinct, therefore not using a queue

    /** Refreshes the references of the entities in entitiesThatRequireUpdate */
    private fun makeEntityReferenceUpdate() {
        entitiesThatRequireUpdate.iterator().let { iterator ->
            while (iterator.hasNext()) {
                if (handleEntityUpdate(iterator.next())) iterator.remove()
            }
        }
        entityUpdatePackets.drainTo(entitiesThatRequireUpdate)
    }

    private fun handleEntityUpdate(entityID: Int): Boolean {
        val entity = EntityUtils.getEntityByID(entityID) as? LivingEntity ?: return false
        getRetry(entity)?.entity = entity
        MobData.currentEntityLiving.refreshReference(entity)
        MobData.previousEntityLiving.refreshReference(entity)
        // update map
        MobData.entityToMob[entity]?.internalUpdateOfEntity(entity)
        return true
    }

    @HandleEvent
    fun onEntitySpawnPacket(event: PacketReceivedEvent) {
        when (val packet = event.packet) {
            is EntitySpawnS2CPacket -> addEntityUpdate(packet.entityId)
            //#if MC < 1.21
            //$$ is MobSpawnS2CPacket -> addEntityUpdate(packet.id)
            //$$ is EntitySpawnS2CPacket -> addEntityUpdate(packet.id)
            //#endif
            is GameJoinS2CPacket -> {
                // one of the first packets that is sent when switching servers inside the BungeeCord Network
                // (please some prove this, I just found it out via Testing)
                shouldClear.set(true)
                allEntitiesViaPacketId.clear()
            }
        }
    }

    private val allEntitiesViaPacketId = mutableSetOf<Int>()

    private fun addEntityUpdate(id: Int) = if (allEntitiesViaPacketId.contains(id)) {
        entityUpdatePackets.add(id)
    } else {
        allEntitiesViaPacketId.add(id)
    }

    @HandleEvent
    fun onDisconnect(event: ClientDisconnectEvent) {
        shouldClear.set(true)
    }

    @HandleEvent
    fun onDebug(event: DebugDataCollectEvent) {
        event.title("Mob Detection")
        if (forceReset) {
            event.addData("Mob Detection is manually disabled!")
        } else {
            event.addIrrelevant {
                add("normal enabled")
                add("Active Mobs: ${MobData.currentMobs.size}")
                val inDistanceMobs = MobData.retries.count { it.value.outsideRange() }
                add("Searching for Mobs: ${MobData.retries.size - inDistanceMobs}")
                add("Mobs over Max Search Count: ${MobData.retries.count { it.value.times > MAX_RETRIES }}")
                add("Mobs outside of Range: $inDistanceMobs")
            }
        }
    }

}
