package at.hannibal2.skyhanni.features.combat.damageindicator import at.hannibal2.skyhanni.utils.compat.deceased import at.hannibal2.skyhanni.utils.compat.formattedTextCompatLessResets import at.hannibal2.skyhanni.utils.compat.findHealthReal

import at.hannibal2.skyhanni.SkyHanniMod
import at.hannibal2.skyhanni.api.event.HandleEvent
import at.hannibal2.skyhanni.config.ConfigUpdaterMigrator
import at.hannibal2.skyhanni.config.features.combat.damageindicator.DamageIndicatorConfig.NameVisibility
import at.hannibal2.skyhanni.data.ScoreboardData
import at.hannibal2.skyhanni.data.SlayerApi
import at.hannibal2.skyhanni.events.BossHealthChangeEvent
import at.hannibal2.skyhanni.events.CheckRenderEntityEvent
import at.hannibal2.skyhanni.events.DamageIndicatorDeathEvent
import at.hannibal2.skyhanni.events.DamageIndicatorDetectedEvent
import at.hannibal2.skyhanni.events.DamageIndicatorFinalBossEvent
import at.hannibal2.skyhanni.events.DebugDataCollectEvent
import at.hannibal2.skyhanni.events.MobEvent
import at.hannibal2.skyhanni.events.chat.SkyHanniChatEvent
import at.hannibal2.skyhanni.events.entity.EntityEnterWorldEvent
import at.hannibal2.skyhanni.events.entity.EntityHealthUpdateEvent
import at.hannibal2.skyhanni.events.minecraft.ServerTickEvent
import at.hannibal2.skyhanni.events.minecraft.SkyHanniRenderWorldEvent
import at.hannibal2.skyhanni.events.minecraft.SkyHanniTickEvent
import at.hannibal2.skyhanni.features.combat.end.DragonFightAPI
import at.hannibal2.skyhanni.features.dungeon.DungeonApi
import at.hannibal2.skyhanni.features.rift.RiftApi
import at.hannibal2.skyhanni.features.rift.area.colosseum.BacteApi
import at.hannibal2.skyhanni.features.rift.area.colosseum.BacteApi.currentPhase
import at.hannibal2.skyhanni.features.slayer.blaze.HellionShield
import at.hannibal2.skyhanni.features.slayer.blaze.HellionShieldHelper.setHellionShield
import at.hannibal2.skyhanni.features.slayer.spider.SlayerSpiderFeatures
import at.hannibal2.skyhanni.skyhannimodule.SkyHanniModule
import at.hannibal2.skyhanni.test.command.CopyNearbyEntitiesCommand
import at.hannibal2.skyhanni.test.command.ErrorManager
import at.hannibal2.skyhanni.utils.EntityUtils.baseMaxHealth
import at.hannibal2.skyhanni.utils.EntityUtils.canBeSeen
import at.hannibal2.skyhanni.utils.EntityUtils.getNameTagWith
import at.hannibal2.skyhanni.utils.EntityUtils.hasNameTagWith
import at.hannibal2.skyhanni.utils.LocationUtils
import at.hannibal2.skyhanni.utils.LocationUtils.distanceToPlayer
import at.hannibal2.skyhanni.utils.LorenzColor
import at.hannibal2.skyhanni.utils.LorenzVec
import at.hannibal2.skyhanni.utils.MobUtils.mob
import at.hannibal2.skyhanni.utils.NumberUtil
import at.hannibal2.skyhanni.utils.NumberUtil.addSeparators
import at.hannibal2.skyhanni.utils.NumberUtil.formatPercentage
import at.hannibal2.skyhanni.utils.NumberUtil.roundTo
import at.hannibal2.skyhanni.utils.NumberUtil.shortFormat
import at.hannibal2.skyhanni.utils.PlayerUtils
import at.hannibal2.skyhanni.utils.RegexUtils.matchMatcher
import at.hannibal2.skyhanni.utils.SimpleTimeMark
import at.hannibal2.skyhanni.utils.SimpleTimeMark.Companion.fromNow
import at.hannibal2.skyhanni.utils.SkyBlockUtils
import at.hannibal2.skyhanni.utils.StringUtils.removeColor
import at.hannibal2.skyhanni.utils.TimeUtils.format
import at.hannibal2.skyhanni.utils.TimeUtils.ticks
import at.hannibal2.skyhanni.utils.collection.CollectionUtils.removeIf
import at.hannibal2.skyhanni.utils.collection.TimeLimitedCache
import at.hannibal2.skyhanni.utils.getLorenzVec
import at.hannibal2.skyhanni.utils.render.WorldRenderUtils.drawDynamicText
import com.google.gson.JsonArray
import net.minecraft.client.network.OtherClientPlayerEntity
import at.hannibal2.skyhanni.utils.render.ModernGlStateManager
import net.minecraft.entity.mob.MobEntity
import net.minecraft.entity.LivingEntity
import net.minecraft.entity.decoration.ArmorStandEntity
import net.minecraft.entity.mob.EndermanEntity
import net.minecraft.entity.mob.MagmaCubeEntity
import net.minecraft.entity.mob.ZombieEntity
import net.minecraft.entity.passive.WolfEntity
import java.util.UUID
import kotlin.math.max
import kotlin.time.Duration
import kotlin.time.Duration.Companion.milliseconds
import kotlin.time.Duration.Companion.seconds
import at.hannibal2.skyhanni.data.mob.Mob as ShMob

typealias EntityData = DamageIndicatorEntityData

// TODO cut class into smaller pieces
@SkyHanniModule
@Suppress("LargeClass")
object DamageIndicatorManager {

    private var mobFinder: MobFinder? = null
    private val maxHealth = mutableMapOf<UUID, Long>()
    private val config get() = SkyHanniMod.feature.combat.damageIndicator

    private val enderSlayerHitsNumberPattern = ".* §[5fd]§l(?<hits>\\d+) Hits?".toPattern()

    private val data = mutableMapOf<UUID, EntityData>()
    private val damagePattern = "[✧✯]?(\\d+[⚔+✧❤♞☄✷ﬗ✯]*)".toPattern()

    private val iconCache = TimeLimitedCache<EntityData, List<String>>(1.seconds)

    fun isDamageSplash(entity: ArmorStandEntity): Boolean {
        if (entity.age > 300) return false
        if (!entity.hasCustomName()) return false
        if (entity.deceased) return false
        val name = entity.customName.formattedTextCompatLessResets().removeColor().replace(",", "")

        return damagePattern.matcher(name).matches()
    }

    fun isBossSpawned(type: BossType) = data.entries.any { it.value.bossType == type }

    fun isBossSpawned(vararg types: BossType) = types.any { isBossSpawned(it) }

    fun getDistanceTo(vararg types: BossType): Double {
        val playerLocation = LocationUtils.playerLocation()
        return data.values.filter { it.bossType in types }.map { it.entity.getLorenzVec().distance(playerLocation) }.let { list ->
            if (list.isEmpty()) Double.MAX_VALUE else list.minOf { it }
        }
    }

    fun getAllMobs(): Collection<LivingEntity> = data.values.map { it.entity }

    fun getNearestDistanceTo(location: LorenzVec): Double {
        return data.values.map { it.entity.getLorenzVec() }.minOfOrNull { it.distance(location) } ?: Double.MAX_VALUE
    }

    fun removeDamageIndicator(type: BossType) {
        data.removeIf { it.value.bossType == type }
    }

    @HandleEvent(onlyOnSkyblock = true)
    fun onServerTick(event: ServerTickEvent) {
        data.forEach {
            it.value.serverTicksAlive++
        }
    }

    @HandleEvent
    fun onWorldChange() {
        mobFinder = MobFinder()
        data.clear()
    }

    @HandleEvent
    fun onChat(event: SkyHanniChatEvent) {
        mobFinder?.handleChat(event.message)
    }

    @HandleEvent
    fun onRenderWorld(event: SkyHanniRenderWorldEvent) {
        if (!isEnabled()) return

        if (SkyBlockUtils.debug) {
            list.removeIf { it.decayAt.isInPast() }
            for (highlight in list) {
                event.drawDynamicText(highlight.location, highlight.text, 1.5, seeThroughBlocks = false)
            }
        }

        ModernGlStateManager.disableDepthTest()
        ModernGlStateManager.disableCull()

        val sizeHealth: Double
        val sizeNameAbove: Double
        val sizeBossName: Double
        val sizeFinalResults: Double
        val smallestDistanceVew: Double
        if (PlayerUtils.isThirdPersonView()) {
            sizeHealth = 2.8
            sizeNameAbove = 2.2
            sizeBossName = 2.4
            sizeFinalResults = 1.8

            smallestDistanceVew = 10.0
        } else {
            sizeHealth = 1.9
            sizeNameAbove = 1.8
            sizeBossName = 2.1
            sizeFinalResults = 1.4

            smallestDistanceVew = 6.0
        }

        for (data in data.values) {
            val vecYOffset = when (data.bossType) {
                BossType.END_ENDSTONE_PROTECTOR -> 3.0
                BossType.SLAYER_SPIDER_5_1 -> 2.0

                else -> 1.0
            }

            if (!data.ignoreBlocks && !data.entity.canBeSeen(70.0, vecYOffset = vecYOffset)) continue
            val showNameAndHealth = data.shouldShowNameAndHealth()

            val entity = data.entity

            var healthText = data.healthText
            val delayedStart = data.delayedStart
            delayedStart?.let {
                if (!it.isInPast()) {
                    val delay = it.timeUntil()
                    healthText = formatDelay(delay)
                }
            }

            val location = if (data.dead && data.deathLocation != null) {
                data.deathLocation!!
            } else {
                val loc = entity.getLorenzVec()
                if (data.dead) data.deathLocation = loc
                loc
            }.add(-0.5, 0.0, -0.5)

            if (showNameAndHealth) {
                event.drawDynamicText(location, healthText, sizeHealth, smallestDistanceVew = smallestDistanceVew)
            }

            if (data.nameAbove.isNotEmpty()) {
                event.drawDynamicText(
                    location,
                    data.nameAbove,
                    sizeNameAbove,
                    -18f,
                    smallestDistanceVew = smallestDistanceVew,
                )
            }

            var bossName = if (showNameAndHealth) {
                when (config.bossName) {
                    NameVisibility.HIDDEN -> ""
                    NameVisibility.FULL_NAME -> data.bossType.fullName
                    NameVisibility.SHORT_NAME -> data.bossType.shortName
                }
            } else ""

            if (data.namePrefix.isNotEmpty()) {
                bossName = data.namePrefix + bossName
            }
            if (data.nameSuffix.isNotEmpty()) {
                bossName += data.nameSuffix
            }
            if (bossName.isNotEmpty()) {
                event.drawDynamicText(location, bossName, sizeBossName, -9f, smallestDistanceVew = smallestDistanceVew)
            }

            val icons = iconCache.getOrPut(data) {
                buildList {
                    if (config.shurikenIndicator && entity.getNameTagWith(3, "§b✯") != null) {
                        add(
                            if (config.compactStatusEffects) "§b✯"
                            else "§bShuriken",
                        )
                    }
                    if (config.twilightIndicator && entity.getNameTagWith(3, "§5ᛤ") != null) {
                        add(
                            if (config.compactStatusEffects) "§5ᛤ"
                            else "§5Twilight",
                        )
                    }
                }
            }

            val iconString = icons.joinToString(if (config.compactStatusEffects) "" else " ")
            var diff = 9f
            if (iconString.isNotEmpty()) {
                event.drawDynamicText(
                    location,
                    iconString,
                    sizeBossName,
                    diff,
                    smallestDistanceVew = smallestDistanceVew,
                )
                diff += 22f
            } else diff += 4f

            if (showNameAndHealth && config.showDamageOverTime) {
                val currentDamage = data.damageCounter.currentDamage
                val currentHealing = data.damageCounter.currentHealing
                if (currentDamage != 0L || currentHealing != 0L) {
                    val formatDamage = "§c" + currentDamage.shortFormat()
                    val formatHealing = "§a+" + currentHealing.shortFormat()
                    val finalResult = if (currentHealing == 0L) {
                        formatDamage
                    } else if (currentDamage == 0L) {
                        formatHealing
                    } else {
                        "$formatDamage §7/ $formatHealing"
                    }
                    event.drawDynamicText(
                        location,
                        finalResult,
                        sizeFinalResults,
                        diff,
                        smallestDistanceVew = smallestDistanceVew,
                    )
                    diff += 9f
                }
                for (damage in data.damageCounter.oldDamages) {
                    val formatDamage = "§c" + damage.damage.shortFormat() + "/s"
                    val formatHealing = "§a+" + damage.healing.shortFormat() + "/s"
                    val finalResult = if (damage.healing == 0L) {
                        formatDamage
                    } else if (damage.damage == 0L) {
                        formatHealing
                    } else {
                        "$formatDamage §7/ $formatHealing"
                    }
                    event.drawDynamicText(
                        location,
                        finalResult,
                        sizeFinalResults,
                        diff,
                        smallestDistanceVew = smallestDistanceVew,
                    )
                    diff += 9f
                }
            }
        }
        ModernGlStateManager.enableDepthTest()
        ModernGlStateManager.enableCull()
    }

    private fun EntityData.shouldShowNameAndHealth() = config.enabled && bossType.bossTypeToggle in config.bossesToShow

    @Suppress("Indentation")
    private fun noDeathDisplay(bossType: BossType): Boolean = when (bossType) {
        BossType.SLAYER_BLAZE_TYPHOEUS_1,
        BossType.SLAYER_BLAZE_TYPHOEUS_2,
        BossType.SLAYER_BLAZE_TYPHOEUS_3,
        BossType.SLAYER_BLAZE_TYPHOEUS_4,
        BossType.SLAYER_BLAZE_QUAZII_1,
        BossType.SLAYER_BLAZE_QUAZII_2,
        BossType.SLAYER_BLAZE_QUAZII_3,
        BossType.SLAYER_BLAZE_QUAZII_4,

        BossType.SLAYER_SPIDER_5_1,

            // TODO f3/m3 4 guardians, f2/m2 4 boss room fighters
        -> true

        else -> false
    }

    private fun tickDamage(damageCounter: DamageCounter) {
        val now = SimpleTimeMark.now()
        if (damageCounter.currentDamage != 0L || damageCounter.currentHealing != 0L) {
            if (damageCounter.firstTick.isFarPast()) {
                damageCounter.firstTick = now
            }

            if (damageCounter.firstTick.passedSince() > 1.seconds) {
                damageCounter.oldDamages.add(
                    0, OldDamage(now, damageCounter.currentDamage, damageCounter.currentHealing),
                )
                damageCounter.firstTick = SimpleTimeMark.farPast()
                damageCounter.currentDamage = 0
                damageCounter.currentHealing = 0
            }
        }
        damageCounter.oldDamages.removeIf { it.time.passedSince() > 5.seconds }
    }

    private fun formatDelay(delay: Duration): String {
        val color = when {
            delay < 1.seconds -> LorenzColor.DARK_PURPLE
            delay < 3.seconds -> LorenzColor.LIGHT_PURPLE

            else -> LorenzColor.WHITE
        }
        val format = delay.format(showMilliSeconds = true)
        return color.getChatColor() + format
    }

    val list = mutableListOf<Highlight>()

    class Highlight(val location: LorenzVec, val text: String, val decayAt: SimpleTimeMark)

    @HandleEvent
    fun onMobSpawn(event: MobEvent.Spawn) {
        val mob = event.mob

        if (SkyBlockUtils.debug) {
            list.add(Highlight(mob.baseEntity.getLorenzVec(), "${mob.name} - ${mob.mobType}", 5.seconds.fromNow()))
        }
        if (!isEnabled()) return
        try {
            val d = grabData(mob) ?: return
            data[d.entity.uuid] = d
            update(d)
        } catch (e: Throwable) {
            ErrorManager.logErrorWithData(
                e, "Error checking damage indicator entity",
                "mob" to mob,
                "mobInfo" to CopyNearbyEntitiesCommand.getMobInfo(mob),
            )
            return
        }
    }

    @HandleEvent
    fun onSkyHanniTick(event: SkyHanniTickEvent) {
        if (!isEnabled()) return
        data.values.forEach(::update)
        // TODO config to define between 100ms and 5 sec
        // TODO fix bug that the time is always 100 ms, even when the config is enabled?
        data.removeIf { (_, value) ->
            val noDeathDisplay = noDeathDisplay(value.bossType)
            val waitForRemoval = if (value.dead && !noDeathDisplay) 4.seconds else 100.milliseconds
            (SimpleTimeMark.now() > value.timeLastTick + waitForRemoval) || (value.dead && noDeathDisplay)
        }
    }

    private fun update(entityData: EntityData) {
        try {
            val entity = entityData.entity
            if (DungeonApi.inDungeon()) {
                checkFinalBoss(entityData.finalDungeonBoss, entity.id)
            }
            val health = entity.findHealthReal().toLong()
            val maxHealth: Long
            val biggestHealth = getMaxHealthFor(entity)
            if (biggestHealth == 0L) {
                val currentMaxHealth = entity.baseMaxHealth.toLong()
                maxHealth = max(currentMaxHealth, health)
                setMaxHealth(entity, maxHealth)
            } else {
                maxHealth = biggestHealth
            }

            entityData.namePrefix = ""
            entityData.nameSuffix = ""
            entityData.nameAbove = ""
            val customHealthText = if (health == 0L) {
                entityData.dead = true
                if (entityData.bossType.isSlayer && config.timeToKillSlayer) {
                    entityData.nameAbove = entityData.timeToKill
                }
                "§cDead"
            } else {
                getCustomHealth(entityData, health, entity, maxHealth) ?: return
            }

            data[entity.uuid]?.let {
                val lastHealth = it.lastHealth
                checkDamage(entityData, health, lastHealth)
                tickDamage(entityData.damageCounter)

                BossHealthChangeEvent(entityData, lastHealth, health, maxHealth).post()
            }
            entityData.lastHealth = health

            if (customHealthText.isNotEmpty()) {
                entityData.healthText = customHealthText
            } else {
                val color = NumberUtil.percentageColor(health, maxHealth)
                entityData.healthText = color.getChatColor() + health.shortFormat()
            }
            entityData.timeLastTick = SimpleTimeMark.now()
        } catch (e: Throwable) {
            ErrorManager.logErrorWithData(
                e, "Error checking damage indicator entity",
                "data" to data,
            )
            return
        }
    }

    @Suppress("ReturnCount")
    private fun getCustomHealth(
        entityData: EntityData,
        health: Long,
        entity: LivingEntity,
        maxHealth: Long,
    ): String? {

        when (entityData.bossType) {
            BossType.DUNGEON_F4_THORN -> {
                val thorn = checkThorn(health, maxHealth)
                if (thorn == null) {
                    val floor = DungeonApi.dungeonFloor
                    ErrorManager.logErrorStateWithData(
                        "Could not detect thorn",
                        "checkThorn returns null",
                        "health" to health,
                        "maxHealth" to maxHealth,
                        "floor" to floor,
                    )
                }
                return thorn
            }

            BossType.SLAYER_SPIDER_5_1 -> {
                entityData.nameAbove = if (entity.mob in SlayerSpiderFeatures.stuckTier5 && config.spiderSlayer.showInvincible) {
                    "§eKill hatchlings!"
                } else ""

                if (SlayerApi.config.spider.phaseDisplay) {
                    entityData.nameSuffix = " §e1/2"
                }
                return ""
            }

            BossType.SLAYER_SPIDER_5_2 -> {
                if (SlayerApi.config.spider.phaseDisplay) {
                    entityData.nameSuffix = " §e2/2"
                }
                return ""
            }

            BossType.SLAYER_ENDERMAN_1,
            BossType.SLAYER_ENDERMAN_2,
            BossType.SLAYER_ENDERMAN_3,
            BossType.SLAYER_ENDERMAN_4,
            -> return checkEnderSlayer(entity as EndermanEntity, entityData, health.toInt(), maxHealth.toInt())

            BossType.SLAYER_BLOODFIEND_1,
            BossType.SLAYER_BLOODFIEND_2,
            BossType.SLAYER_BLOODFIEND_3,
            BossType.SLAYER_BLOODFIEND_4,
            -> return checkVampireSlayer(entity as OtherClientPlayerEntity, entityData, health.toInt(), maxHealth.toInt())

            BossType.SLAYER_BLAZE_1,
            BossType.SLAYER_BLAZE_2,
            BossType.SLAYER_BLAZE_3,
            BossType.SLAYER_BLAZE_4,
            BossType.SLAYER_BLAZE_QUAZII_2,
            BossType.SLAYER_BLAZE_QUAZII_3,
            BossType.SLAYER_BLAZE_QUAZII_4,
            BossType.SLAYER_BLAZE_TYPHOEUS_2,
            BossType.SLAYER_BLAZE_TYPHOEUS_3,
            BossType.SLAYER_BLAZE_TYPHOEUS_4,
            -> return checkBlazeSlayer(entity as MobEntity, entityData, health.toInt(), maxHealth.toInt())

            BossType.NETHER_MAGMA_BOSS -> return checkMagmaCube(
                entity as MagmaCubeEntity,
                entityData,
                health.toInt(),
                maxHealth.toInt(),
            )

            BossType.SLAYER_ZOMBIE_5 -> {
                if ((entity as ZombieEntity).hasNameTagWith(3, "§fBoom!")) {
                    // TODO fix
//                    val ticksAlive = entity.ticksExisted % (20 * 5)
//                    val remainingTicks = (5 * 20).toLong() - ticksAlive
//                    val format = formatDelay(remainingTicks * 50)
//                    entityData.nameSuffix = " §f§lBOOM - $format"
                    if (SlayerApi.config.zombie.boomDisplay) {
                        entityData.nameSuffix = " §f§lBOOM!"
                    }
                }
            }

            BossType.SLAYER_WOLF_3,
            BossType.SLAYER_WOLF_4,
            -> {
                if ((entity as WolfEntity).hasNameTagWith(2, "§bCalling the pups!")) {
                    return "Pups!"
                }
            }

            BossType.NETHER_BARBARIAN_DUKE -> {
                val location = entity.getLorenzVec()
                entityData.ignoreBlocks = location.y == 117.0 && location.distanceToPlayer() < 15
            }

            BossType.BACTE -> return checkBacte(entityData)


            BossType.END_ENDER_DRAGON -> return checkEnderDragon(entityData)


            else -> return ""
        }
        return ""
    }

    private fun checkEnderDragon(entityData: EntityData): String {
        DragonFightAPI.currentType?.let {
            entityData.namePrefix = "§c§l$it "
        }
        return DragonFightAPI.currentHp?.let {
            "§c" + it.shortFormat()
        }.orEmpty()
    }

    private fun checkBacte(entityData: EntityData): String {
        if (!config.showBactePhase) return ""
        if (currentPhase == BacteApi.Phase.NOT_ACTIVE) return ""
        if (RiftApi.config.area.colosseum.bactePhaseDisplay) {
            entityData.namePrefix = "§c${currentPhase.ordinal}/${BacteApi.Phase.PHASE_5.ordinal} "
        }
        return ""
    }

    private fun checkBlazeSlayer(entity: MobEntity, entityData: EntityData, health: Int, maxHealth: Int): String {
        var found = false
        for (shield in HellionShield.entries) {
            entity.getNameTagWith(3, shield.name)?.let { armorStand ->
                val number = armorStand.name.formattedTextCompatLessResets().split(" ♨")[1].substring(0, 1)
                entity.setHellionShield(shield)
                if (SlayerApi.config.blazes.hellion.coloredMobs) {
                    entityData.nameAbove = shield.formattedName + " $number"
                }
                found = true
                break
            }
        }
        if (!found) {
            entity.setHellionShield(null)
        }

        if (!SlayerApi.config.blazes.phaseDisplay) return ""

        var calcHealth = health
        val calcMaxHealth: Int
        entityData.namePrefix = when (entityData.bossType) {
            BossType.SLAYER_BLAZE_1,
            BossType.SLAYER_BLAZE_2,
            -> {
                val step = maxHealth / 2
                calcMaxHealth = step
                if (health > step) {
                    calcHealth -= step
                    "§c1/2 "
                } else {
                    calcHealth = health
                    "§a2/2 "
                }
            }

            BossType.SLAYER_BLAZE_3,
            BossType.SLAYER_BLAZE_4,
            -> {
                val step = maxHealth / 3
                calcMaxHealth = step
                if (health > step * 2) {
                    calcHealth -= step * 2
                    "§c1/3 "
                } else if (health > step) {
                    calcHealth -= step
                    "§e2/3 "
                } else {
                    calcHealth = health
                    "§a3/3 "
                }
            }

            else -> return ""
        }

        return NumberUtil.percentageColor(
            calcHealth.toLong(), calcMaxHealth.toLong(),
        ).getChatColor() + calcHealth.shortFormat()
    }

    private fun checkMagmaCube(
        entity: MagmaCubeEntity,
        entityData: EntityData,
        health: Int,
        maxHealth: Int,
    ): String? {
        val slimeSize = entity.size
        val crimsonIsleConfig = SkyHanniMod.feature.crimsonIsle
        val prefix = when (slimeSize) {
            24 -> "§c1/6"
            22 -> "§e2/6"
            20 -> "§e3/6"
            18 -> "§e4/6"
            16 -> "§e5/6"
            else -> {
                val color = NumberUtil.percentageColor(health.toLong(), 10_000_000)
                if (crimsonIsleConfig.magmaBossDisplay) {
                    entityData.namePrefix = "§a6/6"
                }
                return color.getChatColor() + health.shortFormat()
            }
        }

        if (crimsonIsleConfig.magmaBossDisplay) {
            entityData.namePrefix = "$prefix §f"
        }

        // hide while in the middle
//        val position = entity.getLorenzVec()
        // TODO other logic or something
//        entityData.healthLineHidden = position.x == -368.0 && position.z == -804.0

        var calcHealth = -1
        for (line in ScoreboardData.sidebarLinesRaw) {
            if (line.contains("▎")) {
                val color: String
                if (line.startsWith("§7")) {
                    color = "§7"
                } else if (line.startsWith("§e")) {
                    color = "§e"
                } else if (line.startsWith("§6") || line.startsWith("§a") || line.startsWith("§c")) {
                    calcHealth = 0
                    break
                } else {
                    ErrorManager.logErrorStateWithData(
                        "Unknown magma boss health sidebar format",
                        "Damage Indicator could not find magma boss bar data",
                        "line" to line,
                        "ScoreboardData.sidebarLinesRaw" to ScoreboardData.sidebarLinesRaw,
                        "calcHealth" to calcHealth,
                        "slimeSize" to slimeSize,
                        "entity" to entity,
                        "entityData" to entityData,
                    )
                    break
                }

                val text = line.replace("\uD83C\uDF81" + color, "")
                val max = 25.0
                val length = text.split("§e", "§7")[1].length
                val missing = (health.toDouble() / max) * length
                calcHealth = (health - missing).toInt()
            }
        }
        if (calcHealth == -1) return null

        val color = NumberUtil.percentageColor(calcHealth.toLong(), maxHealth.toLong())
        return color.getChatColor() + calcHealth.shortFormat()
    }

    private fun checkEnderSlayer(
        entity: EndermanEntity,
        entityData: EntityData,
        health: Int,
        maxHealth: Int,
    ): String? {
        var calcHealth = health
        val calcMaxHealth: Int
        entityData.namePrefix = when (entityData.bossType) {
            BossType.SLAYER_ENDERMAN_1,
            BossType.SLAYER_ENDERMAN_2,
            BossType.SLAYER_ENDERMAN_3,
            -> {
                val step = maxHealth / 3
                calcMaxHealth = step
                if (health > step * 2) {
                    calcHealth -= step * 2
                    "§c1/3 "
                } else if (health > step) {
                    calcHealth -= step
                    "§e2/3 "
                } else {
                    calcHealth = health
                    "§a3/3 "
                }
            }

            BossType.SLAYER_ENDERMAN_4 -> {
                val step = maxHealth / 6
                calcMaxHealth = step
                if (health > step * 5) {
                    calcHealth -= step * 5
                    "§c1/6 "
                } else if (health > step * 4) {
                    calcHealth -= step * 4
                    "§e2/6 "
                } else if (health > step * 3) {
                    calcHealth -= step * 3
                    "§e3/6 "
                } else if (health > step * 2) {
                    calcHealth -= step * 2
                    "§e4/6 "
                } else if (health > step) {
                    calcHealth -= step
                    "§e5/6 "
                } else {
                    calcHealth = health
                    "§a6/6 "
                }
            }

            else -> return null
        }
        var result = NumberUtil.percentageColor(
            calcHealth.toLong(), calcMaxHealth.toLong(),
        ).getChatColor() + calcHealth.shortFormat()

        if (!SlayerApi.config.endermen.phaseDisplay) {
            result = ""
            entityData.namePrefix = ""
        }

        // Hit phase
        var hitPhaseText: String? = null
        val armorStandHits = entity.getNameTagWith(3, " Hit")
        if (armorStandHits != null) {
            val maxHits = when (entityData.bossType) {
                BossType.SLAYER_ENDERMAN_1 -> 15
                BossType.SLAYER_ENDERMAN_2 -> 30
                BossType.SLAYER_ENDERMAN_3 -> 60
                BossType.SLAYER_ENDERMAN_4 -> 100
                else -> 100
            }
            val hits = enderSlayerHitsNumberPattern.matchMatcher(armorStandHits.name.formattedTextCompatLessResets()) {
                group("hits").toInt()
            } ?: error("No hits number found in ender slayer name '${armorStandHits.name.formattedTextCompatLessResets()}'")

            hitPhaseText = NumberUtil.percentageColor(hits.toLong(), maxHits.toLong()).getChatColor() + "$hits Hits"
        }

        val ridingEntity = entity.vehicle
        // Laser phase
        if (config.enderSlayer.laserPhaseTimer && ridingEntity != null) {
            val totalTimeAlive = 8.2.seconds

            val ticksAlive = ridingEntity.age.ticks
            val remainingTime = totalTimeAlive - ticksAlive
            val formatDelay = formatDelay(remainingTime)
            if (config.enderSlayer.showHealthDuringLaser || hitPhaseText != null) {
                entityData.nameSuffix = " §f$formatDelay"
            } else {
                return formatDelay
            }
        }
        hitPhaseText?.let {
            return it
        }

        return result
    }

    private fun checkVampireSlayer(
        entity: OtherClientPlayerEntity,
        entityData: EntityData,
        health: Int,
        maxHealth: Int,
    ): String {
        val config = config.vampireSlayer

        if (config.percentage) {
            val percentage = (health.toDouble() / maxHealth).formatPercentage()
            entityData.nameSuffix = " §e$percentage"
        }

        if (config.maniaCircles) {
            entity.vehicle?.let {
                val existed = it.age
                if (existed > 40) {
                    val end = (20 * 26) - existed
                    val time = end.toDouble() / 20
                    entityData.nameAbove = "Mania Circles: §b${time.roundTo(1)}s"
                    return ""
                }
            }
        }

        if (config.hpTillSteak) {
            val rest = maxHealth * 0.2
            val showHealth = health - rest
            if (showHealth < 300) {
                entityData.nameAbove = if (showHealth > 0) {
                    "§cHP till Steak: ${showHealth.addSeparators()}"
                } else "§cSteak!"
            }
        }

        return ""
    }

    @Suppress("CyclomaticComplexMethod", "ReturnCount")
    private fun checkThorn(realHealth: Long, realMaxHealth: Long): String? {
        val maxHealth: Int
        val health = if (DungeonApi.isOneOf("F4")) {
            maxHealth = 4

            if (realMaxHealth == 300_000L) {
                // no derpy
                when {
                    realHealth == 1L -> 0
                    realHealth <= 66_000 -> 1
                    realHealth <= 144_000 -> 2
                    realHealth <= 222_000 -> 3
                    realHealth <= 300_000 -> 4

                    else -> return null
                }
            } else {
                // derpy
                when {
                    realHealth == 1L -> 0
                    realHealth <= 132_000 -> 1
                    realHealth <= 288_000 -> 2
                    realHealth <= 444_000 -> 3
                    realHealth <= 600_000 -> 4

                    else -> return null
                }
            }
        } else if (DungeonApi.isOneOf("M4")) {
            maxHealth = 6

            if (realMaxHealth == 900_000L) {
                // no derpy
                when {
                    realHealth == 1L -> 0
                    realHealth <= 135_000 -> 1
                    realHealth <= 288_000 -> 2
                    realHealth <= 441_000 -> 3
                    realHealth <= 594_000 -> 4
                    realHealth <= 747_000 -> 5
                    realHealth <= 900_000L -> 6

                    else -> return null
                }
            } else {
                // derpy
                when {
                    realHealth == 1L -> 0
                    realHealth <= 270_000 -> 1
                    realHealth <= 576_000 -> 2
                    realHealth <= 882_000 -> 3
                    realHealth <= 1_188_000 -> 4
                    realHealth <= 1_494_000 -> 5
                    realHealth <= 1_800_000 -> 6

                    else -> return null
                }
            }
        } else {
            ErrorManager.logErrorStateWithData(
                "Thorn in wrong floor detected",
                "Invalid floor for thorn",
                "dungeonFloor" to DungeonApi.dungeonFloor,
            )
            return null
        }
        val color = NumberUtil.percentageColor(health.toLong(), maxHealth.toLong())
        return color.getChatColor() + health + "/" + maxHealth
    }

    private fun checkDamage(entityData: EntityData, health: Long, lastHealth: Long) {
        val damage = lastHealth - health
        val healing = health - lastHealth
        if (damage > 0 && entityData.bossType != BossType.DUMMY) {
            val damageCounter = entityData.damageCounter
            damageCounter.currentDamage += damage
        }
        if (healing > 0) {
            // Hide auto heal every 10 ticks (with rounding errors)
            if ((healing == 15_000L || healing == 15_001L) && entityData.bossType == BossType.SLAYER_ZOMBIE_5) return

            val damageCounter = entityData.damageCounter
            damageCounter.currentHealing += healing
        }
    }

    private fun grabData(mob: ShMob): EntityData? {
        val entity = mob.baseEntity
        if (data.contains(entity.uuid)) return data[entity.uuid]

        val entityResult = mobFinder?.tryAdd(mob) ?: return null

        val entityData = EntityData(
            mob,
            entityResult.ignoreBlocks,
            entityResult.delayedStart,
            entityResult.finalDungeonBoss,
            entityResult.bossType,
            foundTime = SimpleTimeMark.now(),
        )
        DamageIndicatorDetectedEvent(entityData).post()
        return entityData
    }

    private fun checkFinalBoss(finalBoss: Boolean, id: Int) {
        if (finalBoss) {
            DamageIndicatorFinalBossEvent(id).post()
        }
    }

    private fun setMaxHealth(entity: LivingEntity, currentMaxHealth: Long) {
        maxHealth[entity.uuid!!] = currentMaxHealth
    }

    private fun getMaxHealthFor(entity: LivingEntity): Long {
        return maxHealth.getOrDefault(entity.uuid!!, 0L)
    }

    @HandleEvent
    fun onEntityJoin(event: EntityEnterWorldEvent<*>) {
        mobFinder?.handleNewEntity(event.entity)
    }

    private val dummyDamageCache = mutableListOf<UUID>()

    @HandleEvent(priority = HandleEvent.HIGH)
    fun onCheckRender(event: CheckRenderEntityEvent<ArmorStandEntity>) {
        if (!isEnabled()) return
        val entity = event.entity

        val entityData = data.values.find {
            val distance = it.entity.getLorenzVec().distance(entity.getLorenzVec())
            distance < 4.5
        } ?: return

        val showNameAndHealth = entityData.shouldShowNameAndHealth()
        if (isDamageSplash(entity)) {
            val name = entity.customName.formattedTextCompatLessResets().removeColor().replace(",", "")

            if (showNameAndHealth && config.hideDamageSplash) {
                event.cancel()
            }
            if (entityData.bossType == BossType.DUMMY) {
                val uuid = entity.uuid
                if (dummyDamageCache.contains(uuid)) return
                dummyDamageCache.add(uuid)
                val dmg = name.toCharArray().filter { Character.isDigit(it) }.joinToString("").toLong()
                entityData.damageCounter.currentDamage += dmg
            }
        } else {
            if (showNameAndHealth && config.hideVanillaNametag) {
                val name = entity.name.formattedTextCompatLessResets()
                if (name.contains("Plasmaflux")) return
                if (name.contains("Overflux")) return
                if (name.contains("Mana Flux")) return
                if (name.contains("Radiant")) return
                if (!entity.hasCustomName()) return
                event.cancel()
            }
        }
    }

    @HandleEvent
    fun onEntityHealthUpdate(event: EntityHealthUpdateEvent) {
        val data = data[event.entity.uuid] ?: return
        if (event.health <= 1) {
            if (!data.firstDeath) {
                data.firstDeath = true
                DamageIndicatorDeathEvent(event.entity, data).post()
            }
        }
    }

    @HandleEvent
    fun onConfigFix(event: ConfigUpdaterMigrator.ConfigFixEvent) {
        event.move(2, "damageIndicator", "combat.damageIndicator")
        event.move(3, "slayer.endermanPhaseDisplay", "slayer.endermen.phaseDisplay")
        event.move(3, "slayer.blazePhaseDisplay", "slayer.blazes.phaseDisplay")

        event.transform(23, "combat.damageIndicator.bossesToShow") { element ->
            val result = JsonArray()
            for (bossType in element as JsonArray) {
                if (bossType.asString == "DUNGEON_ALL") continue
                result.add(bossType)
            }

            result
        }
    }

    private val backendEnabled get() = SkyHanniMod.feature.dev.damageIndicatorBackend

    @HandleEvent
    fun onDebug(event: DebugDataCollectEvent) {
        event.title("Damage Indicator")
        if (!backendEnabled) {
            event.addData("Damage Indicator is manually disabled!")
        } else {
            event.addIrrelevant {
                add("normal enabled")
                add("Active mobs: ${data.size}")
                for (entityData in data.values) {
                    val type = entityData.bossType
                    val loc = entityData.entity.getLorenzVec()
                    add("  - $type ${loc.printWithAccuracy(1)}")
                }
            }
        }
    }

    private fun isEnabled() = SkyBlockUtils.inSkyBlock && backendEnabled
}
