package at.hannibal2.skyhanni.features.event.diana import at.hannibal2.skyhanni.utils.compat.deceased

import at.hannibal2.skyhanni.SkyHanniMod
import at.hannibal2.skyhanni.api.event.HandleEvent
import at.hannibal2.skyhanni.data.IslandType
import at.hannibal2.skyhanni.data.title.TitleManager
import at.hannibal2.skyhanni.events.SecondPassedEvent
import at.hannibal2.skyhanni.events.chat.SkyHanniChatEvent
import at.hannibal2.skyhanni.events.diana.InquisitorFoundEvent
import at.hannibal2.skyhanni.events.entity.EntityHealthUpdateEvent
import at.hannibal2.skyhanni.events.minecraft.KeyPressEvent
import at.hannibal2.skyhanni.skyhannimodule.SkyHanniModule
import at.hannibal2.skyhanni.utils.ChatUtils
import at.hannibal2.skyhanni.utils.EntityUtils
import at.hannibal2.skyhanni.utils.HypixelCommands
import at.hannibal2.skyhanni.utils.KeyboardManager
import at.hannibal2.skyhanni.utils.LorenzVec
import at.hannibal2.skyhanni.utils.PlayerUtils
import at.hannibal2.skyhanni.utils.RegexUtils.hasGroup
import at.hannibal2.skyhanni.utils.RegexUtils.matchMatcher
import at.hannibal2.skyhanni.utils.RegexUtils.matchMatchers
import at.hannibal2.skyhanni.utils.RegexUtils.matches
import at.hannibal2.skyhanni.utils.SimpleTimeMark
import at.hannibal2.skyhanni.utils.SoundUtils
import at.hannibal2.skyhanni.utils.SoundUtils.playSound
import at.hannibal2.skyhanni.utils.StringUtils.cleanPlayerName
import at.hannibal2.skyhanni.utils.collection.CollectionUtils.removeIf
import at.hannibal2.skyhanni.utils.getLorenzVec
import at.hannibal2.skyhanni.utils.repopatterns.RepoPattern
import net.minecraft.client.MinecraftClient
import net.minecraft.client.network.OtherClientPlayerEntity
import java.util.concurrent.ConcurrentHashMap
import java.util.regex.Matcher
import kotlin.time.Duration.Companion.seconds

@SkyHanniModule
object InquisitorWaypointShare {

    private val config get() = SkyHanniMod.feature.event.diana.inquisitorSharing

    private val patternGroup = RepoPattern.group("diana.waypoints.inquisitor")

    /**
     * REGEX-TEST: §9Party §8> User Name§f: §rx: 2.3, y: 4.5, z: -6.7
     * REGEX-TEST: §9Party §8> UserName§f: §rA MINOS INQUISITOR has spawned near [Foraging Island ] at Coords 1 2 -3
     * REGEX-TEST: §9Party §8> §b[MVP§9+§b] _088§f: §rx: 86, y: 73, z: -29 I dug up an inquisitor come over here!
     * REGEX-TEST: §9Party §8> §6[MVP§0++§6] scaryron§f: §rx: -67, y: 75, z: 116 | Minos Inquisitor spawned at [ ⏣ Mountain ]!
     */
    @Suppress("MaxLineLength")
    private val inquisitorCoordsPattern by patternGroup.list(
        "coords",
        "(?<party>§9Party §8> )?(?<playerName>.+)§f: §rx: (?<x>[^ ,]+),? y: (?<y>[^ ,]+),? z: (?<z>[^ ,]+).*",
        "(?<party>§9Party §8> )?(?<playerName>.+)§f: §rA MINOS INQUISITOR has spawned near \\[(?<area>.*)] at Coords (?<x>[^ ]+) (?<y>[^ ]+) (?<z>[^ ]+)",
    )

    /**
     * REGEX-TEST: §9Party §8> User Name§f: §rInquisitor dead!
     */
    private val diedPattern by patternGroup.pattern(
        "died",
        "(?<party>§9Party §8> )?(?<playerName>.*)§f: §rInquisitor dead!",
    )

    /**
     * REGEX-TEST: §c§lUh oh! §r§eYou dug out a §r§2Minos Inquisitor§r§e!
     */
    private val inquisitorFoundChatPattern by patternGroup.pattern(
        "dug",
        ".* §r§eYou dug out a §r§2Minos Inquisitor§r§e!",
    )

    private var inquisitor = -1
    private var lastInquisitor = -1
    private var lastShareTime = SimpleTimeMark.farPast()

    private val inquisitorsNearby = ConcurrentHashMap<Int, OtherClientPlayerEntity>()

    private val _waypoints = ConcurrentHashMap<String, SharedInquisitor>()
    val waypoints: Map<String, SharedInquisitor>
        get() = _waypoints

    class SharedInquisitor(
        val fromPlayer: String,
        val displayName: String,
        val location: LorenzVec,
        val spawnTime: SimpleTimeMark,
    )

    @HandleEvent
    fun onSecondPassed(event: SecondPassedEvent) {
        if (!isEnabled()) return

        if (event.repeatSeconds(3)) {
            inquisitorsNearby.removeIf { it.value.deceased }
        }

        _waypoints.removeIf { it.value.spawnTime.passedSince() > 75.seconds }
    }

    @HandleEvent
    fun onWorldChange() {
        _waypoints.clear()
        inquisitorsNearby.clear()
    }

    private val inquisitorTime = mutableListOf<SimpleTimeMark>()

    @HandleEvent
    fun onInquisitorFound(event: InquisitorFoundEvent) {
        val inquisitor = event.inquisitorEntity
        inquisitorsNearby[inquisitor.id] = inquisitor
        GriffinBurrowHelper.update()

        lastInquisitor = inquisitor.id
        checkInquisFound()
    }

    // We do not know if the chat message or the entity spawn happens first.
    // We only want to run foundInquisitor when both happens in under 1.5 seconds
    private fun checkInquisFound() {
        inquisitorTime.add(SimpleTimeMark.now())

        val lastTwo = inquisitorTime.takeLast(2)
        if (lastTwo.size != 2) return

        if (lastTwo.all { it.passedSince() < 1.5.seconds }) {
            inquisitorTime.clear()
            foundInquisitor(lastInquisitor)
        }
    }

    @HandleEvent(onlyOnIsland = IslandType.HUB, receiveCancelled = true)
    fun onChat(event: SkyHanniChatEvent) {
        if (!isEnabled()) return
        val message = event.message

        if (inquisitorFoundChatPattern.matches(message)) {
            checkInquisFound()
        }

        inquisitorCoordsPattern.matchMatchers(message) {
            if (!detectFromChat()) return@matchMatchers
            event.blockedReason = "inquisitor_waypoint"
        }

        diedPattern.matchMatcher(message) {
            if (block()) return
            val rawName = group("playerName")
            val name = rawName.cleanPlayerName()
            _waypoints.remove(name)
            GriffinBurrowHelper.update()
        }
    }

    private fun foundInquisitor(inquisId: Int) {
        lastShareTime = SimpleTimeMark.farPast()
        inquisitor = inquisId

        if (config.instantShare) {
            // add repo kill switch
            sendInquisitor()
        } else {
            val keyName = KeyboardManager.getKeyName(config.keyBindShare)
            val message = "§l§bYou found an Inquisitor! Click §l§chere §l§bor press §c$keyName to share the location!"
            ChatUtils.clickableChat(
                message,
                onClick = {
                    sendInquisitor()
                },
                "§eClick to share!",
                oneTimeClick = true,
            )
        }
    }

    @HandleEvent
    fun onEntityHealthUpdate(event: EntityHealthUpdateEvent) {
        if (!isEnabled()) return
        if (event.health > 0) return

        val entityId = event.entity.id
        if (entityId == inquisitor) {
            sendDeath()
        }
        inquisitorsNearby.remove(entityId)
    }

    @HandleEvent
    fun onKeyPress(event: KeyPressEvent) {
        if (!isEnabled()) return
        if (MinecraftClient.getInstance().currentScreen != null) return
        if (event.keyCode == config.keyBindShare) sendInquisitor()
    }

    private fun sendDeath() {
        if (!isEnabled()) return
        if (lastShareTime.passedSince() < 5.seconds) return

        // already dead
        if (inquisitor == -1) return
        inquisitor = -1
        HypixelCommands.partyChat("Inquisitor dead!")
    }

    private fun sendInquisitor() {
        if (!isEnabled()) return
        if (lastShareTime.passedSince() < 5.seconds) return
        lastShareTime = SimpleTimeMark.now()

        if (inquisitor == -1) {
            ChatUtils.debug("Trying to send inquisitor via chat, but no Inquisitor is nearby.")
            return
        }

        val inquisitor = EntityUtils.getEntityByID(inquisitor)
        if (inquisitor == null) {
            ChatUtils.chat("§cInquisitor out of range!")
            return
        }

        if (inquisitor.deceased) {
            ChatUtils.chat("§cInquisitor is dead")
            return
        }
        val location = inquisitor.getLorenzVec()
        val x = location.x.toInt()
        val y = location.y.toInt()
        val z = location.z.toInt()
        HypixelCommands.partyChat("x: $x, y: $y, z: $z ")
    }

    private fun Matcher.block(): Boolean = !hasGroup("party") && !config.globalChat

    private fun Matcher.detectFromChat(): Boolean {
        if (block()) return false
        val rawName = group("playerName")
        val x = group("x").trim().toDoubleOrNull() ?: return false
        val y = group("y").trim().toDoubleOrNull() ?: return false
        val z = group("z").trim().toDoubleOrNull() ?: return false
        val location = LorenzVec(x, y, z)

        val name = rawName.cleanPlayerName()
        val displayName = rawName.cleanPlayerName(displayName = true)
        if (!waypoints.containsKey(name)) {
            ChatUtils.chat("$displayName §l§efound an inquisitor at §l§c${x.toInt()} ${y.toInt()} ${z.toInt()}!")
            if (name != PlayerUtils.getName()) {
                TitleManager.sendTitle("§dINQUISITOR §efrom §b$displayName")
                playUserSound()
            }
        }
        val inquis = SharedInquisitor(name, displayName, location, SimpleTimeMark.now())
        _waypoints[name] = inquis
        GriffinBurrowHelper.update()
        return true
    }

    private fun isEnabled() = DianaApi.isDoingDiana() && config.enabled

    fun maybeRemove(inquis: SharedInquisitor) {
        if (inquisitorsNearby.isEmpty()) {
            _waypoints.remove(inquis.fromPlayer)
            GriffinBurrowHelper.update()
            ChatUtils.chat("Inquisitor from ${inquis.displayName} §enot found, deleting.")
        }
    }

    @JvmStatic
    fun playUserSound() {
        with(config.sound) {
            SoundUtils.createSound(name, pitch).playSound()
        }
    }
}
