package at.hannibal2.skyhanni.features.rift.everywhere

import at.hannibal2.skyhanni.api.event.HandleEvent
import at.hannibal2.skyhanni.events.ConfigLoadEvent
import at.hannibal2.skyhanni.events.GuiRenderEvent
import at.hannibal2.skyhanni.events.SecondPassedEvent
import at.hannibal2.skyhanni.features.rift.RiftApi
import at.hannibal2.skyhanni.skyhannimodule.SkyHanniModule
import at.hannibal2.skyhanni.utils.ConditionalUtils
import at.hannibal2.skyhanni.utils.InventoryUtils
import at.hannibal2.skyhanni.utils.ItemUtils.getInternalName
import at.hannibal2.skyhanni.utils.ItemUtils.getLore
import at.hannibal2.skyhanni.utils.NumberUtil.roundTo
import at.hannibal2.skyhanni.utils.RegexUtils.matchMatcher
import at.hannibal2.skyhanni.utils.RenderUtils.renderRenderables
import at.hannibal2.skyhanni.utils.StringUtils.removeColor
import at.hannibal2.skyhanni.utils.collection.RenderableCollectionUtils.addString
import at.hannibal2.skyhanni.utils.renderables.Renderable
import at.hannibal2.skyhanni.utils.repopatterns.RepoPattern

@SkyHanniModule
object CruxTalismanDisplay {

    private val config get() = RiftApi.config.cruxTalisman

    /**
     * WRAPPED-REGEX-TEST: "  §23 §2Shy§7: §e88§7/§a100 §7kills"
     * WRAPPED-REGEX-TEST: "  §82 §8Shadow§7: §e42§7/§a50 §7kills"
     * WRAPPED-REGEX-TEST: "  §e- §eVolt§7: §e6§7/§a10 §7kills"
     * WRAPPED-REGEX-TEST: "  4 Puff§7: §a§lMAXED"
     */
    @Suppress("MaxLineLength")
    private val progressPattern by RepoPattern.pattern(
        "rift.everywhere.crux.progress",
        ".*(?<tier>(?:§.)?[IV1-4-]+)\\s+(?<name>(?:§.)?\\w+)§.:\\s*(?<progress>(?:§.)*MAXED|§.\\d+§./§.\\d+).*",
    )

    private const val PARTIAL_NAME = "CRUX_TALISMAN"
    private var display = emptyList<Renderable>()
    private val cruxes = mutableListOf<Crux>()
    private val bonusesLine = mutableListOf<String>()
    private var maxed = false
    private var percentValue = 0.0

    @HandleEvent
    fun onRenderOverlay(event: GuiRenderEvent.GuiOverlayRenderEvent) {
        if (!isEnabled()) return
        config.position.renderRenderables(
            display,
            posLabel = "Crux Talisman Display",
        )
    }

    private fun update() {
        display = drawDisplay()
    }

    private fun drawDisplay() = buildList<Renderable> {
        var showAsMaxed = maxed
        if (!config.compactWhenMaxed && maxed) showAsMaxed = false

        var percent = 0
        if (cruxes.isNotEmpty()) {
            addString("§7Crux Talisman Progress: ${if (showAsMaxed) "§a§lMAXED!" else "§a$percentValue%"}")
            if (!showAsMaxed) {
                for (line in cruxes) {
                    percent += if (config.compactWhenMaxed) {
                        if (!line.maxed) {
                            "(?<progress>\\d+)/\\d+".toRegex().find(line.progress.removeColor())?.groupValues?.get(1)?.toInt() ?: 0
                        } else 100
                    } else {
                        if (line.progress.contains("MAXED"))
                            100
                        else {
                            "(?<progress>\\d+)/\\d+".toRegex().find(line.progress.removeColor())?.groupValues?.get(1)?.toInt() ?: 0
                        }
                    }
                    addString("  ${line.tier} ${line.name}: ${line.progress}")
                }
            }
        }
        val totalPercentage = cruxes.size * 100
        percentValue = ((percent.toDouble() / totalPercentage) * 100).roundTo(1)
        if (bonusesLine.isNotEmpty() && config.showBonuses.get()) {
            addString("§7Bonuses:")
            bonusesLine.forEach { addString("  $it") }
        }
    }

    @HandleEvent
    fun onSecondPassed(event: SecondPassedEvent) {
        if (!isEnabled()) return
        if (!event.repeatSeconds(2)) return
        if (!InventoryUtils.getItemsInOwnInventory().any { it.getInternalName().startsWith(PARTIAL_NAME) }) return

        cruxes.clear()
        bonusesLine.clear()
        maxed = false
        var bonusFound = false
        val inventoryStack = InventoryUtils.getItemsInOwnInventory()
        for (stack in inventoryStack) {
            line@ for (line in stack.getLore()) {
                progressPattern.matchMatcher(line) {
                    val tier = group("tier").replace("-", "0")
                    val name = group("name")
                    val progress = group("progress")
                    val crux = Crux(name, tier, progress, progress.contains("MAXED"))
                    cruxes.add(crux)
                }
                if (line.startsWith("§7Total Bonuses")) {
                    bonusFound = true
                    continue@line
                }
                if (bonusFound) {
                    if (line.isEmpty()) {
                        bonusFound = false
                        continue@line
                    }
                    bonusesLine.add(line)
                }
            }
        }
        maxed = cruxes.all { it.maxed }
        update()
    }

    @HandleEvent
    fun onConfigLoad(event: ConfigLoadEvent) {
        ConditionalUtils.onToggle(config.showBonuses) { update() }
    }

    data class Crux(val name: String, val tier: String, val progress: String, val maxed: Boolean)

    fun isEnabled() = RiftApi.inRift() && config.enabled && !RiftApi.inMirrorVerse
}
