package at.hannibal2.skyhanni.utils import at.hannibal2.skyhanni.utils.compat.formattedTextCompat

import at.hannibal2.skyhanni.SkyHanniMod
import at.hannibal2.skyhanni.api.event.HandleEvent
import at.hannibal2.skyhanni.config.core.config.KeyBind
import at.hannibal2.skyhanni.config.features.chat.ChatPromptUtils
import at.hannibal2.skyhanni.data.ChatManager.deleteChatLine
import at.hannibal2.skyhanni.data.ChatManager.editChatLine
import at.hannibal2.skyhanni.events.MessageSendToServerEvent
import at.hannibal2.skyhanni.events.chat.SkyHanniChatEvent
import at.hannibal2.skyhanni.mixins.hooks.ChatLineData
//#if MC < 1.21
//$$ import at.hannibal2.skyhanni.mixins.transformers.AccessorMixinGuiNewChat
//#endif
import at.hannibal2.skyhanni.skyhannimodule.SkyHanniModule
import at.hannibal2.skyhanni.utils.ChatUtils.CHAT_PREFIX
import at.hannibal2.skyhanni.utils.ChatUtils.DEBUG_PREFIX
import at.hannibal2.skyhanni.utils.ChatUtils.USER_ERROR_PREFIX
import at.hannibal2.skyhanni.utils.ConfigUtils.jumpToEditor
import at.hannibal2.skyhanni.utils.StringUtils.removeColor
import at.hannibal2.skyhanni.utils.StringUtils.stripHypixelMessage
import at.hannibal2.skyhanni.utils.TimeUtils.ticks
import at.hannibal2.skyhanni.utils.chat.TextHelper
import at.hannibal2.skyhanni.utils.chat.TextHelper.asComponent
import at.hannibal2.skyhanni.utils.chat.TextHelper.onClick
import at.hannibal2.skyhanni.utils.chat.TextHelper.prefix
import at.hannibal2.skyhanni.utils.chat.TextHelper.send
import at.hannibal2.skyhanni.utils.compat.MinecraftCompat
import at.hannibal2.skyhanni.utils.compat.addChatMessageToChat
import at.hannibal2.skyhanni.utils.compat.command
import at.hannibal2.skyhanni.utils.compat.hover
import at.hannibal2.skyhanni.utils.compat.url
import net.minecraft.client.MinecraftClient
import net.minecraft.client.gui.hud.ChatHudLine
import net.minecraft.text.Text
import java.util.LinkedList
import java.util.Queue
import kotlin.reflect.KProperty0
import kotlin.time.Duration.Companion.milliseconds
import kotlin.time.times

@SkyHanniModule
object ChatUtils {

    // TODO log based on chat category (error, warning, debug, user error, normal)
    private val log = LorenzLogger("chat/mod_sent")
    var lastButtonClicked = 0L

    private const val DEBUG_PREFIX = "[SkyHanni Debug] §7"
    private const val USER_ERROR_PREFIX = "§c[SkyHanni] "
    private const val CHAT_PREFIX = "[SkyHanni] "

    /**
     * Sends a debug message to the chat and the console.
     * This is only sent if the debug feature is enabled.
     *
     * @param message The message to be sent
     *
     * @see DEBUG_PREFIX
     */
    fun debug(
        message: String,
        replaceSameMessage: Boolean = false,
    ) {
        if (SkyBlockUtils.debug && internalChat(DEBUG_PREFIX + message, replaceSameMessage)) {
            consoleLog("[Debug] $message")
        }
    }

    /**
     * Sends a message to the user that they did something incorrectly.
     * We should tell them what to do instead as well.
     *
     * @param message The message to be sent
     *
     * @see USER_ERROR_PREFIX
     */
    fun userError(
        message: String,
        replaceSameMessage: Boolean = false,
    ) {
        internalChat(USER_ERROR_PREFIX + message, replaceSameMessage)
    }

    /**
     * Sends a message to the user
     * @param message The message to be sent
     * @param prefix Whether to prefix the message with the chat prefix, default true
     * @param prefixColor Color that the prefix should be, default yellow (§e)
     * @param replaceSameMessage Replace the old message with this new message if they are identical
     *
     * @see CHAT_PREFIX
     */
    fun chat(
        message: String,
        prefix: Boolean = true,
        prefixColor: String = "§e",
        replaceSameMessage: Boolean = false,
        onlySendOnce: Boolean = false,
        messageId: Int? = null,
    ) {
        if (prefix) {
            internalChat(prefixColor + CHAT_PREFIX + message, replaceSameMessage, onlySendOnce, messageId = messageId)
        } else {
            internalChat(message, replaceSameMessage, onlySendOnce, messageId = messageId)
        }
    }

    private val messagesThatAreOnlySentOnce = mutableSetOf<String>()

    private fun internalChat(
        message: String,
        replaceSameMessage: Boolean,
        onlySendOnce: Boolean = false,
        messageId: Int? = null,
    ): Boolean {
        val text = message.asComponent()
        if (onlySendOnce && !messagesThatAreOnlySentOnce.add(message)) return false
        return if (replaceSameMessage || messageId != null) {
            text.send(messageId ?: message.getUniqueMessageIdForString())
            chat(text, false)
        } else chat(text)
    }

    fun chat(message: Text, send: Boolean = true): Boolean {
        val formattedMessage = message.formattedTextCompat()
        log.log(formattedMessage)

        if (!MinecraftCompat.localPlayerExists) {
            consoleLog(formattedMessage.removeColor())
            return false
        }

        if (send) addChatMessageToChat(message)
        return true
    }

    /**
     * Sends a message to the user that they can click and run an action
     * @param message The message to be sent
     * @param onClick The runnable to be executed when the message is clicked
     * @param hover The string to be shown when the message is hovered
     * @param expireAt When the click action should expire, default never
     * @param prefix Whether to prefix the message with the chat prefix, default true
     * @param prefixColor Color that the prefix should be, default yellow (§e)
     * @param replaceSameMessage Replace the old message with this new message if they are identical
     *
     * @see CHAT_PREFIX
     */
    fun clickableChat(
        message: String,
        onClick: () -> Unit,
        hover: String = "§eClick here!",
        expireAt: SimpleTimeMark = SimpleTimeMark.farFuture(),
        prefix: Boolean = true,
        prefixColor: String = "§e",
        oneTimeClick: Boolean = false,
        replaceSameMessage: Boolean = false,
    ) {
        val msgPrefix = if (prefix) prefixColor + CHAT_PREFIX else ""

        val rawText = msgPrefix + message
        val text = TextHelper.text(rawText) {
            this.onClick(expireAt, oneTimeClick, onClick)
            this.hover = hover.asComponent()
        }

        if (replaceSameMessage) text.send(rawText.getUniqueMessageIdForString())
        else chat(text)
    }

    /**
     * Sends a message to the user that they can click the message or use the given [keyBind] to run the [code] block.
     *
     * [message] supports the %KEY% placeholder which will be replaced with the effective key string of the [keyBind].
     */
    fun chatConsumerPrompt(
        message: String,
        keyBind: KeyBind,
        consumer: () -> Boolean,
        hover: String = "§eThis Message is a Chat Prompt and can be clicked!",
        prefix: Boolean = true,
        prefixColor: String = "§e",
    ) {
        val msgPrefix = if (prefix) prefixColor + CHAT_PREFIX else ""

        // TODO isnt the permanent click action essentially a small memory leak that bunches up over time?
        val rawText = msgPrefix + message.replace("%KEY%", keyBind.getEffectiveKeyString())
        val text = TextHelper.text(rawText) {
            this.onClick(SimpleTimeMark.now().plus(keyBind.getEffectiveExpirationDuration()), true, consumer)
            this.hover = hover.asComponent()
        }
        ChatPromptUtils.setActivePrompt(keyBind, consumer)
        chat(text)
    }

    fun chatPrompt(
        message: String,
        keyBind: KeyBind,
        code: () -> Unit,
        hover: String = "§eThis Message is a Chat Prompt and can be clicked!",
        prefix: Boolean = true,
        prefixColor: String = "§e",
    ) {
        chatConsumerPrompt(
            message, keyBind,
            consumer = {
                code.invoke()
                true
            },
            hover, prefix, prefixColor,
        )
    }

    /**
     * Sends the message in chat.
     * Show the lines when on hover.
     * Offer option to click on the chat message to copy the lines to clipboard.
     * Useful for quick debug infos
     */
    fun clickToClipboard(message: String, lines: List<String>) {
        val text = lines.joinToString("\n") { "§7$it" }
        clickableChat(
            "$message §7(hover for info)",
            hover = "$text\n \n§eClick to copy to clipboard!",
            onClick = {
                ClipboardUtils.copyToClipboard(text.removeColor())
            },
        )
    }

    private val uniqueMessageIdStorage = mutableMapOf<String, Int>()
    private fun String.getUniqueMessageIdForString() = uniqueMessageIdStorage.getOrPut(this) {
        getUniqueMessageId()
    }

    private var lastUniqueMessageId = 123242

    fun getUniqueMessageId() = lastUniqueMessageId++

    /**
     * Sends a message to the user that they can click and run a command
     * @param message The message to be sent
     * @param hover The message to be shown when the message is hovered
     * @param command The command to be executed when the message is clicked
     * @param prefix Whether to prefix the message with the chat prefix, default true
     * @param prefixColor Color that the prefix should be, default yellow (§e)
     *
     * @see CHAT_PREFIX
     */
    fun hoverableChat(
        message: String,
        hover: List<String>,
        command: String? = null,
        prefix: Boolean = true,
        prefixColor: String = "§e",
    ) {
        val msgPrefix = if (prefix) prefixColor + CHAT_PREFIX else ""

        chat(
            TextHelper.text(msgPrefix + message) {
                this.hover = TextHelper.multiline(hover)
                if (command != null) {
                    this.command = command
                }
            },
        )
    }

    /**
     * Sends a message to the user that they can click and run a command
     * @param message The message to be sent
     * @param url The url to be opened
     * @param autoOpen Automatically opens the url as well as sending the clickable link message
     * @param hover The message to be shown when the message is hovered
     * @param prefix Whether to prefix the message with the chat prefix, default true
     * @param prefixColor Color that the prefix should be, default yellow (§e)
     *
     * @see CHAT_PREFIX
     */
    fun clickableLinkChat(
        message: String,
        url: String,
        hover: String = "§eOpen $url",
        autoOpen: Boolean = false,
        prefix: Boolean = true,
        prefixColor: String = "§e",
        replaceSameMessage: Boolean = false,
    ) {
        val msgPrefix = if (prefix) prefixColor + CHAT_PREFIX else ""
        val text = TextHelper.text(msgPrefix + message) {
            this.url = url
            this.hover = "$prefixColor$hover".asComponent()
        }

        if (replaceSameMessage) text.send(message.getUniqueMessageIdForString())
        else chat(text)

        if (autoOpen) OSUtils.openBrowser(url)
    }

    /**
     * Sends a message to the user that combines many message components e.g. clickable, hoverable and regular text
     * @param components The list of components to be joined together to form the final message
     * @param prefix Whether to prefix the message with the chat prefix, default true
     * @param prefixColor Color that the prefix should be, default yellow (§e)
     *
     * @see CHAT_PREFIX
     */
    fun multiComponentMessage(
        components: List<Text>,
        prefix: Boolean = true,
        prefixColor: String = "§e",
    ) {
        val msgPrefix = if (prefix) prefixColor + CHAT_PREFIX else ""
        chat(TextHelper.join(components).prefix(msgPrefix))
    }

    /**
     * This does the same as if you entered the given string in the chat gui and pressed enter with the only differnce of no history.
     */
    fun executeAsChatInput(message: String) {
//         //#if MC < 1.21
//         ClientCommandHandler.instance.executeCommand(MinecraftCompat.localPlayer, message)
//         //#else
//         //$$ MinecraftClient.getInstance().networkHandler.sendChatMessage(message)
//         //#endif
    }

    private val chatGui get() = MinecraftClient.getInstance().inGameHud.chatHud

    //#if MC < 1.21
    //$$ var chatLines: MutableList<ChatHudLine>
    //$$     get() = (chatGui as AccessorMixinGuiNewChat).chatLines_skyhanni
    //$$     set(value) {
    //$$         (chatGui as AccessorMixinGuiNewChat).chatLines_skyhanni = value
    //$$     }
    //$$
    //$$ var drawnChatLines: MutableList<ChatHudLine>
    //$$     get() = (chatGui as AccessorMixinGuiNewChat).drawnChatLines_skyhanni
    //$$     set(value) {
    //$$         (chatGui as AccessorMixinGuiNewChat).drawnChatLines_skyhanni = value
    //$$     }
    //#else
    var chatLines: MutableList<ChatHudLine>
        get() = chatGui.messages
        set(value) {
            chatGui.messages = value
        }

    var drawnChatLines: MutableList<ChatHudLine.Visible>
        get() = chatGui.visibleMessages
        set(value) {
            chatGui.visibleMessages = value
        }
    //#endif

    /** Edits the first message in chat that matches the given [predicate] to the new [component]. */
    fun editFirstMessage(
        component: (Text) -> Text,
        reason: String,
        predicate: (ChatHudLine) -> Boolean,
    ) {
        chatLines.editChatLine(component, predicate, reason)
        refreshChat()
    }

    /**
     * Deletes a maximum of [amount] messages in chat that match the given [predicate].
     */
    fun deleteMessage(
        reason: String,
        amount: Int = 1,
        predicate: (ChatHudLine) -> Boolean,
    ) {
        chatLines.deleteChatLine(amount, reason, predicate)
        refreshChat()
    }

    private fun refreshChat() {
        DelayedRun.onThread.execute {
            chatGui.reset()
        }
    }

    private var deleteNext: Pair<String, (String) -> Boolean>? = null

    @HandleEvent(priority = HandleEvent.HIGH)
    fun onChat(event: SkyHanniChatEvent) {
        val (reason, predicate) = deleteNext ?: return
        this.deleteNext = null

        if (predicate(event.message)) {
            event.blockedReason = reason
        }
    }

    @HandleEvent
    fun onSendMessage(event: MessageSendToServerEvent) {
        if (event.senderIsSkyhanni()) return
        lastMessageSent = SimpleTimeMark.now()
    }

    fun deleteNextMessage(
        reason: String,
        predicate: (String) -> Boolean,
    ) {
        deleteNext = reason to predicate
    }

    private var lastMessageSent = SimpleTimeMark.farPast()
    private val sendQueue: Queue<String> = LinkedList()
    private val messageDelay = 300.milliseconds

    fun getTimeWhenNewlyQueuedMessageGetsExecuted() =
        (lastMessageSent + sendQueue.size * messageDelay).takeIf { !it.isInPast() } ?: SimpleTimeMark.now()

    @HandleEvent
    fun onTick() {
        if (lastMessageSent.passedSince() > messageDelay) {
            MinecraftCompat.localPlayer.networkHandler.sendChatMessage(sendQueue.poll() ?: return)
            lastMessageSent = SimpleTimeMark.now()
        }
    }

    fun sendMessageToServer(message: String) {
        if (canSendInstantly()) {
            MinecraftCompat.localPlayerOrNull?.let {
                it.networkHandler.sendChatMessage(message)
                lastMessageSent = SimpleTimeMark.now()
                return
            }
        }
        sendQueue.add(message)
    }

    private fun canSendInstantly() = sendQueue.isEmpty() && lastMessageSent.passedSince() > messageDelay

    fun MessageSendToServerEvent.isCommand(commandWithSlash: String) = splitMessage.takeIf {
        it.isNotEmpty()
    }?.get(0) == commandWithSlash

    fun MessageSendToServerEvent.isCommand(commandsWithSlash: Collection<String>) =
        splitMessage.takeIf { it.isNotEmpty() }?.get(0) in commandsWithSlash

    fun MessageSendToServerEvent.senderIsSkyhanni() = originatingModContainer?.id == "skyhanni"

    fun MessageSendToServerEvent.eventWithNewMessage(message: String) =
        MessageSendToServerEvent(message, message.split(" "), this.originatingModContainer)

    fun chatAndOpenConfig(message: String, property: KProperty0<*>) {
        clickableChat(
            message,
            onClick = { property.jumpToEditor() },
            "§eClick to find setting in the config!",
        )
    }

    fun clickToActionOrDisable(
        message: String,
        option: KProperty0<*>,
        actionName: String,
        action: () -> Unit,
        oneTimeClick: Boolean = false,
    ) {
        val hint = if (SkyHanniMod.feature.chat.hideClickableHint) "" else
            "\n§e[CLICK to $actionName or disable this feature]"
        clickableChat(
            "$message$hint",
            onClick = {
                if (KeyboardManager.isShiftKeyDown() || KeyboardManager.isModifierKeyDown()) {
                    option.jumpToEditor()
                } else {
                    action()
                }
            },
            hover = "§eClick to $actionName!\n§eShift-Click or Control-Click to disable this feature!",
            oneTimeClick = oneTimeClick,
            replaceSameMessage = true,
        )
    }

    fun clickToActionOrEnableAuto(
        message: String,
        option: KProperty0<*>,
        actionName: String,
        action: () -> Unit,
        oneTimeClick: Boolean = false,
    ) {
        val hint = if (SkyHanniMod.feature.chat.hideClickableHint) "" else
            "\n§e[CLICK to $actionName one Time or Enable automatic]"
        clickableChat(
            "$message$hint",
            onClick = {
                if (KeyboardManager.isShiftKeyDown() || KeyboardManager.isModifierKeyDown()) {
                    option.jumpToEditor()
                } else {
                    action()
                }
            },
            hover = "§eClick to $actionName!\n§eShift-Click or Control-Click to do it automatically in the Future!",
            oneTimeClick = oneTimeClick,
            replaceSameMessage = true,
        )
    }

    var ChatHudLine.fullComponent: Text
        get() = (this as ChatLineData).skyHanni_fullComponent
        set(value) {
            (this as ChatLineData).skyHanni_fullComponent = value
        }

    //#if MC < 1.16
    //$$ val ChatLine.chatMessage get() = chatComponent.formattedText.stripHypixelMessage()
    //$$ fun ChatLine.passedSinceSent() = (Minecraft.getMinecraft().ingameGUI.updateCounter - updatedCounter).ticks
    //#elseif MC < 1.21
    //$$ val ChatHudLine<Text>.chatMessage get() = text.formattedTextCompat().stripHypixelMessage()
    //$$ fun ChatHudLine<Text>.passedSinceSent() = (MinecraftClient.getInstance().inGameHud.ticks - creationTick).ticks
    //#else
    val ChatHudLine.chatMessage get() = content.formattedTextCompat().stripHypixelMessage()
    fun ChatHudLine.passedSinceSent() = (MinecraftClient.getInstance().inGameHud.ticks - creationTick).ticks
    //#endif

    fun consoleLog(text: String) {
        SkyHanniMod.consoleLog(text)
    }

    @Suppress("UnusedParameter")
    fun suggestInChat(message: String) {
        // TODO
        chat("Chat Suggestion is not implemented yet!")
    }

}
